/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wave.c
 * WAVE sound ( writer )
 *
 * TODO:
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpsound.h>
#include <libxmm/lpcodeca.h>
#include <libxmm/lpfiltera.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include "wave.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		8192
#define	SND_FORMAT_TAG		0x55

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_PluginCodecAudio	*pCodec;

    int				data_size, data_start;
    char			*rbuffer;

    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Global data
 */

extern XMM_PluginSound	plugin_info;

/*
 * Prototypes
 */

static int WRITE32( XMM_PluginIO *pIO, uint32_t val );
static int WRITE16( XMM_PluginIO *pIO, uint16_t val );
static int WRITE8( XMM_PluginIO *pIO, unsigned char val );
static int WRITEn( XMM_PluginIO *pIO, void *data, uint32_t size );

/*
 * Initialize
 */
static XMM_PluginSound *wave_Init( void *xmm )
{
  XMM_PluginSound *sound;
  struct priv_t *priv;

  if(( sound = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginSound ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_ERR_ALLOC, "(WAVE) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &sound[1];
  sound->sys.priv = (void *) priv;
  sound->sys.xmm = xmm;

  return sound;
}

/*
 * Close
 */
static void wave_Close( XMM_PluginSound *sound )
{
  struct priv_t *priv = sound->sys.priv;
  int		pos;

  /* Stop audio chain */
  xmm_FilterAChainStop( sound->sys.xmm );

  /* Close Codec */
  if( priv->pCodec )	priv->pCodec->Close( priv->pCodec );

  /* Fix / Rewrite Header */
  pos = priv->pIO->Tell( priv->pIO );
  priv->pIO->Seek( priv->pIO, 4, XMM_SEEK_SET );
  WRITE32( priv->pIO, pos - 4 );

  priv->pIO->Seek( priv->pIO, priv->data_start - 4, XMM_SEEK_SET );
  WRITE32( priv->pIO, priv->data_size );

  priv->pIO->Close( priv->pIO );

  free( sound );
}

/*
 * Control function
 */
static int wave_Control( XMM_PluginSound *sound, uint32_t cmd, uint32_t arg, void *data )
{
  struct priv_t *priv = sound->sys.priv;

  switch( cmd )
  {
    case XMM_CTLQUERY_SFORMAT:
		return XMM_CTLRET_TRUE;

    case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

    case XMM_CTLGET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;
  
    case XMM_CTLSET_VOLUME:
		return XMM_CTLRET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_SOUND )	return XMM_CTLRET_UNKNOWN;
  return XMM_CTLRET_INVALID;	/* No SOUND command */
}

/*
 * Start output
 */
static int wave_Start( XMM_PluginSound *sound, XMM_SoundFormat *sformat, int fsize, int fcount )
{
  struct priv_t 	*priv = sound->sys.priv;
  const char		*ofn;
  int			dest_af_size, wave_fmt_head_size;
  WAVEFORMATEX		wave_fmt_head;
  XMM_AudioFormat	src_af, *dest_af;

  if(( ofn = xmm_AcOutputFilename( sound->sys.xmm, NULL, ".wav" )) == NULL )
	return XMM_RET_ERROR;

  if(( priv->pIO = xmm_IOOpen( sound->sys.xmm, ofn, XMM_IO_WRITE )) == NULL )
	return XMM_RET_ERROR;

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  /* Allocated write buffer */
  if(( priv->rbuffer = malloc( fsize )) == NULL )
  {
	xmm_SetError( sound->sys.xmm, XMM_ERR_ALLOC, "(WAVE) read buffer." );
	return XMM_RET_ERROR;
  }

  /* Set input data format */
  src_af.formatID = SND_FORMAT_TAG;
  src_af.channels = sformat->channels;
  src_af.samprate = sformat->samprate;
  src_af.bitrate = sformat->samprate * sformat->channels * ( sformat->format & XMM_SOUND_MASK_SIZE );
  src_af.format = sformat->format;
  src_af.blockSize = sformat->channels * ( sformat->format & XMM_SOUND_MASK_SIZE );
  src_af.extraSize = 0;
  src_af.desc[0] = '\0';

  /* Open codec */
  if(( priv->pCodec = xmm_CodecAudioOpen( sound->sys.xmm,
	XMM_CODEC_MODE_ENCODE, &src_af )) == NULL )	return XMM_RET_ERROR;

  /* Get output data format */
  priv->pCodec->Control( priv->pCodec, XMM_CTLGET_FORMAT_SIZE, &dest_af_size );
  priv->pCodec->Control( priv->pCodec, XMM_CTLGET_FORMAT_DATA, &dest_af );

  /* Fill wave format with XMMP audio format info */
  wave_fmt_head.wFormatTag	= dest_af->formatID;
  wave_fmt_head.nChannels	= dest_af->channels;
  wave_fmt_head.nSamplesPerSec	= dest_af->samprate;
  wave_fmt_head.nAvgBytesPerSec	= dest_af->bitrate / 8;
  wave_fmt_head.nBlockAlign	= dest_af->blockSize;
  wave_fmt_head.wBitsPerSample	= ( dest_af->format & XMM_SOUND_MASK_SIZE );
  if( dest_af->formatID != 0x001 )	wave_fmt_head.wBitsPerSample = 0;
  wave_fmt_head.cbSize		= dest_af->extraSize;

  /* No cbSize member for PCM ( ID = 0x0001 ) */
  if( dest_af->formatID == 0x0001 )	wave_fmt_head_size = 16;
  else	wave_fmt_head_size = 18;

  /* Write header */
  WRITE32( priv->pIO, RIFF_ID_RIFF );
  WRITE32( priv->pIO, 0 );
  WRITE32( priv->pIO, RIFF_ID_WAVE );
  WRITE32( priv->pIO, RIFF_ID_fmt );
  WRITE32( priv->pIO, wave_fmt_head_size + dest_af->extraSize );
  WRITEn( priv->pIO, &wave_fmt_head, wave_fmt_head_size );
  if( dest_af->extraSize )	WRITEn( priv->pIO, &dest_af[1], dest_af->extraSize );
  WRITE32( priv->pIO, RIFF_ID_data );
  WRITE32( priv->pIO, 0 );

  priv->data_start = priv->pIO->Tell( priv->pIO );
  priv->data_size = 0;

  xmm_logging( 2, "WAVE! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
				sformat->samprate, sformat->channels, sformat->format, fsize );

  /* Set source format */
  xmm_FilterAChainInput( sound->sys.xmm, sformat->samprate, sformat->channels, sformat->format );

  /* Set dest format */
  priv->bufferSize = xmm_FilterAChainOutput( sound->sys.xmm, dest_af->samprate, dest_af->channels, dest_af->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)src_af.bitrate;
  priv->bufferSSize = ( sformat->channels * ( sformat->format & XMM_SOUND_MASK_SIZE ) / 8 );

  /* return buffer size ( = number of samples ) */
  return priv->bufferSize;
}

/*
 * Write data
 */
static int wave_Write( XMM_PluginSound *sound, void *data, uint32_t samples )
{
  struct priv_t *priv = sound->sys.priv;
  int		proc, written = 0, conv, encoded, done, ret;
  uint8_t	*ptr;

  /*
   * samples > bufferSize: We have to split data for xmm_FilterAChain()
   */

  while( samples > 0 )
  {
	proc = samples;
	if( proc > priv->bufferSize )	proc = priv->bufferSize;

	conv = proc;
	ptr = data;

	if( xmm_FilterAChainActive( sound->sys.xmm ))
	{
	    if(( ret = xmm_FilterAChain( sound->sys.xmm, data, proc, &ptr, &conv )) != XMM_RET_OK )
		return ret;
	}

	encoded = priv->bufferSize * priv->bufferSSize;
	priv->pCodec->Encode( priv->pCodec, ptr, conv, priv->rbuffer, &encoded, 0 );

	done = priv->pIO->Write( priv->pIO, priv->rbuffer, 1, encoded );
	priv->data_size += done;

	written += done;
	data += ( proc * priv->bufferSSize );
	samples -= proc;
  }

  if( written == 0 )	return XMM_RET_ERROR;

  return written;
}

/*
 * Flush data
 */
static void wave_Flush( XMM_PluginSound *sound )
{
}

/*
 * Plugin info
 */

XMM_PluginSound	plugin_info = {	{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_SOUND,
				0,
				XMM_VERSION_NUM,
				"",
				"WAVE",
				"Sound: WAVE",
				"Copyright (c) 2000, 2001 by Arthur Kleer",
				NULL, NULL },
				wave_Init, wave_Close, wave_Control,
				wave_Start, wave_Write, wave_Flush };

/*
 * Data output ( internal )
 */

static int WRITEn( XMM_PluginIO *pIO, void *data, uint32_t size )
{
  return pIO->Write( pIO, data, size, 1 );
}

static int WRITE32( XMM_PluginIO *pIO, uint32_t val )
{
  return pIO->Write( pIO, &val, 4, 1 );
}

static int WRITE16( XMM_PluginIO *pIO, uint16_t val )
{
  return pIO->Write( pIO, &val, 2, 1 );
}

static int WRITE8( XMM_PluginIO *pIO, unsigned char val )
{
  return pIO->Write( pIO, &val, 1, 1 );
}
