/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2001 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * xmmplay.c
 * Very simple example of how to use the LinuX MultiMedia Library
 * However, it is complex enough for a command line player.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/system.h>
#include <libxmmplay/xmmplay.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/*
 * Global data
 */

/*
 * Prototypes
 */

static int EventCB( void *priv, XMM_Event *event );

/*
 * Main()
 */
int main( int argc, char *argv[] )
{
  char			*file = argv[1];
  double		ctime, seekval, delay;
  XMM_Play		*xmmplay;
  XMM_FileInfo		info;
  XMM_ClipInfo		ci;
  XMM_ControlScale	scale;
  XMM_Event		event;
  int			i, factor = XMM_GRAPH_RESIZE_ORIG, loop = 1, ret;
  char			*graph_plugin = NULL, *sound_plugin = NULL;
  char			*output_plugin = NULL, *gconv_plugin = NULL;
  double		seek_value = 0.0;

  printf( "\n * xmmplay %s\n", XMM_VERSION_STRING );
  printf( " * LinuX MultiMedia Project - command line player\n");
  printf( " * www.frozenproductions.com\n\n");

  if( argc < 1 )
  {
	printf( "\nUsage: xmmplay [options] file\n\n" );
	printf( "Options:\n" );
	printf( "  -g plugin\t\tuse plugin as Graph plugin\n" );
	printf( "  -s plugin\t\tuse plugin as Sound plugin\n" );
	printf( "  -o plugin\t\tuse plugin as Output plugin\n" );
	printf( "  -2       \t\tdouble size mode\n" );
	printf( "  -3       \t\tfullscreen mode\n\n" );
	exit(1);
  }

  for( i = 1; i < argc; i++ )
  {
	if( !strcmp( argv[i], "-g" ))	graph_plugin = argv[++i];
	else if( !strcmp( argv[i], "-s" ))	sound_plugin = argv[++i];
	else if( !strcmp( argv[i], "-o" ))	output_plugin = argv[++i];
	else if( !strcmp( argv[i], "-gc" ))	gconv_plugin = argv[++i];
	else if( !strcmp( argv[i], "-seek" ))	seek_value = strtod( argv[++i], NULL );
	else if( !strcmp( argv[i], "-2" ))	factor = XMM_GRAPH_RESIZE_DOUBLE;
	else if( !strcmp( argv[i], "-3" ))	factor = XMM_GRAPH_RESIZE_FULLSCREEN;
	else file = argv[i];
  }

  printf( "Playing '%s'...\n", file );
  if( graph_plugin )	printf( "Using graph plugin: %s\n", graph_plugin );
  if( sound_plugin )	printf( "Using sound plugin: %s\n", sound_plugin );
  if( output_plugin )	printf( "Using output plugin: %s\n\n", output_plugin );
  if( gconv_plugin )	printf( "Using gconv plugin: %s\n", gconv_plugin );
  if( factor != XMM_GRAPH_RESIZE_ORIG )	printf( "Scaling value: %i\n", -factor );

  if(( xmmplay = xmmplay_Init()) == NULL )
  {
	printf( "\nError initializing xmmplay library: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
  }

  if( output_plugin )
  {
    if( xmm_AcOutputPluginName( XMMPLAY_2XMM(xmmplay), output_plugin ) == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
  }

  if( graph_plugin )
  {
    if( xmm_AcGraphPluginName( XMMPLAY_2XMM(xmmplay), graph_plugin ) == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
  }

  if( sound_plugin )
  {
    if( xmm_AcSoundPluginName( XMMPLAY_2XMM(xmmplay), sound_plugin ) == NULL )
    {
	printf( "\nError setting output plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
  }

  if( gconv_plugin )
  {
    if( xmm_AcGConvPluginName( XMMPLAY_2XMM(xmmplay), gconv_plugin ) == NULL )
    {
	printf( "\nError setting converter plugins: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
    }
  }

  if( xmm_SetEventCB( XMMPLAY_2XMM(xmmplay), EventCB, (void *)xmmplay, 1 ) < XMM_RET_OK )
  {
	printf( "\nError setting event callback: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	exit(1);
  }

  if( xmmplay_Open( xmmplay, file, XMM_INPUT_CF_MODE_DOUT | XMM_INPUT_CF_MODE_READ ) < XMM_RET_OK )
  {
	printf( "\nError opening file %s: %s\n", argv[1], xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	xmmplay_Exit( xmmplay );
	exit(1);
  }

  sleep(1);

  xmmplay_Scale( xmmplay, 0, 0, factor );

  if( xmmplay_Play( xmmplay ) < XMM_RET_OK )
  {
	printf( "\nError starting playback: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
	xmmplay_Close( xmmplay );
	xmmplay_Exit( xmmplay );
	exit(1);
  }

  xmmplay_Info( xmmplay, &info, &ci, &delay, NULL );
  printf( "\nName = %s, A/V Diff = %.2f Playtime = %.2i:%.2i:%.2i\n",
	ci.name, delay, 
	XMMPLAY_TIME_H( ci.playtime ),
	XMMPLAY_TIME_M( ci.playtime ),
	XMMPLAY_TIME_S( ci.playtime ));

  if( info.vstreams )
  {
	printf( "Video: %ix%i, %.2f FPS, %i kbps, Aspect %.2f:%.2f ( %i Frames, %i bytes )\n",
				info.vi[0].width, info.vi[0].height,
				info.vi[0].framerate, info.vi[0].bitrate,
				info.vi[0].aspect_val, info.vi[0].aspect_div,
				info.vi[0].tFrames, info.vi[0].tSize );
  }

  if( info.astreams )
  {
	printf( "Audio: %i Hz, %i kbps%s, %i Channels\n",
			info.ai[0].format.samprate,
			abs( info.ai[0].bitrate ) / 1000,
			info.ai[0].bitrate < 0 ? " (variable)" : "",
			info.ai[0].format.channels );
  }

  printf( "\n" );

  xmmplay_Seek( xmmplay, seek_value );

  /* Status can never be XMM_PAUSED, as we do not pause the input */
  while(( xmmplay_Status( xmmplay ) & XMM_PBS_PLAYING ) && ( loop == 1 ))
  {
	if( seek_value != -1 )
	{
		xmmplay_Seek( xmmplay, seek_value );
		seek_value = -1;
	}

	ctime = xmmplay_Info( xmmplay, NULL, NULL, NULL, &seekval );
	printf( "%.2i:%.2i:%.2i ( %3.2f%% )  \r", XMMPLAY_TIME_H( ctime ), XMMPLAY_TIME_M( ctime ), XMMPLAY_TIME_S( ctime ), seekval * 100 );
	fflush( stdout );


	/* Sleep some time */
	xmmSYS_usleep( 500000 );

	/* Check for event */
	if( xmm_PollEvent( XMMPLAY_2XMM(xmmplay), 0, &event ) <= 0 )	continue;

	/* We have an event */
	switch( event.type )
	{
	    case XMM_EVENT_KEY:
		{
		    XMM_Event_Key *key = (XMM_Event_Key *) &event;

#ifdef DEBUG
		    printf( "Key event: scancode %i ASCII %i ( '%c' ) state %i mode = %x\n", key->scode, key->sym, key->sym, key->state, key->mod );
#endif

		    if(( key->state == XMM_KEY_RELEASED ) &&
			    (( key->sym >= '1' ) && ( key->sym <= '3' )))
		    {
			xmm_logging( 1, "xmmplay: Scaling...\n" );

			/* fill XMM_ControlScale structure */
#if 0
			scale.width = width;
			scale.height = height;
#else
			scale.width = 0;
			scale.height = 0;
#endif
			scale.flags = 0;

			if( key->sym == '1' )	scale.flags = XMM_GRAPH_RESIZE_ORIG;
			if( key->sym == '2' )	scale.flags = XMM_GRAPH_RESIZE_DOUBLE;
			if( key->sym == '3' )	scale.flags = XMM_GRAPH_RESIZE_FULLSCREEN;

			ret = xmmplay_Scale( xmmplay, scale.width, scale.height, scale.flags );
			if( ret < 0 )
			{
			    fprintf( stderr, "Error resizing output: %s\n", xmm_ErrorString( XMMPLAY_2XMM( xmmplay )));
			}
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 'q' ))
		    {
			xmm_logging( 1, "xmmplay: Quitting...\n" );
			loop = 0;
		    }

		    if(( key->state == XMM_KEY_RELEASED ) && ( key->sym == 's' ))
		    {
			xmm_logging( 1, "xmmplay: Seeking...\n" );
			seek_value = 0.55;
		    }

		    break;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event.type );
		break;  
    }
  }

  xmmplay_Stop( xmmplay );
  xmmplay_Close( xmmplay );
  xmmplay_Exit( xmmplay );

  return 0;
}

/*
 * Event handler ( for not queueable events )
 */

static int EventCB( void *priv, XMM_Event *event )
{

  switch( event->type )
  {
	/* Select from Table of Contents */
	case XMM_EVENT_TOC:
		{
		    int i;
		    XMM_Event_TOC *etoc = (XMM_Event_TOC *) event;

		    fprintf( stdout, "\nTable of Contents:\n" );
		    for( i = 0; i < etoc->entries; i++ )	fprintf( stdout, "\t%2i: %s\n", i, etoc->entry[i] );

		    fprintf( stdout, "Please make your selection: " );
		    scanf( "%i", &etoc->selected );
		    return 0;
		}

	/* Authorization requiered */
	case XMM_EVENT_AUTH:
		{
		    char *passwd;
		    XMM_Event_Auth *eauth = (XMM_Event_Auth *) event;

		    fprintf( stdout, "\nAuthorization required:\n" );
		    fprintf( stdout, "User: " );
		    scanf( "%s", eauth->user );
		    passwd = getpass( "Password:" );
		    if( passwd == NULL )	break;
		    strcpy( eauth->passwd, passwd );
		    return 0;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event->type );
		break;  
  }

  return XMM_RET_ERROR;
}
