/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * libmpg123.c
 * Interface for libmpg123, by Arthur Kleer 01.10.2001
 * Based on interface.c ( mpglib )
 */

#include "libmpg123.h"

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#define	DEBUG
#endif

#ifdef VERBOSE
#include "mpg123head.h"
#endif

/*
 * Global data
 */

/* Some functions need mpstr object, but don't get it as argument */
struct mpstr	*gmp = NULL;

/*
 * Prototypes
 */

static unsigned long read_head(struct mpstr *mp, mpg123_reader_t *reader, void *priv );

#ifdef USE_MMX
int _synth_1to1_MMX(real *bandPtr,int channel,unsigned char *out,int *pnt);
void make_decode_tables_MMX(long scale);
#endif

#ifdef PENTIUM_OPT
int _synth_1to1_pent(real *bandPtr,int channel,unsigned char *out,int *pnt);
#endif

/*
 * Initialize
 */
int MPG123_Init( mpg123_t *mpg123, uint32_t flags )
{
  static int	init = 1;
  struct mpstr	*mp = (struct mpstr *) mpg123;

  memset( mp, 0, sizeof( struct mpstr ));
  mp->framesize = 0;
  mp->fsizeold = -1;
  mp->bsize = 0;
  mp->head = mp->tail = NULL;
  mp->fr.single = -1;


#ifdef USE_MMX
  if( flags & MMACCEL_X86_MMX )
  {
	mp->fr.synth = _synth_1to1_MMX;
#ifdef DEBUG
	fprintf( stdout, "libmpg123! Using MMX.\n" );
#endif
  }
  else
#endif
#ifdef PENTIUM_OPT
  if(( flags & MMACCEL_CPUMASK ) >= MMACCEL_CPU_586 )
  {
	mp->fr.synth = _synth_1to1_pent;
#ifdef DEBUG
	fprintf( stdout, "libmpg123! Using 586.\n" );
#endif
  }
  else
#endif
  mp->fr.synth = synth_1to1;
  mp->bsnum = 0;
  mp->synth_bo = 1;

  if( init )
  {
	init = 0;
#ifdef USE_MMX
	if( flags & MMACCEL_X86_MMX )
	{
#ifdef DEBUG
	    fprintf( stdout, "libmpg123! Creating tables using MMX.\n" );
#endif
	    make_decode_tables_MMX( 32767 );
	}
	else
#endif
	make_decode_tables( 32767 );
	init_layer2();
	init_layer3( SBLIMIT );
  }

  return MPG123_OK;
}

/*
 * Exit
 */
void MPG123_Exit( mpg123_t *mpg123 )
{
}

/*
 * Decode one frame
 */
int MPG123_DecodeFrame( mpg123_t *mpg123, unsigned char *out, int *done, mpg123_reader_t *reader, void *priv )
{
  struct mpstr	*mp = (struct mpstr *) mpg123;
  int		ret;
#ifdef VERBOSE
  MPG123_Info	mpg123_info;
  char		*tab_mpeg[] = { "1", "2", "?", "2.5" };
  char		*tab_layer[] = { "?", "I", "II", "III" };
  char		*tab_flag[] = { "No", "Yes" };
  char		*tab_mode[] = { "Stereo", "Joint stereo", "Dual channel", "Mono" };
#endif

  /* Check if decoding already in process */
  if( gmp != NULL )	return	MPG123_ERROR;
  gmp = mp;

  /* Sync stream / Decode header */
  if( mp->framesize == 0 )
  {
	mp->header = read_head( mp, reader, priv );
	if( mp->header == 0 )
	{
#ifdef DEBUG
	    fprintf( stderr, "libmpg123! ERROR: invalid header ( bq size = %i )\n", reader->size( priv ));
#endif
	    gmp = NULL;
	    return MPG123_MOREDATA;
	}
	decode_header( &mp->fr, mp->header );

#ifdef VERBOSE
	MPG123_DecodeInfo( mp->header, &mpg123_info );

	fprintf( stderr, "libmpg123: INFO: MPEG %s (audio) Layer %s, %i kbps, %i Hz, %s\n",
					tab_mpeg[mpg123_info.mpeg],
					tab_layer[mpg123_info.layer],
					mpg123_info.bitrate,
					mpg123_info.samprate,
					tab_mode[mpg123_info.mode] );

	fprintf( stderr, "libmpg123: INFO: bytes / frame = %.2f (%i) time / frame = %.4fs\n",
					mpg123_info.bpf,
					mpg123_info.framesize,
					mpg123_info.tpf );

	fprintf( stderr, "libmpg123: INFO: Emphasis = %i, CRC: %s, Copyright: %s, Original: %s\n",
					mpg123_info.emphasis,
					tab_flag[ mpg123_info.crc ],
					tab_flag[ mpg123_info.copyright ],
					tab_flag[ mpg123_info.original ] );
#endif
	mp->framesize = mp->fr.framesize;
  }

  wordpointer = mp->bsspace[mp->bsnum] + 512;
  mp->bsnum = (mp->bsnum + 1) & 0x1;
  bitindex = 0;

#ifdef DEBUG
  fprintf( stderr, "libmpg123! bq size %i ( needed %i )\n", reader->size( priv ), mp->framesize );
#endif

  /* Read frame data */
  ret = reader->read( priv, wordpointer, mp->framesize );
  if( ret < 0 )
  {
	gmp = NULL;
	return MPG123_ERROR;
  } 
  else if( ret < mp->framesize )
  {
#ifdef DEBUG
	fprintf( stderr, "libmpg123! ERROR: not enough data. Only %i read, framesize = %i\n", ret, mp->framesize );
#endif
	gmp = NULL;
	return MPG123_MOREDATA;
  }

  /* Decode frame */
  *done = 0;
  if( mp->fr.error_protection )	getbits(16);

  switch( mp->fr.lay )
  {
	case 1:
		do_layer1( &mp->fr,(unsigned char *) out, done );
        	break;
	case 2:
		do_layer2( &mp->fr,(unsigned char *) out, done );
        	break;
	case 3:
		do_layer3( &mp->fr,(unsigned char *) out, done );
        	break;
  }

  mp->fsizeold = mp->framesize;
  mp->framesize = 0;

  gmp = NULL;
  return MPG123_OK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * *
 * Needed by mpglib ( from mpglib [interface.c ] ) *
 * * * * * * * * * * * * * * * * * * * * * * * * * */

int set_pointer(long backstep)
{
  unsigned char *bsbufold;
  if(gmp->fsizeold < 0 && backstep > 0) {
    fprintf(stderr,"Can't step back %ld!\n",backstep);
    return MP3_ERR;
  }
  bsbufold = gmp->bsspace[gmp->bsnum] + 512;
  wordpointer -= backstep;
  if (backstep)
    memcpy(wordpointer,bsbufold+gmp->fsizeold-backstep,backstep);
  bitindex = 0;
  return MP3_OK;
}

/* * * * * * * * *
 * Internal code *
 * * * * * * * * */

static int head_check( unsigned long head )
{
  if(( head & 0xffe00000 ) != 0xffe00000 )	return 0;	// minimum

  if( !(( head >> 17 ) & 3 ))		return 0;	// layer = 1,2,3

  if((( head >> 12 ) & 0xf ) == 0xf )	return 0;	// bitrate_index != 0xF

  if( !(( head >> 12 ) & 0xf ))		return 0;	// bitrate_index != 0x0

  if((( head >> 10 ) & 0x3 ) == 0x3 )	return 0;	// sampling_freq != 0x3

  if((( head >> 19) & 1 ) == 1 &&
	(( head >> 17 ) & 3 ) == 3 &&
	(( head >> 16 ) & 1 ) == 1 )	return 0;	// MPEG 2.0, layer 1, err_prot

  if(( head & 0xffff0000 ) == 0xfffe0000 )	return 0;

  return 1;
}

static unsigned long read_head(struct mpstr *mp, mpg123_reader_t *reader, void *priv )
{
	unsigned long head;

	reader->read( priv, &head, 1 );
	head <<= 8;
	reader->read( priv, &head, 1 );
	head <<= 8;
	reader->read( priv, &head, 1 );
	head <<= 8;
	reader->read( priv, &head, 1 );

	while( 1 )
	{
	    if( !head_check( head ) || !decode_header( &mp->fr, head ))
	    {
		int i = 0;
		unsigned char tmp[4];

		/* step in byte steps through next 64K */
		do
		{
		    i++;

		    if( reader->read( priv, tmp, 1 ) != 1 )	return 0;

		    head <<= 8;
		    head |= tmp[0];
		    head &= 0xffffffff;

		    if( head_check( head ))	break;
//		    {
//			if( mpg123->oldhead )
//			{
//				if(( mpg123->oldhead & HDRCONSTMASK ) ==
//					( newhead & HDRCONSTMASK ))	break;
//			}
//			else	break;
//		    }
		} while( i < 65536 );

		if( i == 65536 )	return 0;
	    }
	    else break;
	}

	return head;
}
