/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * iotest.c
 *
 * Demonstrates:
 *  - the use of I/O plugins
 *  - handling of events ( TOC, Auth )
 *
 * Uses:
 *  - libxmm		- LinuX MultiMedia Library
 *
 * Usage:
 *  iotest <filename>	- filename has to be any kind of input ( file, http, VCD )
 *
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/*
 * Includes
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>

#include <libxmm/libxmm.h>
#include <libxmm/util/utils.h>


/*
 * Definitions
 */

#define	BLOCKSIZE	4096

/*
 * Event handler
 */

static int EventCB( void *priv, XMM_Event *event );

/*
 * main()
 */

int main( int argc, char *argv[] )
{
  void			*xmm;
  XMM_PluginIO		*pIO;
  FILE			*stream;
  uint8_t		data[BLOCKSIZE];

  /* Initialize libxmm */  
  if(( xmm = xmm_Init()) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return 1;
  }

  /* Set event handler */
  xmm_SetEventCB( xmm, EventCB, xmm, 0 );

  /* Open I/O */
  if(( pIO = xmm_IOOpen( xmm, argv[1], XMM_IO_READ )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_Exit( xmm );
	return 1;
  }

  printf( "File %s: %li bytes\n", argv[1], xmm_IOSize( pIO ));

  stream = fopen( "vcd-track-output.mpeg", "w" );

{
  int done = 0;

  while( 1 )
  {
	if( xmm_IOEof( pIO ))	break;	

	xmm_IORead( pIO, data, BLOCKSIZE, 1 );
	fwrite( data, BLOCKSIZE, 1, stream );

	done += BLOCKSIZE;
	fprintf( stderr, "%i done\r", done );
  }
}

  fclose( stream );

  /* Close I/O */
  xmm_IOClose( pIO );

  /* Free libxmm */
  xmm_Exit( xmm );

  return 0;
}

/*
 * Event handler
 * This probably never will be called. It only demonstrates the
 * TOC and Authorization events.
 * We poll for key events in main(), as we need the data defined
 * there for scaling...
 */

static int EventCB( void *priv, XMM_Event *event )
{

  switch( event->type )
  {
	/* Select from Table of Contents */
	case XMM_EVENT_TOC:
		{
		    int i;
		    XMM_Event_TOC *etoc = (XMM_Event_TOC *) event;

		    fprintf( stdout, "\nTable of Contents:\n" );
		    for( i = 0; i < etoc->entries; i++ )	fprintf( stdout, "\t%2i: %s\n", i, etoc->entry[i] );
		
		    fprintf( stdout, "Please make your selection: " );
		    scanf( "%i", &etoc->selected );
		    return 0;
		}

	/* Authorization requiered */
	case XMM_EVENT_AUTH:
		{
		    char *passwd;
		    XMM_Event_Auth *eauth = (XMM_Event_Auth *) event;

		    fprintf( stdout, "\nAuthorization required:\n" );
		    fprintf( stdout, "User: " );
		    scanf( "%s", eauth->user );
		    passwd = getpass( "Password:" );
		    if( passwd == NULL )	break;
		    strcpy( eauth->passwd, passwd );
		    return 0;
		}

	/* Unknown Event: return error */
	default:
		fprintf( stderr, "ERROR: Unable to handler event of type %i\n", event->type );
		break;  
  }

  return XMM_RET_ERROR;
}
