/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stest.c
 *
 * Demonstrates:
 *  - the use of sound plugins ( including control() API )
 *
 * Uses:
 *  - libxmm		- LinuX MultiMedia Library
 *  - libm		- C Math Library
 *
 * Usage:
 *  stest
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

#define	PCM_BUFFER_SIZE		4096

#define	TONEFREQ		1000
#define	SAMPLERATE		48000
#define	CHANNELS		1
#undef	FORMAT_BE		1

/*
 * Includes
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <libxmm/libxmm.h>
#include <libxmm/endian.h>
#include <libxmm/util/utils.h>

/*
 * Sound plugin, that should be used
 */

#define SOUND_PLUGIN LIBDIR "Plugins/Filter/Output/Audio/oss.so"

/*
 * Prototypes
 */

void tone( uint8_t *data, int samples, int freq, int samplerate );

/*
 * main()
 */

int main( int argc, char *argv[] )
{
  void			*xmm, *buffer;
  XMM_PluginFilterAudio	*pSound;
  XMM_AudioFormat	saf, daf;
  uint32_t		buffersize;
  int			ret, i;

  /* Initialize libxmm */  
  if(( xmm = xmm_Init()) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	return 1;
  }

#ifdef FORMAT_BE
  saf.format = XMM_AUDIO_FMT_S16BE;
#else
  saf.format = XMM_AUDIO_FMT_S16LE;
#endif
  saf.samprate = SAMPLERATE;
  saf.channels = CHANNELS;
  saf.bitrate = 64000;	//SAMPLERATE * saf.channels * 2;

  /* Open Sound Plugin */
  if(( pSound = xmm_FilterAudioLoad( SOUND_PLUGIN, xmm, &saf, &daf, 0 )) == NULL )
  {
	fprintf( stderr, "ERROR: %s\n", xmm_ErrorString( xmm ));
	xmm_Exit( xmm );
	return 1;
  }

  /* Get buffer size */
  ret = xmm_FilterAudioControl( pSound, XMM_CTLGET_BUFFER_TOTAL, 0, &buffersize );
  if( ret != XMM_CTLRET_ARG )	fprintf( stderr, "WARNING: Unable to get buffer size.\n" );

#if 0
  /* Check if format supported */
  ret = xmm_FilterAudioControl( pSound, XMM_CTLQUERY_AFORMAT, 0, &sformat );
  if( ret == XMM_CTLRET_TRUE )	fprintf( stdout, "INFO: Sound format supported.\n" );
  else	if( ret == XMM_CTLRET_FALSE )	fprintf( stdout, "INFO: Sound format not supported.\n" );
  else	fprintf( stderr, "WARNING: Unable to query sound format.\n" );
#endif

  /* output */
  if(( buffer = malloc( buffersize )) == NULL )
  {
	fprintf( stderr, "ERROR: Unable to allocate buffer.\n" );
	xmm_FilterAudioClose( pSound );
	xmm_Exit( xmm );
	return 1;
  }

  for( i = 0; i < 40; i++ )
  {
	tone( buffer, buffersize / 2, TONEFREQ, SAMPLERATE );
	xmm_FilterAudioProcess( pSound, buffer, buffersize, NULL, NULL, NULL );
  }

  free( buffer );

  /* Close sound output */
  xmm_FilterAudioClose( pSound );

  /* Free libxmm */
  xmm_Exit( xmm );

  return 0;
}

/*
 * Below is no interesting code, related to LinuX MultiMedia Project
 */

void tone( uint8_t *data, int samples, int freq, int samplerate )
{
  int		i;
  double	scale;
  static int	x = 0;
  int16_t	tmp;

  scale = ( 2 * 3.14159265359 * freq ) / samplerate;

  for( i = 0; i < samples; i++ )
  {
	tmp = (int16_t) (( 1 << 15 ) - 1 ) * sin( scale * x++ );
#ifdef FORMAT_BE
	((int16_t *)data)[i] = XMM_INT32_BE( tmp );
#else
	((int16_t *)data)[i] = tmp;
#endif
  }

  if( x > samplerate )	x -= samplerate;
}
