/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_achain_h
#define INC_libxmm_achain_h

/** @file achain.h
 * Linux MultiMedia Project API: Audio Filter Layer ( Chain )
 *
 * The audio filter chain does a sequence of operations on the
 * audio data ( decoding, conversions, filters, encoding, ... )
 */

#include <libxmm/afilter.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Types
 */

/**
 * Audio filter chain node.
 * Complete information on a node in the audio filter chain
 */
typedef struct xmm_FilterAChainNode_s
{
	XMM_PluginFilterAudio	*filter;	/**< filter */

	XMM_AudioFormat		saf;		/**< input format */
	XMM_AudioFormat		daf;		/**< output format */

	uint32_t		size;		/**< work buffer size */
	void			*data;		/**< work buffer for filter */
} XMM_FilterAChainNode;

/*
 * Prototypes
 */

	/**
	 * Init chain.
	 * Initialize audio filter chain/subsystem.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainInit( void *xmm );

	/**
	 * Free chain.
	 * Free audio filter chain/subsystem.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainExit( void *xmm );

	/**
	 * Lock chain.
	 * Lock audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainLock( void *xmm );

	/**
	 * Unlock chain.
	 * Unlock audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainUnlock( void *xmm );

	/**
	 * Chain status.
	 * Check if the audio filter chain is active.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion 1 is returned if chain is active,
	 * 0 if not. Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainActive( void *xmm );

	/**
	 * Start chain.
	 * Start audio filter chain. All needed filters will be started.
	 *
	 * \param xmm XMM object
	 * \param iaf input audio format. ( e.g. format from DeMUX, with extra data )
	 * \param saf the fixed [by the codec] audio format, without extra data
	 * \param fai supported audio formats ( by sound output or your application )
	 * \param fmtidx idx of format, that should be used, -1 for autoselect
	 * \param coaf address of a XMM_AudioFormat address: chain output audio format
	 * \param blocksize size of the output buffer, for sound output this
	 * normally is the fragment size.
	 *
	 * \return Upon successful completion the buffer size is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainPrepare( void *_xmm, XMM_AudioFormat *iaf, XMM_FilterAudioInfo *fai, int fmtidx, XMM_AudioFormat **coaf );

int xmm_FilterAChainStart( void *_xmm, int blocksize );

int xmm_FilterAChainSourceSize( void *_xmm, uint32_t dsize, uint32_t *ssize );

	/**
	 * Process chain.
	 * Process audio filter chain.
	 *
	 * \param xmm XMM object.
	 * \param sdata source data.
	 * \param ssize source data size.
	 * \param ddata address of a pointer. destination buffer.
	 * \param dsize destination data size.
	 * \param flags flags
	 * \param samples number of decoded samples
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChain( void *_xmm, uint8_t *sdata, uint32_t ssize, uint8_t **ddata, uint32_t *dsize, uint32_t *flags, uint32_t *samples );

	/**
	 * Stop chain.
	 * Stop audio filter chain.
	 *
	 * \note Will stop all stuff started in xmm_FilterAChainStart().
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainStop( void *xmm );

	/**
	 * Add plugin.
	 * Add plugin to audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainAdd( void *xmm, XMM_PluginFilterAudio *pfa );

	/**
	 * Add plugin list.
	 * Add plugin list to audio filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainAddList( void *xmm, XMM_List *pAudioFilterList );

	/**
	 * Remove plugins.
	 * Remove all plugins from audio filter chain
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterAChainFree( void *xmm );


#ifdef __cplusplus
}
#endif

#endif
