/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_afilter_h
#define INC_libxmm_afilter_h

/** @file afilter.h
 * Linux MultiMedia Project API: Audio Filter Layer ( Plugin )
 *
 * Audio Filter Plugins are used to process audio data.
 * ( decode audio, modify audio, output audio, visualization ).
 */

#include <libxmm/plugin.h>
#include <libxmm/aformat.h>
#include <libxmm/util/buffer.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_FILTER_AOF_ Audio filter (open) flags
 *
 * Audio filter (open) flags
 *
 * @{
 */

	/** Only query plugin */
#define	XMM_FILTER_AOF_QUERY		0x0080
	/** Compare source and destination channel number */
#define	XMM_FILTER_AOF_CMPCHAN		0x0100
	/** Compare source and destination sample rate */
#define	XMM_FILTER_AOF_CMPSRATE		0x0200
	/** Compare source and destination bitrate */
#define	XMM_FILTER_AOF_CMPBRATE		0x0400
	/** Destination is read-only */
#define	XMM_FILTER_AOF_DESTRO		0x0800
	/** Compare channels, samprate, bitrate. Destination is read-only */
#define	XMM_FILTER_AOF_MEMBERS		( XMM_FILTER_AOF_DESTRO | XMM_FILTER_AOF_CMPCHAN | XMM_FILTER_AOF_CMPSRATE | XMM_FILTER_AOF_CMPBRATE )

/** @} */

/** @defgroup XMM_FILTER_FLAG_ Filter flags
 *
 * Filter flags
 *
 * @{
 */

	/** Audio codec plugin */
#define	XMM_FILTER_FLAG_ACODEC		0x0100
	/** Audio output plugin */
#define	XMM_FILTER_FLAG_AOUTPUT		0x0200

/** @} */

/** @defgroup XMM_FILTER_ACF_ Audio filter capabilities / flags
 *
 * Audio filter capabilities / flags
 *
 * @{
 */

	/** Plugin is filter.
	 * Plugin supports modification of the data within the
	 * same codec and format. */
#define	XMM_FILTER_ACF_FILTER		0x00000100
	/** Plugin is codec.
	 * Plugin supports conversion between two different codecs. */
#define	XMM_FILTER_ACF_CODEC		0x00000200
	/** Plugin is converter.
	 * Plugin supports conversion between two different formats
	 * of the same codec. */
#define	XMM_FILTER_ACF_CONVERTER	0x00000400
	/** Plugin does output.
	 * Plugin supports output. Data will not be modified. */
#define	XMM_FILTER_ACF_OUTPUT		0x00000800

	/** Plugin is able to decode data */
#define	XMM_FILTER_ACF_DECODE		0x00001000
	/** Plugin is able to encode data */
#define	XMM_FILTER_ACF_ENCODE		0x00002000

	/** Autoselect. Format for output may be choosen by system. */
#define	XMM_FILTER_ACF_AUTOSELECT	0x00010000

/** @} */

/**
 * Audio filter info.
 * Information on the formats supported by a audio filter plugin.
 */

typedef struct xmm_FilterAudioInfo_s
{
    uint32_t			caps;		/**< filter capabilities */

    char			file[32];	/**< filter plugin file name */
    char			name[32];	/**< filter name */
    char			desc[96];	/**< filter description */
    char			copy[96];	/**< filter copyright info */

    int				nfmt;		/**< number of supported audio formats */
    XMM_AudioFormat		*fmt;		/**< supported audio formats */

} XMM_FilterAudioInfo;

/**
 * Audio Filter Plugin.
 * This structure / type represents an audio filter plugin.
 */

typedef struct xmm_PluginFilterAudio_s	XMM_PluginFilterAudio;

struct xmm_PluginFilterAudio_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open filter.
     * Initialize / Open filter plugin for the desired format conversion.
     *
     * \param xmm XMM object
     * \param saf source audio format
     * \param daf destination audio format
     * \param flags definitions from the #XMM_FILTER_AOF_ group.
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     * If #XMM_FILTER_AOF_QUERY was used in flags, NULL will be returned if
     * filterc supports desired format conversions, otherwise an error code
     * ( negative value ).
     *
     * \note The desc member of the #XMM_AudioFormat struct (daf) will be set.
     */
    XMM_PluginFilterAudio *	(*Open)( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags );
    /** xmm_FilterAudioClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginFilterAudio *filter );
    /** xmm_FilterAudioControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data );
    /** xmm_FilterAudioProcess() is a wrapper for this member */
    int				(*Process)( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags );
    /** xmm_FilterAudioProcessBQ() is a wrapper for this member */
    int				(*ProcessBQ)( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags );
};

/*
 * Prototypes
 */

	/**
	 * Load audio filter.
	 * Load audio filter plugin and initialize it.
	 *
	 * \param filename audio filter plugin
	 * \param xmm XMM object
	 * \param saf source audio format
	 * \param daf destination audio format
	 * \param flags definitions from the #XMM_FILTER_AOF_ group.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 *
	 * \note The desc member of the #XMM_AudioFormat struct (daf)
	 * will be set.
	 */
XMM_PluginFilterAudio	*xmm_FilterAudioLoad( char *filename, void *_xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags );

	/**
	 * Find / Load audio filter.
	 * Find plugin, that is able to handle the specified audio format
	 * conversion and open it.
	 *
	 * \param xmm XMM object
	 * \param saf source audio format
	 * \param daf destination audio format
	 * \param flags definitions from the #XMM_FILTER_AOF_ group.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 *
	 * \note The desc member of the #XMM_AudioFormat struct (daf)
	 * will be set.
	 */
XMM_PluginFilterAudio	*xmm_FilterAudioFind( void *_xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags );

	/**
	 * (Re)open audio filter.
	 * Open/Initialize new audio filter from an existing audio filter.
	 *
	 * \param filter Address of audio filter plugin.
	 * \param saf source audio format
	 * \param daf destination audio format
	 * \param flags definitions from the #XMM_FILTER_AOF_ group.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 *
	 * \note The desc member of the #XMM_AudioFormat struct (daf)
	 * will be set.
	 */
XMM_PluginFilterAudio	*xmm_FilterAudioOpen( XMM_PluginFilterAudio *filter, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags );

	/**
	 * Close filter / Free plugin.
	 *
	 * \param filter Address of audio filter plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_FilterAudioCloseFree( XMM_PluginFilterAudio *filter );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close filter.
	 * Close filter and free all resources used by filter.
	 *
	 * \param filter Address of audio filter plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioClose( XMM_PluginFilterAudio *filter );

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param filter Address of audio filter plugin.
	 * \param cmd command.
	 * \param param 1st argument. An integer argument.
	 * \param data 2nd argument. Address of data.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioControl( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data );

	/**
	 * Process data.
	 * Process some data.
	 *
	 * \param filter Address of audio filter plugin.
	 * \param src Address of input data (buffer).
	 * \param isize Size of input data.
	 * \param dest Address of output data (buffer).
	 * \param osize Address used to store size of output data.
	 * \param flags Address of the flag word.
	 *
	 * \return Upon successful completion the size of the used input data
	 * is returned. Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioProcess( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags );

	/**
	 * Process data.
	 * Process some data from a buffer queue.
	 *
	 * \note Not supported by all filter plugins.
	 *
	 * \param filter Address of audio filter plugin.
	 * \param bq Address of buffer queue (input data).
	 * \param dest Address of output data (buffer).
	 * \param osize Address used to store size of output data.
	 * \param flags Address of the flag word.
	 *
	 * \return Upon successful completion the size of the used input data
	 * is returned. Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterAudioProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Get filter info list.
	 * Get information about installed audio filters.
	 *
	 * \param xmm XMM object
	 * \param flags Flags from the #XMM_FILTER_ACF_ group.
	 *
	 * \return List of XMM_FilterAudioInfo objects.
	 */
XMM_List *xmm_FilterAudioInfoList( void *xmm, uint32_t flags );

	/*
	 * Get audio filter info.
	 * Get information about a specified audio filter.
	 *
	 * \param xmm XMM object
	 * \param filename audio filter filename
	 *
	 * \return Address of an info structure. Should be freed after use.
	 */
XMM_FilterAudioInfo *xmm_FilterAudioInfo( void *_xmm, char *filename );

#ifdef __cplusplus
}
#endif

#endif
