/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_aformat_h
#define INC_libxmm_aformat_h

/** @file aformat.h
 * Linux MultiMedia Project API: Audio format
 *
 * Audio format is a complete description of an audio format.
 */

#include <inttypes.h>

#ifdef __cplusplus
extern "C"
{
#endif


/** @defgroup XMM_AUDIO_MASK_ Audio format masks
 *
 * Audio format masks
 *
 * @{
 */
	/** Codec. Codec used to encode data. */
#define	XMM_AUDIO_MASK_CODEC		0xFFFF0000
	/** Signed. Data is signed. */
#define	XMM_AUDIO_MASK_SIGNED		0x00008000
	/** Big endian. Data is in big endian. */
#define	XMM_AUDIO_MASK_BE		0x00004000
	/** Size. Sample size in bits ( e.g. 8 or 16 ). */
#define	XMM_AUDIO_MASK_SIZE		0x000000FF

/** @} */
/** @defgroup XMM_AUDIO_CODEC_ Audio codecs
 *
 * Audio codecs
 *
 * @{
 */

	/** PCM. */
#define	XMM_AUDIO_CODEC_PCM		0x00010000
	/** IEEE. */
#define	XMM_AUDIO_CODEC_IEEE		0x00020000
	/** u-Law. CCITT G.711 u-law */
#define	XMM_AUDIO_CODEC_ULAW		0x00030000
	/** a-Law. CCITT G.711 a-law */
#define	XMM_AUDIO_CODEC_ALAW		0x00040000
	/** 4-bit ADPCM. CCITT G.721 ( 4-bit ADPCM )*/
#define	XMM_AUDIO_CODEC_G721		0x00050000
	/** 3-bit, 5-bit ADPCM. CCITT G.723 ( 3-bit, 5-bit ADPCM )*/
#define	XMM_AUDIO_CODEC_G723		0x00060000
	/** 2-bit ADPCM. CCITT G.726 ( 2-bit ADPCM )*/
#define	XMM_AUDIO_CODEC_G726		0x00070000
	/** MPEG. MPEG 1.0/2.0 Audio Layer 1/2/3. */
#define	XMM_AUDIO_CODEC_MPEG		0x00080000
	/** AC3. Dolby Digital (5.1) */
#define	XMM_AUDIO_CODEC_AC3		0x00090000
	/** GSM 610. Micro$oft GSM 6.10 */
#define	XMM_AUDIO_CODEC_GSM610MS	0x000A0000
	/** Ogg/Vorbis. Ogg/Vorbis audio */
#define	XMM_AUDIO_CODEC_VORBIS		0x000B0000

	
	/** DivX ;-) Audio ( WM-Audio V1 ) */
#define	XMM_AUDIO_CODEC_WMA1		0x80010000
	/** DivX ;-) Audio ( WM-Audio V2 ) */
#define	XMM_AUDIO_CODEC_WMA2		0x80020000
	/** M$-ADPCM */
#define	XMM_AUDIO_CODEC_ADPCM_MS	0x80030000
	/** IMA/DVI-ADPCM */
#define	XMM_AUDIO_CODEC_ADPCM_IMA	0x80040000
	/** IMA/DVI-ADPCM */
#define	XMM_AUDIO_CODEC_VOX_META	0x80750000

/** @} */
/** @defgroup XMM_AUDIO_FMT_ Audio formats
 *
 * Audio formats
 *
 * @{
 */
	/** Unsigned 4 bit, needed for ADPCM */
#define XMM_AUDIO_FMT_U4		( 0x04 | XMM_AUDIO_CODEC_PCM )
	/** Unsigned 8 bit */
#define XMM_AUDIO_FMT_U8		( 0x08 | XMM_AUDIO_CODEC_PCM )
	/** Signed 16 bit Little endian */
#define XMM_AUDIO_FMT_S16LE		( 0x10 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED )
	/** Signed 16 bit Big endian */
#define XMM_AUDIO_FMT_S16BE		( 0x10 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED | XMM_AUDIO_MASK_BE )
	/** Signed 24 bit Little endian */
#define XMM_AUDIO_FMT_S24LE		( 0x18 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED )
	/** Signed 24 bit Big endian */
#define XMM_AUDIO_FMT_S24BE		( 0x18 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED | XMM_AUDIO_MASK_BE )
	/** Signed 32 bit Little endian */
#define XMM_AUDIO_FMT_S32LE		( 0x20 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED )
	/** Signed 32 bit Big endian */
#define XMM_AUDIO_FMT_S32BE		( 0x20 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED | XMM_AUDIO_MASK_BE )
	/** Signed 8 bit */
#define XMM_AUDIO_FMT_S8		( 0x08 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_SIGNED )
	/** Unsigned 16 bit Little endian */
#define XMM_AUDIO_FMT_U16LE		( 0x10 | XMM_AUDIO_CODEC_PCM )
	/** Unsigned 16 bit Big endian */
#define XMM_AUDIO_FMT_U16BE		( 0x10 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_BE )
	/** Unsigned 24 bit Little endian */
#define XMM_AUDIO_FMT_U24LE		( 0x18 | XMM_AUDIO_CODEC_PCM )
	/** Unsigned 24 bit Big endian */
#define XMM_AUDIO_FMT_U24BE		( 0x18 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_BE )
	/** Unsigned 32 bit Little endian */
#define XMM_AUDIO_FMT_U32LE		( 0x20 | XMM_AUDIO_CODEC_PCM )
	/** Unsigned 32 bit Big endian */
#define XMM_AUDIO_FMT_U32BE		( 0x20 | XMM_AUDIO_CODEC_PCM | XMM_AUDIO_MASK_BE )

	/** IEEE 32 bit floating point Little endian */
#define	XMM_AUDIO_FMT_IEEE32LE		( 0x20 | XMM_AUDIO_CODEC_IEEE | XMM_AUDIO_MASK_SIGNED )
	/** IEEE 32 bit floating point Big endian */
#define	XMM_AUDIO_FMT_IEEE32BE		( 0x20 | XMM_AUDIO_CODEC_IEEE | XMM_AUDIO_MASK_SIGNED | XMM_AUDIO_MASK_BE )
	/** IEEE 64 bit floating point Little endian */
#define	XMM_AUDIO_FMT_IEEE64LE		( 0x40 | XMM_AUDIO_CODEC_IEEE | XMM_AUDIO_MASK_SIGNED )
	/** IEEE 64 bit floating point Big endian */
#define	XMM_AUDIO_FMT_IEEE64BE		( 0x40 | XMM_AUDIO_CODEC_IEEE | XMM_AUDIO_MASK_SIGNED | XMM_AUDIO_MASK_BE )

	/** AC3 */
#define	XMM_AUDIO_FMT_AC3		( 0x10 | XMM_AUDIO_CODEC_AC3 | XMM_AUDIO_MASK_SIGNED )	/* NOT used */
	/** MPEG audio data */
#define	XMM_AUDIO_FMT_MPEG		( 0x10 | XMM_AUDIO_CODEC_MPEG | XMM_AUDIO_MASK_SIGNED )
	/** -Law */
#define	XMM_AUDIO_FMT_ULAW		( 0x08 | XMM_AUDIO_CODEC_ULAW )
	/** A-Law */
#define	XMM_AUDIO_FMT_ALAW		( 0x08 | XMM_AUDIO_CODEC_ALAW )

/** @} */
/** @defgroup XMM_AUDIO_EXT_ Audio extensions
 *
 * Audio extensions ( extraType member )
 *
 * @{
 */
#define	XMM_AUDIO_EXT_WAVE	0x0001

/** @} */

/*
 * Types
 */

/**
 * Audio format.
 * Contains information about the audio data
 */
typedef struct xmm_AudioFormat_s
{
    /** Codec and data format. One of the #XMM_AUDIO_FMT_ values */
    uint32_t			format;
    /** Sample rate. Number of samples per second */
#define	XMM_AUDIO_ANY_SRATE	0xFFFFFFFF
    uint32_t			samprate;
    /** Channels. Number of channels */
#define	XMM_AUDIO_ANY_CHANN	0xFFFF
    uint16_t			channels;
    /** Blocksize. Block size of data / Data alignment */
    uint16_t			blockSize;
    /** Bitrate. Number of bits per second, negative value indicates VBR */
#define	XMM_AUDIO_ANY_BRATE	0x7FFFFFFF
    int32_t			bitrate;
    /** Extrasize. Extra data size */
    uint16_t			extraSize;
    /** Extratype. Type of the extra data. */
    uint16_t			extraType;
    /** Description. Format description */
    char			desc[64];
} XMM_AudioFormat;

/*
 * Prototypes
 */

	/**
	 * Audio Format: codec description.
	 * Fill string, describing a codec of an audio format.
	 *
	 * \param format #XMM_AudioFormat format member
	 * \param string will be filled with codec description
	 *
	 * \note the string has to take 64 charachters ( desc member
	 * of #XMM_AudioFormat )
	 */
void	xmm_aformat_codec_desc( uint32_t format, char *string );

#ifdef __cplusplus
}
#endif

#endif
