/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_endian_h
#define INC_libxmm_endian_h

/** @file endian.h
 * Linux MultiMedia Project: endian-specific stuff
 */

#include <inttypes.h>
#include <xmmsysconf.h>

#ifdef __cplusplus
extern "C"
{
#endif

/* xmmconfig.h defines XMM_BYTE_ORDER which expands to one of these */
#define	XMM_LITTLE_ENDIAN	1234
#define	XMM_BIG_ENDIAN		4321

/*
 * Basic conversion ( little endian <--> big endian )
 */

#define XMM_SWAP16_CONST(val)	((uint16_t)(			\
    (((uint16_t) (val) & (uint16_t) 0x00ffU) << 8) |		\
    (((uint16_t) (val) & (uint16_t) 0xff00U) >> 8)))

#define XMM_SWAP32_CONST(val)	((uint32_t)(			\
    (((uint32_t) (val) & (uint32_t) 0x000000ffU) << 24) |	\
    (((uint32_t) (val) & (uint32_t) 0x0000ff00U) <<  8) |	\
    (((uint32_t) (val) & (uint32_t) 0x00ff0000U) >>  8) |	\
    (((uint32_t) (val) & (uint32_t) 0xff000000U) >> 24)))

#define XMM_SWAP64_CONST(val)	((uint64_t)( \
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x00000000000000ffU)) << 56) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x000000000000ff00U)) << 40) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x0000000000ff0000U)) << 24) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x00000000ff000000U)) <<  8) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x000000ff00000000U)) >>  8) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x0000ff0000000000U)) >> 24) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0x00ff000000000000U)) >> 40) |	\
    (((uint64_t) (val) & (uint64_t) XMM_INT64_CONST(0xff00000000000000U)) >> 56)))

/*
 * * * * optimized for x86 * * *
 */

/* 16 bit */
#if defined (__i386__) && defined (__GNUC__) && __GNUC__ >= 2
#define	XMM_SWAP16_X86(val) (__extension__		\
    ({ register uint16_t __v;				\
	if(__builtin_constant_p(val))			\
	    __v = XMM_SWAP16_CONST(val);		\
	else						\
	    __asm__ __const__ ("rorw $8, %w0"		\
			      : "=r" (__v)		\
			      : "0" ((uint16_t)(val)));	\
	__v; }))

#define	XMM_SWAP16(val)	(XMM_SWAP16_X86(val))
#else
#define	XMM_SWAP16(val)	(XMM_SWAP16_CONST(val))
#endif

/* 32 bit */
#if defined (__i386__) && defined (__GNUC__) && __GNUC__ >= 2
#if !defined(__i486__) && !defined(__i586__) \
      && !defined(__pentium__) && !defined(__i686__) && !defined(__pentiumpro__)
#define XMM_SWAP32_X86(val) (__extension__			\
    ({ register uint32_t __v;					\
	if(__builtin_constant_p(val))				\
	    __v = XMM_SWAP32_CONST(val);			\
	else							\
	    __asm__ __const__ ("rorw $8, %w0\n\t"		\
			       "rorl $16, %0\n\t"		\
			       "rorw $8, %w0"			\
			       : "=r" (__v)			\
			       : "0" ((uint32_t) (val)));	\
	__v; }))
#else /* 486, pentium, ... have bswap */
#define XMM_SWAP32_X86(val) (__extension__			\
    ({ register uint32_t __v;					\
	if(__builtin_constant_p(val))				\
	    __v = XMM_SWAP32_CONST(val);			\
	else							\
	    __asm__ __const__ ("bswap %0"			\
			       : "=r" (__v)			\
			       : "0" ((uint32_t) (val)));	\
	__v; }))

#endif
#define XMM_SWAP32(val)	(XMM_SWAP32_X86(val))
#else
#define XMM_SWAP32(val)	(XMM_SWAP32_CONST(val))
#endif

/* 64 bit */
#if defined (__i386__) && defined (__GNUC__) && __GNUC__ >= 2
#define XMM_SWAP64_X86(val) (__extension__			\
    ({ union { uint64_t __ll; uint32_t __l[2]; } __r;		\
	if (__builtin_constant_p(val))				\
	    __r.__ll = XMM_SWAP64_CONST(val);			\
	else							\
	    {							\
		union { uint64_t __ll; uint32_t __l[2]; } __w;	\
		__w.__ll = ((uint64_t) val);			\
		__r.__l[0] = XMM_SWAP32(__w.__l[1]);		\
		__r.__l[1] = XMM_SWAP32(__w.__l[0]);		\
	    }							\
	__r.__ll; }))

#define XMM_SWAP64(val)	(XMM_SWAP64_X86(val))
#else
#define XMM_SWAP64(val)	(XMM_SWAP64_CONST(val))
#endif

/*
 *
 */

#if XMM_BYTE_ORDER == XMM_LITTLE_ENDIAN
#define	XMM_INT16_LE(val)	(val)
#define	XMM_INT32_LE(val)	(val)
#define	XMM_INT64_LE(val)	(val)
#define	XMM_INT16_BE(val)	(XMM_SWAP16(val))
#define	XMM_INT32_BE(val)	(XMM_SWAP32(val))
#define	XMM_INT64_BE(val)	(XMM_SWAP64(val))
#else
#define	XMM_INT16_LE(val)	(XMM_SWAP16(val))
#define	XMM_INT32_LE(val)	(XMM_SWAP32(val))
#define	XMM_INT64_LE(val)	(XMM_SWAP64(val))
#define	XMM_INT16_BE(val)	(val)
#define	XMM_INT32_BE(val)	(val)
#define	XMM_INT64_BE(val)	(val)
#endif

#ifdef __cplusplus
}
#endif

#endif
