/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpfilterv_h
#define INC_libxmm_lpfilterv_h

/** @file lpfilterv.h
 * Linux MultiMedia Project API: Video Filter Layer ( Plugin )
 *
 * Video Filter Plugins are used to modify (raw/uncompressed ) video
 * data. For example libxmm uses this plugin type to do conversions
 * if graph format is not supported by Graph Output Plugin.
 * As well, it can be used to apply some effect to the data or
 * it can be used for visualization purposes ( e.g. histogram ).
 *
 * There are some special plugins used for graph conversion.
 * Each plugin is a set of conversion functions. ( e.g. MMX, SSE ).
 * The user / the application has to set the desired set / plugin.
 */

#include <libxmm/plugin.h>
#include <libxmm/lpgraph.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Image format.
 * Contains information about the image
 */

typedef struct xmm_ImageFormat_s
{
    uint32_t			codec;			/**< Format FOURCC */
    uint32_t			width;			/**< Image width */
    uint32_t			height;			/**< Image height */
    uint32_t			flags;			/**< Flags */
    uint32_t			bwidth;			/**< Buffer width */
    uint32_t			bheight;		/**< Buffer height */
} XMM_ImageFormat;

/**
 * Video Filter Plugin.
 * This structure / type represents a video filter plugin.
 */

typedef struct xmm_PluginFilterVideo_s	XMM_PluginFilterVideo;

struct xmm_PluginFilterVideo_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open filter.
     * Initialize / Open filter plugin. This call makes a copy of the
     * plugin data and initializes the plugin.
     *
     * \param xmm XMM object
     * \param sif source video format
     * \param dif destination video format
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginFilterVideo *	(*Open)( void *xmm, XMM_ImageFormat *sif, XMM_ImageFormat *dif );
    /** xmm_FilterVideoClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginFilterVideo *filter );
    /** xmm_FilterVideoControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t param, void *data );
    /** xmm_FilterVideoProc() is a wrapper for this member */
    int				(*Proc)( XMM_PluginFilterVideo *filter, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );
};

/*
 * Prototypes
 */

	/**
	 * Open video filter.
	 * Load video filter plugin and initialize it.
	 *
	 * \param xmm XMM object
	 * \param filename video filter plugin
	 * \param sif source video format
	 * \param dif destination video format
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 */
XMM_PluginFilterVideo	*xmm_FilterVideoOpen( void *xmm, char *filename, XMM_ImageFormat *sif, XMM_ImageFormat *dif );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close filter.
	 * Close filter and free all resources used by plugin.
	 *
	 * \param filter Pointer to a video filter plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterVideoClose( XMM_PluginFilterVideo *filter );

	/**
	 * Control call.
	 * This call is used to control the plugin.
	 *
	 * \param filter Pointer to a filter plugin.
	 * \param cmd command.
	 * \param arg 1st argument. An integer argument.
	 * \param data 2nd argument. Pointer to some data.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterVideoControl( XMM_PluginFilterVideo *filter, uint32_t cmd, uint32_t arg, void *data );

	/**
	 * Process data.
	 * Process some data.
	 *
	 * \param filter Pointer to a video filter plugin.
	 * \param sdata Pointer to source data
	 * \param ddata Pointer to a Pointer. Pointer to the destination data
	 * will be stored there.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 *
	 * \note This is only a wrapper.
	 */
int	xmm_FilterVideoProc( XMM_PluginFilterVideo *filter, uint8_t *sdata[], int stride[], int width, int height, uint8_t *ddata[], int dx, int dy );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Video Filter Chain ---------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Init chain.
	 * Initialize video filter chain/subsystem.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainInit( void *xmm );

	/**
	 * Free chain.
	 * Free video filter chain/subsystem.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainExit( void *xmm );

	/**
	 * Lock chain.
	 * Lock video filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainLock( void *xmm );

	/**
	 * Unlock chain.
	 * Unlock video filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainUnlock( void *xmm );

	/**
	 * Chain status.
	 * Check if the video filter chain is active.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion 1 is returned if chain is active,
	 * 0 if not. Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainActive( void *xmm );

	/**
	 * Input format.
	 * Set Input / Source format for video filter chain.
	 *
	 * \param xmm XMM object
	 * \param format Format FourCC
	 * \param width Image width
	 * \param height Image height
	 * \param flags Flags
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainInput( void *xmm, uint32_t format, int width, int height, int flags );

	/**
	 * Output format.
	 * Set Output / Target format and size for video filter chain.
	 *
	 * \param xmm XMM object
	 * \param format Format FourCC
	 * \param width Image width
	 * \param height Image height
	 * \param flags Flags
	 * \param bwidth Buffer width
	 * \param bheight Buffer height
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainOutput( void *xmm, uint32_t format, int width, int height, int flags, int bwidth, int bheight );

	/**
	 * Write data.
	 * Write some data into the video filter chain.
	 *
	 * \param xmm XMM object
	 * \param data Pointer(s) to the data or planes. When using RGB data or
	 * packed YUV only data[0] is used.
	 * \param stride Bytes per line. Two reasons for this:
	 * - The internal buffer of a decoder may be bigger then the image.
	 * - The lines in the internal buffer of a decoder may be aligned.
	 * \param x horizontal position in the source image.
	 * \param y vertical position in the source image.
	 * \param width horizontal pixels that should be processed.
	 * \param height vertical pixels that should be processed.
	 * \param flags Flags. ( NOT used yet. For future extensions )
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainData( void *_xmm, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags );

	/**
	 * Process chain.
	 * Process video filter chain.
	 *
	 * \param xmm XMM object
	 * \param data Pointer to image buffer.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChain( void *xmm, uint8_t *data[] );

	/**
	 * Stop chain.
	 * Stop video filter chain.
	 *
	 * \note Will stop all stuff started in xmm_FilterVChainOutput().
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainStop( void *xmm );

	/**
	 * Add plugin.
	 * Add plugin to video filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainAdd( void *xmm, XMM_PluginFilterVideo *pfv );

	/**
	 * Add plugin list.
	 * Add plugin list to video filter chain.
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainAddList( void *xmm, XMM_List *pVideoFilterList );

	/**
	 * Remove plugins.
	 * Remove all plugins from video filter chain
	 *
	 * \param xmm XMM object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_FilterVChainFree( void *xmm );

#ifdef __cplusplus
}
#endif

#endif
