/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpinput_h
#define INC_libxmm_lpinput_h

/** @file lpinput.h
 * Linux MultiMedia Project API: Input (DeMUX) Layer ( Plugin )
 *
 * Input (DeMUX) Plugins are used to read decompressed data. In direct stream
 * copy mode, it maybe compressed data ( e.g. MJPEG ).
 * Plugins of this type support up to 2 play modes.
 * - Read mode: The stream type specific part of the interface can be used
 *   to get data. The application has to do the synchronization if needed.
 * - Direct out mode: The plugin handles the data and passed it directly to
 *   the Sound/Graph Output or somewhere else. "Somewhere else" is e.g. used
 *   by the CDDA ( CD Digital Audio ) plugin in direct out mode. The plugin
 *   uses the audio connection between the CD driver and the sound card.
 *   This playback mode, will need (almost) no CPU resources.
 *   In direct out mode, the playback will be controlled using the
 *   Control() call and XMM_CTLINP_ commands.
 */

#include <libxmm/plugin.h>
#include <libxmm/aformat.h>
#include <libxmm/lpcodecv.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_INPUT_CF_ Input (DeMUX) Capabilities and Flags
 *
 * Input (DeMUX) Capabilities and Flags
 *
 * @{
 */

	/** Read mode. stream type specific functions can be used. */
#define	XMM_INPUT_CF_MODE_READ		0x00000001
	/** Direct out mode. Use XMMP output subsystem or device. */
#define	XMM_INPUT_CF_MODE_DOUT		0x00000002
	/** Video output */
#define	XMM_INPUT_CF_VIDEO		0x00000010
	/** Audio output */
#define	XMM_INPUT_CF_AUDIO		0x00000020
	/** Volume. Volume can be get/set using Control() call */
#define	XMM_INPUT_CF_VOLUME		0x00000100

/** @} */
/** @defgroup XMM_PBS_ Playback status 
 *
 * Playback status
 *
 * @{
 */

	/** Stopped. Not playback at this time */
#define	XMM_PBS_STOPPED			0x0001
	/** Playing. Playback is running. */
#define	XMM_PBS_PLAYING			0x0002
	/** Paused. Playback is paused. */
#define	XMM_PBS_PAUSED			0x0004
	/** Error. Unkown status or error while playback. */
#define	XMM_PBS_ERROR			0x0008

/** @} */
/** @defgroup XMM_CIF_ Valid #XMM_ClipInfo members
 *
 * Valid members of the #XMM_ClipInfo type
 *
 * @{
 */

#define XMM_CIF_NAME			0x00010000
#define XMM_CIF_AUTHOR			0x00020000
#define XMM_CIF_ALBUM			0x00040000
#define XMM_CIF_CONTENT			0x00080000
#define XMM_CIF_COPYRIGHT		0x00100000
#define XMM_CIF_SOFTWARE		0x00200000
#define XMM_CIF_DATE			0x00400000
#define XMM_CIF_COMMENT			0x00800000

/** @} */
/** @defgroup XMM_CIL_ Length of #XMM_ClipInfo (string) members
 *
 * Length of #XMM_ClipInfo (string) members
 *
 * @{
 */

#define XMM_CIL_NAME			64
#define XMM_CIL_AUTHOR			64
#define XMM_CIL_ALBUM			64
#define XMM_CIL_CONTENT			64
#define XMM_CIL_COPYRIGHT		64
#define XMM_CIL_SOFTWARE		64
#define XMM_CIL_DATE			9
#define XMM_CIL_COMMENT			107

/** @} */

	/** Maximum stream number. Maximum number of streams for each type. */
#define	XMM_MAXSTREAMS			32

/**
 * Audio info.
 * Contains audio stream information
 */

typedef struct xmm_AudioInfo_s
{
    /** Audio format. Contains audio format information. */
    XMM_AudioFormat		fmt;
    /** Samples. Total number of samples in the stream. */
    uint32_t			tSamples;
    /** Size. Total stream size. */
    uint32_t			tSize;
    /** Offset. Stream offset ( ms ). */
    uint32_t			offset;
} XMM_AudioInfo;

/**
 * Video info.
 * Contains video stream information
 */

typedef struct xmm_VideoInfo_s
{
    XMM_VideoFormat		fmt;
    /** Frames. Total number of frames in the stream. */
    uint32_t			tFrames;
    /** Size. Total stream size. */
    uint32_t			tSize;
    /** Offset. Stream offset ( ms ). */
    uint32_t			offset;
} XMM_VideoInfo;

/**
 * Clip info.
 * Contains some general information about the media.
 */

typedef struct xmm_ClipInfo_s
{
    /** Flags. Flags: XMM_CLIPINFO_ */
    uint32_t			flags;
    /** Name. Presentation name. */
    char			name[XMM_CIL_NAME];
    /** Author. Author / Artist of the presentation. */
    char			author[XMM_CIL_AUTHOR];
    /** Album. Presentation context. */
    char			album[XMM_CIL_ALBUM];
    /** Content. Presentation description. */
    char			content[XMM_CIL_CONTENT];
    /** Copyright. Copyrigth information. */
    char			copyright[XMM_CIL_COPYRIGHT];
    /** Software. Encoder info. */
    char			software[XMM_CIL_SOFTWARE];
    /** Date. Creation date. */
    char			date[XMM_CIL_DATE];
    /** Comment. Additional information */
    char			comment[XMM_CIL_COMMENT];
    /** Size. File size. */
    uint32_t			size;
    /** Playtime. Media length. */
    double			playtime;
} XMM_ClipInfo;

/**
 * File info.
 * Contains information about all Audio/Video streams in file.
 */

typedef struct xmm_FileInfo_s
{
    /** Description of the file format. ( e.g. Ogg/Vorbis, MPEG Program ) */
    char		desc[256];
    /** General presentation info. */
    XMM_ClipInfo	clip;
    /** Video streams. Number of video streams. zero means only audio. */
    int			vstreams;
    /** Video stream info. Stream info for each video stream. */
    XMM_VideoInfo	vi[XMM_MAXSTREAMS];
    /** Audio streams. Number of audio streams. zero means onyl video. */
    int			astreams;
    /** Audio stream info. Stream info for each audio stream. */
    XMM_AudioInfo	ai[XMM_MAXSTREAMS];
} XMM_FileInfo;

/**
 * Input (DeMUX) Plugin.
 * This structure / type represents an Input (DeMUX) plugin.
 */

typedef struct xmm_PluginInput_s	XMM_PluginInput;

struct xmm_PluginInput_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize plugin.
     * Initialize / create plugin. This call makes a copy of the
     * plugin data and initializes the plugin.
     *
     * \param xmm XMM object
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginInput *		(*Init)( void *xmm );

    /** xmm_InputOpen() is a wrapper for this member.
     * \note The wrapper gets an additional argument */
    int				(*Open)( XMM_PluginInput *input, char *filename, int flags );
    /** xmm_InputClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginInput *input );

    /** xmm_InputControl() is a wrapper for this member */
    int				(*Control)( XMM_PluginInput *input, uint32_t cmd, uint32_t arg, void *data );
    /** xmm_InputSeek() is a wrapper for this member */
    int				(*Seek)( XMM_PluginInput *input, int vstream, int astream, double seek );
    /** xmm_InputInfo() is a wrapper for this member */
    int				(*Info)( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval );

    /** xmm_InputAudioStreams() is a wrapper for this member */
    int				(*AudioStreams)( XMM_PluginInput *input );
    /** xmm_InputAudioInfo() is a wrapper for this member */
    int				(*AudioInfo)( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai );
    /** xmm_InputAudioRead() is a wrapper for this member */
    int				(*AudioRead)( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );
    /** xmm_InputAudioPTS() is a wrapper for this member */
    int				(*AudioPTS)( XMM_PluginInput *input, int stream, uint32_t *pts );
    /** xmm_InputAudioSeek() is a wrapper for this member */
    int				(*AudioSeek)( XMM_PluginInput *input, int stream, uint32_t pts );

    /** xmm_InputVideoStreams() is a wrapper for this member */
    int				(*VideoStreams)( XMM_PluginInput *input );
    /** xmm_InputVideoInfo() is a wrapper for this member */
    int				(*VideoInfo)( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame );
    /** xmm_InputVideoRead() is a wrapper for this member */
    int				(*VideoRead)( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags );
    /** xmm_InputVideoPTS() is a wrapper for this member */
    int				(*VideoPTS)( XMM_PluginInput *input, int stream, uint32_t *videoPTS );
    /** xmm_InputVideoSeek() is a wrapper for this member */
    int				(*VideoSeek)( XMM_PluginInput *input, int stream, uint32_t frame );

    /** xmm_InputCheck() is a wrapper for this member.
     * \note The wrapper gets a #XMM_PluginInput object as additional argument. */
    int				(*Check)( void *xmm, char *filename );
    /** xmm_InputInfoFile() is a wrapper for this member.
     * \note The wrapper gets a #XMM_PluginInput object as additional argument. */
    int				(*FileInfo)( void *xmm, char *filename, XMM_ClipInfo *ci );
};

/*
 * Prototypes
 */

	/**
	 * Open file.
	 * Find matching plugin, initialize it and open file.
	 *
	 * \param xmm XMM object
	 * \param filename file to open. Maybe a string starting with type://
	 * \param flags Flags from the #XMM_INPUT_CF_ group.
	 * \param mode pointer to an integer, that will contain the mode.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 *
	 * \note Before calling this, the xmm library has to be initialized.
	 */
XMM_PluginInput	*xmm_InputOpen( void *_xmm, char *filename, int flags, int *mode );

	/**
	 * Close file.
	 * Close file, and free plugin and all used resources.
	 *
	 * \param input pointer to an input plugin
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputClose( XMM_PluginInput *input );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Controll call.
	 * This call is used to control the plugin.
	 *
	 * \param input pointer to an input plugin
	 * \param cmd command.
	 * \param param 1st argument. An integer argument.
	 * \param data 2nd argument. Pointer to some data.
	 *
	 * \return One of the #XMM_CTLRET_ values.
	 */
int	xmm_InputControl( XMM_PluginInput *input, uint32_t cmd, uint32_t param, void *data );

	/**
	 * Seek.
	 * Seek percentual.
	 *
	 * \param input pointer to an input plugin
	 * \param vstream video stream number
	 * \param astream audio stream number
	 * \param seek new position in percent of the total length.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputSeek( XMM_PluginInput *input, int vstream, int astream, double seek );

	/**
	 * File information.
	 * Get file information. e.g.: playtime, current position in percent
	 *
	 * \param input pointer to an input plugin
	 * \param ci pointer to a #XMM_ClipInfo struct
	 * \param seekval current position in percent
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputInfo( XMM_PluginInput *input, XMM_ClipInfo *ci, double *seekval );

	/**
	 * Audio streams.
	 * Get audio stream number.
	 *
	 * \param input pointer to an input plugin
	 *
	 * \return Upon successful completion the stream number is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputAudioStreams( XMM_PluginInput *input );

	/**
	 * Audio information.
	 * Get audio stream information.
	 *
	 * \param input pointer to an input plugin
	 * \param stream audio stream number
	 * \param ai if ai != NULL this struct will be filled with stream info.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputAudioInfo( XMM_PluginInput *input, int stream, XMM_AudioInfo *ai );

	/**
	 * Audio data.
	 * Read audio data.
	 *
	 * \param input pointer to an input plugin
	 * \param stream audio stream number
	 * \param buffer pointer to an area, where data will be stored.
	 * \note memory are must be large enough for the requested data amount.
	 * \param size requested number of bytes.
	 * \param flags Flags ( for future use )
	 *
	 * \return Upon successful completion the number of read samples is
	 * retured. Otherwise an error code ( negative value ).
	 */
int	xmm_InputAudioRead( XMM_PluginInput *input, int stream, uint8_t *buffer, uint32_t size, int flags );

	/**
	 * Audio timestamp.
	 * Get audio timestamp.
	 *
	 * \param input pointer to an input plugin
	 * \param stream video stream number
	 * \param pts Used to store the audio timestamp ( milliseconds ).
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InputAudioPTS( XMM_PluginInput *input, int stream, uint32_t *pts );

	/**
	 * Seek audio stream.
	 * Seek to requested sample in audio stream.
	 *
	 * \param input pointer to an input plugin
	 * \param stream audio stream number
	 * \param pts new stream position
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InputAudioSeek( XMM_PluginInput *input, int stream, uint32_t pts );

	/**
	 * Video streams.
	 * Get video stream number.
	 *
	 * \param input pointer to an input plugin
	 *
	 * \return Upon successful completion the stream number is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputVideoStreams( XMM_PluginInput *input );

	/**
	 * Video information.
	 * Get video stream information.
	 *
	 * \param input pointer to an input plugin
	 * \param stream video stream number
	 * \param vi if vi != NULL this struct will be filled with stream info.
	 * \param cFrame if cFrame != NULL the current sample number will be
	 * stored.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_InputVideoInfo( XMM_PluginInput *input, int stream, XMM_VideoInfo *vi, uint32_t *cFrame );

	/**
	 * Video data.
	 * Read video data.
	 *
	 * \param input pointer to an input plugin
	 * \param stream video stream number
	 * \param buffer 3 pointers to the data planes. For RGB and packed YUV
	 * only the first is used.
	 * \param flags May contain value '1', in this case frame will be dropped
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InputVideoRead( XMM_PluginInput *input, int stream, uint8_t *buffer[], int flags );

	/**
	 * Video timestamp.
	 * Get video timestamp.
	 *
	 * \param input pointer to an input plugin
	 * \param stream video stream number
	 * \param videoPTS Used to store the video timestamp ( milliseconds ).
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InputVideoPTS( XMM_PluginInput *input, int stream, uint32_t *videoPTS );

	/**
	 * Seek video stream.
	 * Seek to requested frame in video stream.
	 *
	 * \param input pointer to an input plugin
	 * \param stream video stream number
	 * \param frame new stream position
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_InputVideoSeek( XMM_PluginInput *input, int stream, uint32_t frame );

	/**
	 * Check.
	 * Check, if file supported by plugin.
	 *
	 * \param input pointer to an input plugin
	 * \param xmm XMM object
	 * \param filename filename to check
	 *
	 * \return Upon successful completion 1 will be returned if filetype
	 * supported by plugin, 0 if not.
	 * Otherwise an error code ( negative value ).
	 *
	 * \note The plugin has NOT to be initialized before calling this.
	 */
int	xmm_InputCheck( XMM_PluginInput *input, void *xmm, char *filename );

	/**
	 * File information.
	 * Get information about a file of a supported type.
	 *
	 * \param input pointer to an input plugin
	 * \param xmm XMM object
	 * \param filename filename to check
	 * \param ci pointer to a clip info struct.
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned
	 * Otherwise an error code ( negative value ).
	 *
	 * \note The plugin has NOT to be initialized before calling this.
	 */
int	xmm_InputFileInfo( XMM_PluginInput *input, void *xmm, char *filename, XMM_ClipInfo *ci );

#ifdef __cplusplus
}
#endif

#endif
