/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_lpio_h
#define INC_libxmm_lpio_h

/** @file lpio.h
 * Linux MultiMedia Project API: (raw) Input/Output Layer ( Plugin )
 *
 * I/O Plugins are used to read or write (raw) data. Plugins of this layer have
 * no knowledge of the data type. In read mode, they simply read data from a
 * file, from the net or from a device ( VCD, DVD ).
 * Some plugins support writing, as well.
 */

#include <libxmm/plugin.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** @defgroup XMM_IO_ I/O modes
 *
 * I/O open mode
 *
 * @{
 */

	/** Open I/O stream for reading. */
#define XMM_IO_READ			0
	/** Open I/O stream for writing. */
#define XMM_IO_WRITE			1
	/** Open I/O stream for writing ( with append ). */
#define XMM_IO_APPEND			2

/** @} */
/** @defgroup XMM_SEEK_ Seek modes
 *
 * Seek Modes
 *
 * @{
 */

	/** Seek from beginning of I/O stream. */
#define XMM_SEEK_SET			0
	/** Seek from current position. */
#define XMM_SEEK_CUR			1
	/** Seek from end of I/O stream. */
#define XMM_SEEK_END			2

/** @} */

/**
 * I/O Plugin.
 * This structure / type represents an I/O plugin.
 */

typedef struct xmm_PluginIO_s		XMM_PluginIO;

struct xmm_PluginIO_s
{
    /**
     * Plugin data.
     * Plugin data. Common to all XMM plugin(types)
     */
    XMM_Plugin			sys;
    /**
     * Initialize / Open plugin.
     * If url matches I/O type of plugin, initialize plugin and open stream.
     *
     * \param xmm XMM object
     * \param url stream to open. stream names are of the form type://. If string
     * doesn't contain type:// pattern, a file is assumed.
     * \param mode I/O open mode. One of the #XMM_IO_ definitions.
     *
     * \return Upon successful completion pointer to the newly
     * created plugin is returned. Otherwise NULL.
     */
    XMM_PluginIO *		(*Open)( void *xmm, const char *url, int mode );
    /** xmm_IOClose() is a wrapper for this member */
    int				(*Close)( XMM_PluginIO *io );
    /** xmm_IORead() is a wrapper for this member */
    int				(*Read)( XMM_PluginIO *io, void *ptr, int size, int nmemb );
    /** xmm_IOWrite() is a wrapper for this member */
    int				(*Write)( XMM_PluginIO *io, void *ptr, int size, int nmemb );
    /** xmm_IOTell() is a wrapper for this member */
    long			(*Tell)( XMM_PluginIO *io );
    /** xmm_IOSeek() is a wrapper for this member */
    int				(*Seek)( XMM_PluginIO *io, long offset, int whence );
    /** xmm_IOEof() is a wrapper for this member */
    int				(*Eof)( XMM_PluginIO *io );
    /** xmm_IOSize() is a wrapper for this member */
    long			(*Size)( XMM_PluginIO *io );
};

/*
 * Prototypes
 */


	/**
	 * Open stream.
	 * Try to open stream. Find plugin, initialize it and open stream.
	 *
	 * \param xmm XMM object
	 * \param url stream to open. stream names are of the form type://.
	 * If string doesn't contain type:// pattern, a file is assumed.
	 * \param mode I/O open mode. One of the #XMM_IO_ definitions.
	 *
	 * \return Upon successful completion pointer to the newly
	 * created plugin is returned. Otherwise NULL.
	 *
	 * \note Before calling this, the xmm library has to be initialized.
	 */
XMM_PluginIO	*xmm_IOOpen( void *xmm, const char *url, int mode );

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/**
	 * Close I/O.
	 * Close I/O file, connection, ... and free all resources used by plugin.
	 *
	 * \param io Pointer to an I/O plugin.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value )
	 */
int	xmm_IOClose( XMM_PluginIO *io );

	/**
	 * Read data.
	 * Read data ( only possible if I/O is opened for reading ).
	 *
	 * \param io Pointer to an I/O plugin.
	 * \param ptr Pointer to data
	 * \param size size of each element
	 * \param nelem number of elements
	 *
	 * \return Upon successful completion the number of read elements is
	 * returned ( NOT the number of read bytes ).
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_IORead( XMM_PluginIO *io, void *ptr, int size, int nelem );

	/**
	 * Write data.
	 * Write data ( only possible if I/O is opened for writing ).
	 *
	 * \param io Pointer to an I/O plugin.
	 * \param ptr Pointer to data
	 * \param size size of each element
	 * \param nelem number of elements
	 *
	 * \return Upon successful completion the number of written elements is
	 * returned ( NOT the number of read bytes ).
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_IOWrite( XMM_PluginIO *io, void *ptr, int size, int nelem );

	/**
	 * Get I/O stream position.
	 * Obtains the current position of the I/O stream.
	 *
	 * \param io Pointer to an I/O plugin.
	 *
	 * \return Upon successful completion the current I/O stream position
	 * is returned. Otherwise an error code ( negative value ).
	 */
long	xmm_IOTell( XMM_PluginIO *io );

	/**
	 * Seek in I/O stream.
	 * Set a new I/O stream position.
	 *
	 * \param io Pointer to an I/O plugin.
	 * \param offset new position
	 * \param mode seek mode. One of the #XMM_SEEK_ definitions.
	 *
	 * \return Upon successful completion XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 */
int	xmm_IOSeek( XMM_PluginIO *io, long offset, int mode );

	/**
	 * Check for end of I/O stream.
	 * Check if end of I/O stream reached.
	 *
	 * \param io Pointer to an I/O plugin.
	 *
	 * \return Upon successful completion 1 is returned if EOF, else if
	 * not EOF 0 is returned. Otherwise an error code ( negative value ).
	 */
int	xmm_IOEof( XMM_PluginIO *io );

	/**
	 * Get I/O stream size.
	 * Gets the size of the I/O stream.
	 *
	 * \param io Pointer to an I/O plugin.
	 *
	 * \return Upon successfull completion the size is returned. Otherwise
	 * an error code ( negative value ). If plugin doesn't support this
	 * feature, this call will return 0.
	 */
long	xmm_IOSize( XMM_PluginIO *io );

#ifdef __cplusplus
}
#endif

#endif
