/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_plugin_h
#define INC_libxmm_plugin_h

/** @file plugin.h
 * Linux MultiMedia Project: plugin management
 */

#include <inttypes.h>
#include <libxmm/util/list.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Types
 */

/**
 * Plugin type.
 * This type contains all data, needed for plugin management.
 */

typedef struct xmm_Plugin_s		XMM_Plugin;

struct xmm_Plugin_s
{
    void			*Handle;	/**< handle: filled by management */
    char			ID[4];		/**< ID: 'XMMP' */
    uint32_t			Type;		/**< Type */
    uint32_t			Flags;		/**< Flags */
    uint32_t			Version;	/**< Version */

    char			Filename[256];	/**< filename: filled by management */
    char			Name[32];	/**< Name */
    char			Desc[96];	/**< Description */
    char			Copy[96];	/**< Copyright string */

    void			*xmm;		/**< XMM Object, Plugin belongs to */
    void			*priv;		/**< Plugin Private data */
};

/*
 * Prototypes
 */

		/**
		 * Register Plugin.
		 * Register plugin with the specified filename.
		 *
		 * \param filename plugin to register
		 *
		 * \return Upon successful completion pointer to the plugin
		 * is returned. Otherwise NULL.
		 */
XMM_Plugin	*xmm_PluginRegister( char *filename );

		/**
		 * Register Plugin from data.
		 * Register plugin from data.
		 *
		 * \param plugin plugin data used to register
		 *
		 * \return Upon successful completion pointer to the plugin
		 * is returned. Otherwise NULL.
		 */
XMM_Plugin	*xmm_PluginRegisterFromData( XMM_Plugin *plugin );

		/**
		 * Register Plugin ( type ).
		 * Register plugin with the specified filename, if type matches.
		 *
		 * \param filename plugin to register
		 * \param type plugin type
		 *
		 * \return Upon successful completion pointer to the plugin
		 * is returned. Otherwise NULL.
		 */
XMM_Plugin	*xmm_PluginRegisterType( char *filename, int type );

		/**
		 * Remove Plugin.
		 * Remove plugin. Free used memory and close DLL.
		 *
		 * \param plugin plugin to free
		 */
void		xmm_PluginRemove( XMM_Plugin *plugin );

		/**
		 * Register all plugins ( type ).
		 * Register all plugins in a specified directory, if type matches.
		 *
		 * \param dirname directory containing plugins
		 * \param type plugin type
		 *
		 * \return Upon successful completion pointer to a list
		 * containing registerd plugins is returned. Otherwise NULL.
		 */
XMM_List	*xmm_PluginRegisterDir( char *dirname, int type );

		/**
		 * Remove all plugins.
		 * Remove all plugins in a list.
		 *
		 * \param list list containing plugin
		 */
void		xmm_PluginRemoveList( XMM_List *list );

#ifdef __cplusplus
}
#endif

#endif
