/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_util_buffer_h
#define INC_libxmm_util_buffer_h

/** @file buffer.h 
 * Linux MultiMedia Project: Buffer Queue
 */

#include <inttypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Types
 */

/*
 * Buffer
 */

typedef struct xmm_buffer_s	XMM_Buffer;

struct xmm_buffer_s
{
    unsigned char	*data;
    uint32_t		seq;

    uint32_t		size;
    uint32_t		pos;
    XMM_Buffer		*next;
    XMM_Buffer		*prev;
};

/**
 * Buffer Queue.
 * Buffer Queue data type. This object represents a buffer queue.
 */

typedef struct xmm_BufferQueue_s
{
    XMM_Buffer		*head, *tail;
    uint32_t		size;
} XMM_BufferQueue;

/*
 * Prototypes
 */

	/**
	 * Initialize buffer queue.
	 * Used to initialize a buffer queue. This function always has
	 * to be called, before any other buffer queue function.
	 *
	 * \param bq buffer queue
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmBQ_Init( XMM_BufferQueue *bq );

	/**
	 * Free buffer queue.
	 * This will clear the buffer queue and free allocated memory.
	 * The buffer queue will be in the same state, as after
	 * the xmmBQ_Init() call
	 *
	 * \param bq buffer queue
	 */
void	xmmBQ_Free( XMM_BufferQueue *bq );

	/**
	 * Get queue size.
	 * Get total size of data in buffer queue
	 *
	 * \param bq buffer queue
	 * \return size of all buffers in queue
	 */
int	xmmBQ_Size( XMM_BufferQueue *bq );

	/**
	 * Add data to buffer queue.
	 * First memory will be allocated. Then data will be copied to the
	 * newly allocated memory and the buffer will be added to the queue.
	 *
	 * \param bq buffer queue
	 * \param data data to add. If data == NULL, no data will be
	 * copied to the newly allocated memory
	 * \param size size of the data
	 *
	 * \return pointer of newly created memory area. This can be used
	 * to write data to the new buffer ( if the function was
	 * called with NULL for data )
	 */
void	*xmmBQ_Add( XMM_BufferQueue *bq, void *data, int size );

	/**
	 * Add data to buffer queue ( with sequence number ).
	 * First memory will be allocated. Then data will be copied to the
	 * newly allocated memory and the buffer will be added to the queue.
	 *
	 * \param bq buffer queue
	 * \param data data to add. If data == NULL, no data will be
	 * copied to the newly allocated memory
	 * \param size size of the data
	 * \param seq sequence number for buffer ( e.g. timestamp )
	 *
	 * \return pointer of newly created memory area. This can be used
	 * to write data to the new buffer ( if the function was
	 * called with NULL for data )
	 */
void	*xmmBQ_AddSeq( XMM_BufferQueue *bq, void *data, int size, uint32_t seq );

	/**
	 * Concatenate two buffer queues.
	 * The buffer queue bqadd will be appended to bq.
	 *
	 * \param bq buffer queue
	 * \param bqadd buffer queue to add
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 *
	 * \note bqadd will be emptry after calling this function
	 */
int	xmmBQ_Cat( XMM_BufferQueue *bq, XMM_BufferQueue *bqadd );

	/**
	 * Read data from buffer queue.
	 *
	 * \param bq buffer queue
	 * \param data data will be copied there. If data == NULL
	 * the data will be removed from queue ( and lost )
	 * \param size bytes to remove from buffer queue
	 *
	 * \return number of bytes successfully read
	 */
int	xmmBQ_Read( XMM_BufferQueue *bq, void *data, int size );

	/**
	 * Read data from buffer queue ( with sequence number )
	 *
	 * \param bq buffer queue
	 * \param data data will be copied there. If data == NULL
	 * the data will be removed from queue ( and lost )
	 * \param size bytes to remove from buffer queue
	 * \param seq Address of an uint32_t for storing the sequence number
	 *
	 * \return number of bytes successfully read
	 */
int	xmmBQ_ReadSeq( XMM_BufferQueue *bq, void *data, int size, uint32_t *seq );

	/**
	 * Read first buffer in queue.
	 *
	 * \param bq buffer queue
	 * \param data data will be copied there. If data == NULL
	 * the data will be removed from queue ( and lost )
	 * \param maxsize maximum amount of data to read. This should
	 * be the size of the memory are pointed to by data.
	 * If it is smaller then the buffer size, the remaining bytes
	 * will be lost.
	 *
	 * \return number of bytes successfully read
	 */
int	xmmBQ_HeadRead( XMM_BufferQueue *bq, void *data, int maxsize );

	/**
	 * Read first buffer in queue ( with sequence number ).
	 *
	 * \param bq buffer queue
	 * \param data data will be copied there. If data == NULL
	 * the data will be removed from queue ( and lost )
	 * \param maxsize maximum amount of data to read. This should
	 * be the size of the memory are pointed to by data.
	 * If it is smaller then the buffer size, the remaining bytes
	 * will be lost.
	 * \param seq Address of an uint32_t for storing the sequence number
	 *
	 * \return number of bytes successfully read
	 */
int	xmmBQ_HeadReadSeq( XMM_BufferQueue *bq, void *data, int maxsize, uint32_t *seq );

	/**
	 * Get size of first buffer.
	 *
	 * \param bq buffer queue
	 *
	 * \return size of the first buffer in queue
	 */
int	xmmBQ_HeadSize( XMM_BufferQueue *bq );

	/**
	 * Get first buffer.
	 * This function is used to get the first buffer in the queue.
	 * This way, data from the queue can be used without the need
	 * to copy the data into another memory area ( faster ).
	 *
	 * \param bq buffer queue
	 *
	 * \return pointer of the first buffer in the queue
	 *
	 * \note The buffer has the size returned by xmmBQ_HeadSize()
	 * \note The buffer has to be removed with xmmBQ_HeadRemove().
	 * If not the data will reamain in the buffer queue.
	 */
void	*xmmBQ_HeadData( XMM_BufferQueue *bq );

	/**
	 * Remove first buffer.
	 * This function call removes the first buffer in the queue.
	 *
	 * \param bq buffer queue
	 */
void	xmmBQ_HeadRemove( XMM_BufferQueue *bq );

	/**
	 * Set size of last buffer.
	 * This function call sets the size of the last buffer in
	 * the queue. It maybe necessary if you have less data to
	 * add, then allocated by xmmBQ_Add() with NULL for data
	 *
	 * \param bq buffer queue
	 * \param size new size of last buffer.
	 * The new size must NOT be bigger then the original value.
	 *
	 * \return Upon successful completion #XMM_RET_OK is returned.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmBQ_TailResize( XMM_BufferQueue *bq, int size );

#ifdef __cplusplus
}
#endif

#endif
