/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_libxmm_util_list_h
#define INC_libxmm_util_list_h

/** @file list.h
 * Linux MultiMedia Project: support for lists
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Types
 */

/**
 * List type.
 * This struct is a list node and the list itself.
 */

typedef struct XMM_List_s XMM_List;

struct XMM_List_s
{
    /** Pointer to data */
    void			*data;
    /** Pointer to next node, NULL if last */
    XMM_List			*next;
    /** Pointer to previous node, NULL if first */
    XMM_List			*prev;
};

/*
 * Prototypes
 */

		/**
		 * Free list.
		 * Remove all items from list
		 *
		 * \param list pointer to a list
		 *
		 * \note after this call list is undefined, should be set
		 * to NULL, before using it again
		 */
void		xmmList_Free( XMM_List *list );

		/**
		 * Concat lists.
		 * Concat two lists.
		 *
		 * \param list pointer to a list
		 * \param list2 pointer to a seconds lists, that will be
		 * appended to the first.
		 *
		 * \return Pointer to the new (maybe old) first node of the
		 * list will be returned. ( NULL means empty list )
		 */
XMM_List	*xmmList_Concat( XMM_List *list, XMM_List *list2 );

		/**
		 * Reverse list.
		 * Reverse the complete list.
		 *
		 * \param list pointer to a list
		 *
		 * \return Pointer to the new (maybe old) first node of the
		 * list will be returned. ( NULL means empty list )
		 */
XMM_List	*xmmList_Reverse( XMM_List *list );

		/**
		 * Length of the list.
		 * Calculate the length of a list.
		 *
		 * \param list pointer to a list
		 *
		 * \return Length of the list. ( zero means empty list )
		 */
int		xmmList_Length( XMM_List *list );

		/**
		 * Get node (position).
		 * Get node at specified position.
		 *
		 * \param list pointer to a list
		 * \param pos position in list ( starting at zero )
		 *
		 * \return Upon successful completion a pointer to the node
		 * is returned. If no node found, NULL will be returned.
		 */
XMM_List	*xmmList_GetNodePos( XMM_List *list, int pos );

		/**
		 * Get node (data).
		 * Get node with key 'data'.
		 *
		 * \param list pointer to a list
		 * \param data key to be searched for
		 *
		 * \return Upon successful completion a pointer to the node
		 * is returned. If no node found, NULL will be returned.
		 */
XMM_List	*xmmList_GetNodeData( XMM_List *list, void *data );

		/**
		 * Append node.
		 * Append node to a list.
		 *
		 * \param list pointer to a list
		 * \param data pointer to node data. will be used as key.
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * Otherwise NULL.
		 */
XMM_List	*xmmList_Append( XMM_List *list, void *data );

		/**
		 * Prepend node.
		 * Prepend node to a list.
		 *
		 * \param list pointer to a list
		 * \param data pointer to node data. will be used as key.
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * Otherwise NULL.
		 */
XMM_List	*xmmList_Prepend( XMM_List *list, void *data );

		/**
		 * Insert node.
		 * Insert node into a list.
		 *
		 * \param list pointer to a list
		 * \param data pointer to node data. will be used as key.
		 * \param pos position in list
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * Otherwise NULL.
		 */
XMM_List	*xmmList_Insert( XMM_List *list, void *data, int pos );

		/**
		 * Remove node.
		 * Remove node from list.
		 *
		 * \param list pointer to a list.
		 * \param rem node to remove
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * ( NULL = list empty ).
		 */
XMM_List	*xmmList_RemoveNode( XMM_List *list, XMM_List *rem );

		/**
		 * Remove node ( data ).
		 * Remove node with key 'data' from list.
		 *
		 * \param list pointer to a list.
		 * \param data pointer to node data.
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * ( NULL = key not found or list empty ).
		 */
XMM_List	*xmmList_RemoveNodePos( XMM_List *list, int pos );

		/**
		 * Remove node ( data ).
		 * Remove node with key 'data' from list.
		 *
		 * \param list pointer to a list.
		 * \param data pointer to node data.
		 *
		 * \return Upon successful completion pointer to the new
		 * (maybe old) first node of the list will be returned.
		 * ( NULL = key not found or list empty ).
		 */
XMM_List	*xmmList_RemoveNodeData( XMM_List *list, void *data );


#ifdef __cplusplus
}
#endif

#endif
