/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef INC_xmmplay_h
#define INC_xmmplay_h

/** @file xmmplay.h 
  * Player API of the LinuX MultiMedia Project
  *
  * libxmmplay is a simple player API for the LinuX MultiMedia Project
  * It can be used by applications for playing media files. The playing
  * is done in the background and the application can control the
  * playing with functions like Open(), Close(), Play(), Stop(), etc.
  */

#include <libxmm/libxmm.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Time calculations
 */

/** Get hours from a time value. */
#define	XMMPLAY_TIME_H( sec )	(int)( sec / 3600 )
/** Get minutes from time value. */
#define	XMMPLAY_TIME_M( sec )	(int)(( sec - XMMPLAY_TIME_H( sec ) * 3600 ) / 60 )
/** Get seconds from time value. */
#define	XMMPLAY_TIME_S( sec )	(int)( sec - XMMPLAY_TIME_H( sec ) * 3600 - XMMPLAY_TIME_M( sec ) * 60 )
/** Get 1/10 seconds from a time value. */
#define XMMPLAY_TIME_T( sec )	(int)( sec - XMMPLAY_TIME_H( sec ) * 3600 - XMMPLAY_TIME_M( sec ) * 60 - XMMPLAY_TIME_S( sec ))

/** Get hours from a pts value. */
#define	XMMPLAY_PTS_H( pts )	(int)(( pts / 3600 ) / 1000 )
/** Get minutes from a pts value. */
#define	XMMPLAY_PTS_M( pts )	(int)((( pts - XMMPLAY_PTS_H( pts ) * 3600 * 1000 ) / 60 ) / 1000 )
/** Get seconds from a pts value. */
#define	XMMPLAY_PTS_S( pts )	(int)(( pts - XMMPLAY_PTS_H( pts ) * 3600 * 1000 - XMMPLAY_PTS_M( pts ) * 60 * 1000 ) / 1000 )
/** Get 1/10 seconds from a pts value. */
#define XMMPLAY_PTS_T( pts )	(int)(( pts - XMMPLAY_PTS_H( pts ) * 3600 * 1000 - XMMPLAY_PTS_M( pts ) * 60 * 1000 - XMMPLAY_PTS_S( pts ) * 1000 ) / 1000 )

/** @defgroup XMMPLAY_OFLAG_ Open flags
 *
 * Open flags
 *
 * @{
 */

	/** Directe stream copy. Copy audio stream directly to sound output */
#define	XMMPLAY_OFLAG_AUDIO_DSC		0x00010000
	/** Directe stream copy. Copy video stream directly to graph output */
#define	XMMPLAY_OFLAG_VIDEO_DSC		0x00100000

	/** Don't play audio. Don't play audio data. */
#define	XMMPLAY_OFLAG_AUDIO_OFF		0x00020000
	/** Don't play video. Don't play video data. */
#define	XMMPLAY_OFLAG_VIDEO_OFF		0x00200000

/** @} */

/*
 * Type cast
 */

	/**
	 * Get XMM object from a XMM_Play object.
	 *
	 * \param xmmplay xmmplay object
	 */
#define	XMMPLAY_2XMM( xmmplay )	((void *) *((uint32_t *)xmmplay))

/*
 * Types
 */

/**
 * libxmmplay object.
 * For each opened media file one object is needed. It contains
 * all information needed by the library to play a media file.
 * \note Components of this structure should never be accessed directly.
 */
struct XMM_Play;
typedef	struct XMM_Play	XMM_Play;

/*
 * Prototypes
 */

	/**
	 * Initialize libxmmplay.
	 * Used to initialize libxmmplay. This function always has
	 * to be called, before any other libxmmplay function.
	 *
	 * \return Upon successful completion XMM_Play object is returned.
	 * Otherwise NULL is returned.
	 */
XMM_Play *xmmplay_Init( void );

	/**
	 * Free all resources.
	 * Calling this function will destroy the libxmmplay object.
	 * Allocated memory will be freed.
	 *
	 * \param xmmplay libxmmplay object
	 */
void	xmmplay_Exit( XMM_Play *xmmplay );

	/**
	 * Set audio output format.
	 * Set an audio output format. This will be used as chain output.
	 *
	 * \param xmmplay libxmmplay object
	 * \param fai audio filter information
	 * \param afidx audio format index
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_SetAudioFormat( XMM_Play *xmmplay, XMM_FilterAudioInfo *fai, int afidx );

	/**
	 * Open file.
	 * Use this function to open a file. Only one file can
	 * be opened for each libxmmplay object.
	 * \note Before opening a new file, the old should be closed using xmmplay_Close()
	 *
	 * \param xmmplay libxmmplay object
	 * \param filename file to open
	 * \param flags (TODO)
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Open( XMM_Play *xmmplay, char *filename, int flags );

	/**
	 * Close file.
	 * Close an opened file. It will free all file related resources.
	 *
	 * \param xmmplay libxmmplay object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Close( XMM_Play *xmmplay );

	/**
	 * Start the playback.
	 *
	 * \param xmmplay libxmmplay object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Play( XMM_Play *xmmplay );

	/**
	 * Pause the playback.
	 *
	 * \param xmmplay libxmmplay object
	 *
	 * \return Upon successful completion the pause status is returned.
	 * ( 1 if playback is paused, otherwise 0 )
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Pause( XMM_Play *xmmplay );

	/**
	 * Stop the playback.
	 *
	 * \param xmmplay libxmmplay object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Stop( XMM_Play *xmmplay );

	/**
	 * Get playback status.
	 *
	 * \param xmmplay libxmmplay object
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Status( XMM_Play *xmmplay );

	/**
	 * Seek in file.
	 *
	 * \param xmmplay libxmmplay object
	 * \param percent position in percent of total file time.
	 * \note Not all files are seekable
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Seek( XMM_Play *xmmplay, double percent );

	/**
	 * Scale the video output.
	 *
	 * \param xmmplay libxmmplay object
	 * \param width output width
	 * \param height output height
	 * \param flags (TODO)
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_Scale( XMM_Play *xmmplay, int width, int height, int flags );

	/**
	 * Get Info.
	 * Get detailed information about video, audio
	 *
	 * \param xmmplay libxmmplay object
	 * \param xmm_info file information will be stored in this structure
	 * \param ci clip information will be stored in this structure
	 * \param avdiff initial delay between audio and video
	 * \param percent current file position in percent of total time
	 *
	 * \return current time
	 */
uint32_t xmmplay_Info( XMM_Play *xmmplay, XMM_FileInfo *xmm_info, XMM_ClipInfo *ci, double *percent );

	/**
	 * Set Volume.
	 * Sets the volume. This can be done using the Sound Output
	 * Plugin or using the Input plugin.
	 *
	 * \param xmmplay libxmmplay object
	 * \param left volume of left speaker
	 * \param right volume of right speaker
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_SetVolume( XMM_Play *xmmplay, int left, int right );

	/**
	 * Set Volume.
	 * Sets the volume. This can be done using the Sound Output
	 * Plugin or using the Input plugin.
	 *
	 * \param xmmplay libxmmplay object
	 * \param left volume of left speaker
	 * \param right volume of right speaker
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_GetVolume( XMM_Play *xmmplay, int *left, int *right );

	/**
	 * Display an input plugin dialog.
	 * This will create the input plugin dialogues.
	 *
	 * \param xmmplay libxmmplay object
	 * \param dialog dialogue to display
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_InputDialogue( XMM_Play *xmmplay, int dialog );

	/**
	 * Set input media information.
	 * This call will set ( or remove ) the media information.
	 *
	 * \param xmmplay libxmmplay object
	 * \param ci Address of a #XMM_ClipInfo type.
	 *
	 * \return Upon successful completion XMM_RET_OK is retured.
	 * Otherwise an error code ( negative value ).
	 */
int	xmmplay_InputSetClipInfo( XMM_Play *xmmplay, XMM_ClipInfo *ci );

#ifdef __cplusplus
}
#endif

#endif
