/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * achain.c
 * Audio Filter Chain ( manage and provide internal plugins for conversion )
 */

#include <stdio.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/xmmctl.h"
#include "libxmm/achain.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/mutex.h"
#include "libxmm/util/list.h"
#include "../xmmpriv.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Types
 */

struct priv_audio_filter_s
{
    XMM_Mutex			*mutex;

    /* internal convesion plugins */
    XMM_PluginFilterAudio	*cAF_cfmt;
    XMM_PluginFilterAudio	*cAF_ccn;
    XMM_PluginFilterAudio	*cAF_csr;

    /* codec (decode/encode) */
    XMM_FilterAChainNode	fdec;		/* decode filter [codec (i)] */
    XMM_FilterAChainNode	fenc;		/* encode filter [codec (o)] */

    /* filter list */
    XMM_List			*cAFlist;
    XMM_List			*pAFlist;
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info_cfmt;
extern XMM_PluginFilterAudio	plugin_info_ccn;
extern XMM_PluginFilterAudio	plugin_info_csr;


/*
 * Initialize Filter chain/subsystem
 */
int xmm_FilterAChainInit( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  if( xmm->pFACpriv != NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Audio filter chain already initialized" );
	return XMM_RET_ERROR;
  }

  /* Allocate private data */
  if(( xmm->pFACpriv = malloc( sizeof( struct priv_audio_filter_s ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to allocate memory for audio filter chain" );
	return XMM_RET_ERROR;
  }
  memset( xmm->pFACpriv, 0, sizeof( struct priv_audio_filter_s ));

  priv = xmm->pFACpriv;

  /* Load internal plugins */
  priv->cAF_cfmt = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_cfmt );
  if( priv->cAF_cfmt == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  priv->cAF_ccn = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_ccn );
  if( priv->cAF_ccn == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  priv->cAF_csr = (XMM_PluginFilterAudio *) xmm_PluginRegisterFromData((XMM_Plugin *) &plugin_info_csr );
  if( priv->cAF_csr == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to register internal plugin" );
	return XMM_RET_ERROR;
  }

  /* Create mutex */
  priv->mutex = xmmMutex_Create();

  return XMM_RET_OK;
}

/*
 * Free Filter chain/subsystem
 */
int xmm_FilterAChainExit( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterAChainStop( xmm );

  /* Free internal plugins */
  if( priv->cAF_cfmt )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_cfmt );
  if( priv->cAF_ccn )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_ccn );
  if( priv->cAF_csr )		xmm_PluginRemove( (XMM_Plugin *)priv->cAF_csr );

  /* Destroy mutex */
  xmmMutex_Destroy( priv->mutex );

  /* Free private chain data */
  free( xmm->pFACpriv );
  xmm->pFACpriv = NULL;

  return XMM_RET_OK;
}

/*
 * Lock audio filter chain
 */
int xmm_FilterAChainLock( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Lock( priv->mutex );
}

/*
 * Unlock audio filter chain
 */
int xmm_FilterAChainUnlock( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  return xmmMutex_Unlock( priv->mutex );
}

/*
 * Check if the audio filter chain is active
 */
int xmm_FilterAChainActive( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Check for codecs */
  if( priv->fdec.filter )	return 1;
  if( priv->fenc.filter )	return 1;

  /* Check for filters */
  if( priv->pAFlist != NULL )	return 1;

  return 0;
}

/*
 * Prepare audio filter chain
 */
int xmm_FilterAChainPrepare( void *_xmm, XMM_AudioFormat *iaf, XMM_FilterAudioInfo *fai, int fmtidx, XMM_AudioFormat **coaf )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  int				i;
  XMM_AudioFormat		*atmp;
  uint32_t			*pcm_match, pcm_match_min, pcm_match_min_idx;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Check for autoselection support */
  if(( fmtidx == -1 ) && (( fai->caps & XMM_FILTER_ACF_AUTOSELECT ) == 0 ))
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Output Filter does NOT support autoselection, ( fmtidx = %i )", fmtidx );
	return XMM_RET_ERROR;
  }

  /* Stop the unstopped chain ( only to be sure, if plugin doesn't stop it ) */
  xmm_FilterAChainStop( xmm );

  /* Save chain input format */
  memcpy( &priv->fdec.saf, iaf, sizeof(XMM_AudioFormat));
  priv->fdec.saf.extraSize = 0;
  priv->fdec.saf.extraType = 0;

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() %i formats\n", fai->nfmt );
  for( i = 0; i < fai->nfmt; i++ )
	xmm_logging( 1, "\t format %i: 0x%x\n", i, fai->fmt[i].format );
#endif

  /* Only if not filters active, and input is non-PCM format */
  if(( priv->cAFlist == NULL ) &&
	(( iaf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
  {

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() Trying direct stream copy\n" );
#endif

	/*
	 * Direct stream copy ???
	 */
	for( i = 0; i < fai->nfmt; i++ )
	{
	    atmp = &fai->fmt[i];

	    if(	( atmp->format == iaf->format ) &&
		(( atmp->samprate == iaf->samprate ) || ( atmp->samprate == XMM_AUDIO_ANY_SRATE )) &&
		(( atmp->channels == iaf->channels ) || ( atmp->channels == XMM_AUDIO_ANY_CHANN )) &&
		(( atmp->bitrate == iaf->bitrate ) || ( atmp->bitrate == XMM_AUDIO_ANY_BRATE )))
	    {
		/* Set saf and daf to same format, to avoid PCM conversion */
		memcpy( &priv->fdec.daf, &fai->fmt[i], sizeof( XMM_AudioFormat ));
		priv->fdec.daf.extraSize = 0;
		priv->fdec.daf.extraType = 0;

		memcpy( &priv->fenc.saf, &priv->fdec.daf, sizeof( XMM_AudioFormat ));

		*coaf = iaf;		/* chain output audio format = input */

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() direct stream copy ( format 0x%x )\n", iaf->format );
#endif

		return XMM_RET_OK;
	    }
	}

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() Trying direct (codec) conversion\n" );
#endif

	/* 
	 * Direct (codec) conversion ???
	 */
	for( i = 0; i < fai->nfmt; i++ )
	{
	    priv->fdec.filter = xmm_FilterAudioFind( xmm, iaf, &fai->fmt[i], XMM_FILTER_AOF_MEMBERS );
	    if( priv->fdec.filter != NULL )
	    {
		/* Set saf and daf to same format, to avoid PCM conversion */
		memcpy( &priv->fdec.daf, &fai->fmt[i], sizeof( XMM_AudioFormat ));
		priv->fdec.daf.extraSize = 0;
		priv->fdec.daf.extraType = 0;

		memcpy( &priv->fenc.saf, &priv->fdec.daf, sizeof( XMM_AudioFormat ));

		/* Get codec filter output format ( with extra data ) */
		xmm_FilterAudioControl( priv->fdec.filter, XMM_CTLGET_AFORMAT_PTR, 0, coaf );

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() direct decoding ( format 0x%x -> 0x%x)\n", iaf->format, (*coaf)->format );
#endif

		return XMM_RET_OK;
	    }
	}
  }

  /* 
   * Find decode filter codec ( we need PCM )
   */
  if(( iaf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	priv->fdec.daf.format = XMM_AUDIO_CODEC_PCM;	/* Decode to PCM */
#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() Conversion, format 0x%x -> PCM forced\n", iaf->format );
#endif

	/* Initialize audio codec */
	priv->fdec.filter = xmm_FilterAudioFind( xmm, iaf, &priv->fdec.daf, 0 );
	if( priv->fdec.filter == NULL )	return XMM_RET_ERROR;

	/* Get codec filter output format ( with extra data ) */
	xmm_FilterAudioControl( priv->fdec.filter, XMM_CTLGET_AFORMAT_PTR, 0, coaf );

	iaf = *coaf;		/* coaf is our new input audio format */
  }
  else
  {
	/* Set input of PCM part of the chain */
	memcpy( &priv->fdec.daf, iaf, sizeof(XMM_AudioFormat));
	priv->fdec.daf.extraSize = 0;
	priv->fdec.daf.extraType = 0;
  }

  /*
   * Now, we have PCM data
   */

  /* Set output of PCM part of the chain */
  memcpy( &priv->fenc.saf, &priv->fdec.daf, sizeof(XMM_AudioFormat));
  priv->fenc.saf.extraSize = 0;
  priv->fenc.saf.extraType = 0;

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() PCM data ( format: 0x%x, channel(s): %i samprate: %i )\n", iaf->format, iaf->channels, iaf->samprate );
#endif
  /*
   * Next we have to find an output format
   */

  if(( pcm_match = malloc( fai->nfmt * sizeof( uint32_t ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to allocate pcm_match data" );

  pcm_match_min = 0;
  pcm_match_min_idx = -1;

  /*
   * Find (closest) supported PCM format
   */
  for( i = 0; i < fai->nfmt; i++ )
  {
	atmp = &fai->fmt[i];
	pcm_match[i] = 0;

	if(( atmp->format == iaf->format ))	pcm_match[i]++;

	if(( atmp->samprate == iaf->samprate ) ||
		( atmp->samprate == XMM_AUDIO_ANY_SRATE ))	pcm_match[i]++;

	if(( atmp->channels == iaf->channels ) ||
		( atmp->channels == XMM_AUDIO_ANY_CHANN ))	pcm_match[i]++;

	if(( atmp->bitrate == iaf->bitrate ) ||
		( atmp->bitrate == XMM_AUDIO_ANY_BRATE ))	pcm_match[i]++;

	if(( atmp->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
		continue;

	if( pcm_match[i] > pcm_match_min )
	{
	    pcm_match_min = pcm_match[i];
	    pcm_match_min_idx = i;
	}
  }

  if( pcm_match_min_idx != -1 )
  {
	memcpy( &priv->fenc.saf, &fai->fmt[pcm_match_min_idx], sizeof(XMM_AudioFormat));
	priv->fenc.saf.extraSize = 0;
	priv->fenc.saf.extraType = 0;

	if( priv->fenc.saf.samprate == XMM_AUDIO_ANY_SRATE )
		priv->fenc.saf.samprate = iaf->samprate;

	if( priv->fenc.saf.channels == XMM_AUDIO_ANY_CHANN )
		priv->fenc.saf.channels = iaf->channels;

	if( priv->fenc.saf.bitrate == XMM_AUDIO_ANY_BRATE )
		priv->fenc.saf.bitrate = iaf->bitrate;

	*coaf = &priv->fenc.saf;			/* chain output audio format */

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() matching PCM output (%i/4)\n", pcm_match[pcm_match_min_idx] );
#endif

	return XMM_RET_OK;
  }

  /*
   * Find encoder plugin ( perfect match )
   */
  for( i = 0; i < fai->nfmt; i++ )
  {
	priv->fenc.filter = xmm_FilterAudioFind( xmm, iaf, &fai->fmt[i], XMM_FILTER_AOF_MEMBERS );
	if( priv->fenc.filter != NULL )
	{
		/* Get codec filter output format ( with extra data ) */
		xmm_FilterAudioControl( priv->fenc.filter, XMM_CTLGET_AFORMAT_PTR, 0, coaf );

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() direct encoding ( format 0x%x -> 0x%x)\n", iaf->format, (*coaf)->format );
#endif

		return XMM_RET_OK;
	}
  }

  return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() Unable to initialize audio chain for format 0x%x", priv->fdec.saf.format );
}

/*
 * Start audio filter chain
 *
 * Returns: needed buffer size ( of source data )
 */
int xmm_FilterAChainStart( void *_xmm, int blocksize )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  int				nblocksize = blocksize;
  XMM_AudioFormat		saf;
  XMM_AudioFormat		daf;
  XMM_List			*le;
  XMM_FilterAChainNode		*facn;
  XMM_PluginFilterAudio		*fa;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Lock audio chain, filter list may not be changed sound plugin unlocks the chain */
  xmm_FilterAChainLock( xmm );

  /* Save source/dest format */
  memcpy( &saf, &priv->fdec.daf, sizeof( XMM_AudioFormat ));
  memcpy( &daf, &priv->fenc.saf, sizeof( XMM_AudioFormat ));

  /*
   * We need to do this in reverse order, the last conversions output must be
   * blocksize
   */

  /* Process Filter List */
  for( le = priv->cAFlist; le; le = le->next )
  {
	fa = (XMM_PluginFilterAudio *) le->data;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() adding filter '%s'\n", fa->sys.Name );
#endif

	/* Allocate node + conversion buffer */
	if(( facn = malloc( sizeof(XMM_FilterAChainNode) + nblocksize )) == NULL )
	    return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() conversion buffer" );

	facn->data = (void *) &facn[1];
	facn->size = nblocksize;
	memcpy( &facn->saf, &saf, sizeof( XMM_AudioFormat ));
	memcpy( &facn->daf, &daf, sizeof( XMM_AudioFormat ));

	facn->filter = fa->Open( xmm, &facn->saf, &facn->daf, 0 );
	if( facn->filter == NULL )
	{
		xmm_logging( 1, __FUNCTION__ "() WARNING: Unable to open filter '%s'\n", fa->sys.Name );
		free( facn );
		continue;
	}

	/* Get input size */
	facn->filter->Control( facn->filter, XMM_CTLGET_DATA_SSIZE, nblocksize, &nblocksize );

	priv->pAFlist = xmmList_Append( priv->pAFlist, (void *)facn );
  }

  /* Samprate */
  if( priv->fdec.daf.samprate != priv->fenc.saf.samprate )
  {
	/* Allocate node + conversion buffer */
	if(( facn = malloc( sizeof(XMM_FilterAChainNode) + nblocksize )) == NULL )
	    return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() conversion buffer" );

	facn->data = (void *) &facn[1];
	facn->size = nblocksize;
	memcpy( &facn->saf, &saf, sizeof( XMM_AudioFormat ));
	memcpy( &facn->daf, &daf, sizeof( XMM_AudioFormat ));

	facn->saf.channels = daf.channels;
	facn->saf.format = daf.format;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() samplerate: channel(s): %i, format: 0x%x, %i Hz -> %i Hz\n", daf.channels, daf.format, saf.samprate, daf.samprate );
#endif

	facn->filter = priv->cAF_csr->Open( xmm, &facn->saf, &facn->daf, 0 );
	if( facn->filter == NULL )
	{
		free( facn );
		return XMM_RET_ERROR;
	}

	/* Get input size */
	facn->filter->Control( facn->filter, XMM_CTLGET_DATA_SSIZE, nblocksize, &nblocksize );

	daf.samprate = saf.samprate;		/* Samprate converted */
	priv->pAFlist = xmmList_Prepend( priv->pAFlist, (void *)facn );
  }

  /* Channels */
  if( priv->fdec.daf.channels != priv->fenc.saf.channels )
  {
	/* Allocate node + conversion buffer */
	if(( facn = malloc( sizeof(XMM_FilterAChainNode) + nblocksize )) == NULL )
	    return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() conversion buffer" );

	facn->data = (void *) &facn[1];
	facn->size = nblocksize;
	memcpy( &facn->saf, &saf, sizeof( XMM_AudioFormat ));
	memcpy( &facn->daf, &daf, sizeof( XMM_AudioFormat ));

	facn->saf.format = daf.format;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() channels: channel(s): %i -> %i, format: 0x%x, %i Hz\n", saf.channels, daf.channels, daf.format, daf.samprate );
#endif

	facn->filter = priv->cAF_ccn->Open( xmm, &facn->saf, &facn->daf, 0 );
	if( facn->filter == NULL )
	{
		free( facn );
		return XMM_RET_ERROR;
	}

	/* Get input size */
	facn->filter->Control( facn->filter, XMM_CTLGET_DATA_SSIZE, nblocksize, &nblocksize );

	daf.channels = saf.channels;		/* Channels converted */
	priv->pAFlist = xmmList_Prepend( priv->pAFlist, (void *)facn );
  }

  /* Format */
  if( priv->fdec.daf.format != priv->fenc.saf.format )
  {
	/* Allocate node + conversion buffer */
	if(( facn = malloc( sizeof(XMM_FilterAChainNode) + nblocksize )) == NULL )
	    return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() conversion buffer" );

	facn->data = (void *) &facn[1];
	facn->size = nblocksize;
	memcpy( &facn->saf, &saf, sizeof( XMM_AudioFormat ));
	memcpy( &facn->daf, &daf, sizeof( XMM_AudioFormat ));

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() format: channel(s): %i, format: 0x%x -> 0x%x, %i Hz\n", daf.channels, saf.format, daf.format, daf.samprate );
#endif

	facn->filter = priv->cAF_cfmt->Open( xmm, &facn->saf, &facn->daf, 0 );
	if( facn->filter == NULL )
	{
		free( facn );
		return XMM_RET_ERROR;
	}

	/* Get input size */
	facn->filter->Control( facn->filter, XMM_CTLGET_DATA_SSIZE, nblocksize, &nblocksize );

	daf.format = saf.format;		/* Format converted */
	priv->pAFlist = xmmList_Prepend( priv->pAFlist, (void *)facn );
  }

  xmm_logging( 2, __FUNCTION__ "() Input: %i Hz, %i Channels, %i Bits ( Size = %i )\n", priv->fdec.daf.samprate, priv->fdec.daf.channels, priv->fdec.daf.format & XMM_AUDIO_MASK_SIZE, nblocksize );
  xmm_logging( 2, __FUNCTION__ "() Output: %i Hz, %i Channels, %i Bits ( Size = %i )\n", priv->fenc.saf.samprate, priv->fenc.saf.channels, priv->fenc.saf.format & XMM_AUDIO_MASK_SIZE, blocksize );

  /* Decoding */
  if( priv->fdec.filter )
  {
xmm_logging( 1, "achain fdec...\n" );
	/* Allocate conversion buffer */
	if(( priv->fdec.data = malloc( nblocksize )) == NULL )
	    return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() conversion buffer" );

	priv->fdec.size = nblocksize;
//	memcpy( &priv->fdec.saf<old>, &saf, sizeof( XMM_AudioFormat ));
//	memcpy( &priv->fdec.daf<old>, &daf, sizeof( XMM_AudioFormat ));
  }

  return nblocksize;
}

/*
 * Get source size.
 * Get needed amount of input data, for the specified amount of output data
 */
int xmm_FilterAChainSourceSize( void *_xmm, uint32_t dsize, uint32_t *ssize )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  int				ret = 0;
  uint32_t			tmp = dsize;
  XMM_List			*le;
  XMM_FilterAChainNode		*facn;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  *ssize = dsize;			/* Needed, if no filters used */

  /* Encode codec filter */
  if( priv->fenc.filter )
  {

	ret = priv->fenc.filter->Control( priv->fenc.filter, XMM_CTLGET_DATA_SSIZE, tmp, ssize );
#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() need %i bytes for %i output (encode)\n", *ssize, tmp );
#endif

	if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;

	tmp = *ssize;
  }

  /* Process Filter List */
  for( le = priv->pAFlist; le; le = le->next )
  {
	facn = (XMM_FilterAChainNode *) le->data;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() Processing filter '%s'\n", facn->filter->sys.Name );
#endif

	ret = facn->filter->Control( facn->filter, XMM_CTLGET_DATA_SSIZE, tmp, ssize );
#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() need %i bytes for %i output (filter)\n", *ssize, tmp );
#endif

	if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;

	tmp = *ssize;
  }

  /* Decode codec filter */
  if( priv->fdec.filter )
  {
	ret = priv->fdec.filter->Control( priv->fdec.filter, XMM_CTLGET_DATA_SSIZE, *ssize, ssize );
#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() need %i bytes for %i output (codec)\n", *ssize, tmp );
#endif

	if( ret != XMM_CTLRET_ARG )	return XMM_RET_OK;

	tmp = *ssize;
  }

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() %i -> %i\n", *ssize, dsize );
#endif

  return ( ret >= 0 ) ? XMM_RET_ERROR : ret;
}

/*
 * Process filter chain
 */
int xmm_FilterAChain( void *_xmm, uint8_t *sdata, uint32_t ssize, uint8_t **ddata, uint32_t *dsize, uint32_t *flags, uint32_t *samples )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  uint32_t			proc;
  XMM_List			*le;
  XMM_FilterAChainNode		*facn;

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() ssize = %i dsize = %i\n", ssize, *dsize );
#endif

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  if( samples )	*samples = ssize * priv->fdec.saf.samprate * 8 / ABS( priv->fdec.saf.bitrate );

  /* Decoding */
  if( priv->fdec.filter )
  {
	proc = priv->fdec.size;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() need %i bytes output after fdec\n", proc );
#endif

	priv->fdec.filter->Process( priv->fdec.filter, sdata, ssize, priv->fdec.data, &proc, NULL );

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() processed = %i, decoded %i\n", ssize, proc );
#endif

	sdata = priv->fdec.data;
	ssize = proc;
  }

  /* Process Filter List */
  for( le = priv->pAFlist; le; le = le->next )
  {
	facn = (XMM_FilterAChainNode *) le->data;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() Processing filter '%s'\n", facn->filter->sys.Name );
#endif

	proc = facn->size;
	facn->filter->Process( facn->filter, sdata, ssize, facn->data, &proc, 0 );
	sdata = facn->data;
	ssize = proc;
  }

  /* Encoding */
  if( priv->fenc.filter )
  {
	proc = priv->fdec.size;
	priv->fenc.filter->Process( priv->fenc.filter, sdata, ssize, priv->fenc.data, &proc, NULL );

	xmm_logging( 1, ">Processed = %i, encoded %i\n", ssize, proc );

	sdata = priv->fenc.data;
	ssize = proc;
  }

  /* Calculate output size */
  *dsize = ssize;
  *ddata = sdata;

#ifdef DEBUG
  xmm_logging( 1, __FUNCTION__ "() dsize = %i samples = %i\n", *dsize, *samples );
#endif

  return XMM_RET_OK;
}

/*
 * Stop chain ( will stop all stuff started in xmm_FilterAChainOutput()
 */
int xmm_FilterAChainStop( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;
  XMM_FilterAChainNode		*facn;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized" );
	return XMM_RET_ERROR;
  }

  /* Codec (decode) */
  if( priv->fdec.filter )	priv->fdec.filter->Close( priv->fdec.filter );
  priv->fdec.filter = NULL;

  /* Free filters */
  for( le = priv->pAFlist; le; le = le->next )
  {
	facn = (XMM_FilterAChainNode *) le->data;

#ifdef DEBUG
	xmm_logging( 1, __FUNCTION__ "() Processing filter '%s'\n", facn->filter->sys.Name );
#endif

	facn->filter->Close( facn->filter );
	free( facn );

  }

  xmmList_Free( priv->pAFlist );
  priv->pAFlist = NULL;

  /* Codec (encode) */
  if( priv->fenc.filter )	priv->fenc.filter->Close( priv->fenc.filter );
  priv->fenc.filter = NULL;

  /* Unlock audio chain */
  xmm_FilterAChainUnlock( xmm );

  return XMM_RET_OK;
}

/*
 * audio filter chain
 * (non-internal) audio filter chain management.
 *
 * Note: Only possible before chain is locked [ in xmm_FilterAChainStart() ]
 */

/*
 * Add plugin to list
 */
int xmm_FilterAChainAdd( void *_xmm, XMM_PluginFilterAudio *pfa )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( pfa == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() filter plugin == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  /* Add plugin */
  priv->cAFlist = xmmList_Append( priv->cAFlist, (void *)pfa );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Add list to filter list
 */
int xmm_FilterAChainAddList( void *_xmm, XMM_List *_pAudioFilterList )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;
  XMM_List			*le;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  if( _pAudioFilterList == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() filter plugin list == NULL\n" );
	return XMM_RET_ERROR;
  }

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  for( le = _pAudioFilterList; le; le = le->next )
	xmm_FilterAChainAdd( xmm, (XMM_PluginFilterAudio *)le->data );

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}

/*
 * Remove filters from chain
 */
int xmm_FilterAChainFree( void *_xmm )
{
  struct priv_audio_filter_s	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Check arguments */
  if( xmm == NULL )	return XMM_RET_ERROR;

  /* */
  priv = xmm->pFACpriv;
  if( priv == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain not initialized." );
	return XMM_RET_ERROR;
  }

  /* Try access to filter chain */
  if( xmmMutex_TryLock( priv->mutex ) != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Audio Filter chain locked. Unable to add filter." );
	return XMM_RET_ERROR;
  }

  xmmList_Free( priv->cAFlist );
  priv->cAFlist = NULL;

  /* Unlock filter chain */
  xmmMutex_Unlock( priv->mutex );

  return XMM_RET_OK;
}
