/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * achain_cn.c
 * Channel conversion
 */

#include <stdio.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/xmmctl.h"
#include "libxmm/achain.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Channel conversion
 */

typedef void (*channel_t)( void *src, void *dst, int size );

#define DIVIDE_FUNC( name, type ) \
static void divide_##name( void *src, void *dst, int size ) \
{ \
  type *src_ptr = (type *)src; \
  type *dst_ptr = (type *)dst; \
 \
  while( size-- > 0 ) \
  { \
	*dst_ptr++ = *src_ptr; \
	*dst_ptr++ = *src_ptr++; \
  } \
}

DIVIDE_FUNC( 8bit, char )
DIVIDE_FUNC( 16bit, short )

#define MERGE_FUNC( name, type, min, max ) \
static void merge_##name( void *src, void *dst, int size ) \
{ \
  int tmp; \
  type *src_ptr = (type *)src; \
  type *dst_ptr = (type *)dst; \
 \
  while( size-- > 0 ) \
  { \
	tmp = ((int)src_ptr[0] + (int)src_ptr[1] ) / 2; \
 \
	if( tmp < min )	tmp = min; \
	else if( tmp > max )	tmp = max; \
	*dst_ptr++ = tmp; \
	src_ptr += 2; \
  } \
}

MERGE_FUNC( 8bit_signed, signed char, -128, 127 )
MERGE_FUNC( 16bit_signed, signed short, -32768, 32767 )
MERGE_FUNC( 8bit_unsigned, unsigned char, 0, 255 )
MERGE_FUNC( 16bit_unsigned, unsigned short, 0, 65535 )

/*
 * Types
 */

/*
 * Private type
 */

struct priv_ccn_t
{
    channel_t			channel;

    XMM_AudioFormat		saf;
    XMM_AudioFormat		daf;
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info_ccn;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *ccn_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_ccn_t	*priv;
  int			ret = 0;

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check audio format */
  if( saf->format != daf->format )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: source(0x%x) != dest(0x%x)", saf->format, daf->format );
	return (void *)ret;
  }

  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: 0x%x not supported ( only PCM supported )", saf->format );
	return (void *)ret;
  }

  if( saf->samprate != daf->samprate )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() samprate: source(%i) != dest(%i)", saf->samprate, daf->samprate );
	return (void *)ret;
  }

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_ccn, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_ccn_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_ccn_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Initialize data */
  memcpy( &priv->saf, saf, sizeof(XMM_AudioFormat));
  memcpy( &priv->daf, daf, sizeof(XMM_AudioFormat));

#ifdef DEBUG
  xmm_logging( 1, "FA-ccn! saf: 0x%x, %i channel(s), %i Hz\n", saf->format, saf->channels, saf->samprate );
  xmm_logging( 1, "FA-ccn! daf: 0x%x, %i channel(s), %i Hz\n", daf->format, daf->channels, daf->samprate );
#endif

  /* Initialize conversion */
  if(( daf->format & XMM_AUDIO_MASK_SIZE ) ==  16 )
  {
	if(( saf->channels == 2 ) && ( daf->channels == 1 ))	
	{
		if( daf->format & XMM_AUDIO_MASK_SIGNED )	priv->channel = merge_16bit_signed;
		else	priv->channel = merge_16bit_unsigned;
	}
	else	if(( saf->channels == 1 ) && ( daf->channels == 2 ))	priv->channel = divide_16bit;
  }
  else
  {
	if(( saf->channels == 2 ) && ( daf->channels == 1 ))	
	{
		if( daf->format & XMM_AUDIO_MASK_SIGNED )	priv->channel = merge_8bit_signed;
		else	priv->channel = merge_8bit_unsigned;
	}
	else	if(( saf->channels == 1 ) && ( daf->channels == 2 ))	priv->channel = divide_8bit;
  }

  if( priv->channel == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unsupported channel conversion: %i --> %i at %i bit\n", saf->channels, daf->channels, ( daf->format & XMM_AUDIO_MASK_SIZE ));
	free( pAFilter );
	return NULL;
  }

  return pAFilter;
}

/*
 * Free filter
 */
static int ccn_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int ccn_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_ccn_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data ) = param * priv->saf.channels / priv->daf.channels;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(FA-ccn) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(FA-ccn) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int ccn_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_ccn_t	*priv = (struct priv_ccn_t *)filter->sys.priv;
  uint32_t		samples;

  samples = isize / ( priv->saf.channels * (( priv->saf.format & XMM_AUDIO_MASK_SIZE ) / 8 ));

#ifdef DEBUG
  xmm_logging( 1, "FA-ccn! Input: %i bytes ( %i samples )\n", isize, samples );
#endif

  priv->channel( src, dest, samples );

  *osize = isize * priv->daf.channels / priv->saf.channels;

#ifdef DEBUG
  xmm_logging( 1, "FA-ccn! Output: %i bytes ( %i samples )\n", *osize, samples );
#endif

  return XMM_RET_OK;
}

/*
 * Process data (BQ)
 */
static int ccn_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio	plugin_info_ccn = {{ NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-CHANNEL",
				    "libxmm channel number converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    ccn_Open, ccn_Close, ccn_Control,
				    ccn_Process, ccn_ProcessBQ };
