/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * achain_fmt.c
 * Format conversion
 */

#include <stdio.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/xmmctl.h"
#include "libxmm/achain.h"
#include "libxmm/error.h"
#include "libxmm/endian.h"
#include "libxmm/util/utils.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Format conversion ( ALSA: kernel/plugin/linear.c )
 */

typedef void (*convert_t)( void *src, void *dst, size_t size );

#define CONV_FUNC(name, srctype, dsttype, val) \
static void conv_##name(void *src_ptr, void *dst_ptr, size_t size) \
{ \
	srctype *srcp = src_ptr; \
	dsttype *dstp = dst_ptr; \
	while (size--) { \
		srctype src = *srcp++; \
		*dstp++ = val; \
	} \
}

CONV_FUNC(8_sign, u_int8_t, u_int8_t, src ^ 0x80)

CONV_FUNC(8_16, u_int8_t, u_int16_t, (u_int16_t)src << 8)
CONV_FUNC(8_16_end, u_int8_t, u_int16_t, (u_int16_t)src)
CONV_FUNC(8_16_sign, u_int8_t, u_int16_t, (u_int16_t)(src ^ 0x80) << 8)
CONV_FUNC(8_16_sign_end, u_int8_t, u_int16_t, (u_int16_t)src ^ 0x80)

CONV_FUNC(8_24, u_int8_t, u_int32_t, (u_int32_t)src << 16)
CONV_FUNC(8_24_end, u_int8_t, u_int32_t, (u_int32_t)src << 8)
CONV_FUNC(8_24_sign, u_int8_t, u_int32_t, (u_int32_t)(src ^ 0x80) << 16)
CONV_FUNC(8_24_sign_end, u_int8_t, u_int32_t, (u_int32_t)(src ^ 0x80) << 8)

CONV_FUNC(8_32, u_int8_t, u_int32_t, (u_int32_t)src << 24)
CONV_FUNC(8_32_end, u_int8_t, u_int32_t, (u_int32_t)src)
CONV_FUNC(8_32_sign, u_int8_t, u_int32_t, (u_int32_t)(src ^ 0x80) << 24)
CONV_FUNC(8_32_sign_end, u_int8_t, u_int32_t, (u_int32_t)src ^ 0x80)

CONV_FUNC(16_8, u_int16_t, u_int8_t, src >> 8)
CONV_FUNC(16_end_8, u_int16_t, u_int8_t, src)
CONV_FUNC(16_8_sign, u_int16_t, u_int8_t, (src >> 8) ^ 0x80)
CONV_FUNC(16_end_8_sign, u_int16_t, u_int8_t, src ^ 0x80)

CONV_FUNC(16_sign, u_int16_t, u_int16_t, src ^ 0x8000)
CONV_FUNC(16_end, u_int16_t, u_int16_t, XMM_SWAP16(src))
CONV_FUNC(16_end_sign, u_int16_t, u_int16_t, XMM_SWAP16(src) ^ 0x8000)
CONV_FUNC(16_sign_end, u_int16_t, u_int16_t, XMM_SWAP16(src ^ 0x8000))
CONV_FUNC(16_end_sign_end, u_int16_t, u_int16_t, src ^ 0x80)

CONV_FUNC(16_24, u_int16_t, u_int32_t, (u_int32_t)src << 8)
CONV_FUNC(16_24_sign, u_int16_t, u_int32_t, (u_int32_t)(src ^ 0x8000) << 8)
CONV_FUNC(16_24_end, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src) << 8)
CONV_FUNC(16_24_sign_end, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src ^ 0x8000) << 8)
CONV_FUNC(16_end_24, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src) << 8)
CONV_FUNC(16_end_24_sign, u_int16_t, u_int32_t, (u_int32_t)(XMM_SWAP16(src) ^ 0x8000) << 8)
CONV_FUNC(16_end_24_end, u_int16_t, u_int32_t, (u_int32_t)src << 8)
CONV_FUNC(16_end_24_sign_end, u_int16_t, u_int32_t, ((u_int32_t)src ^ 0x80) << 8)

CONV_FUNC(16_32, u_int16_t, u_int32_t, (u_int32_t)src << 16)
CONV_FUNC(16_32_sign, u_int16_t, u_int32_t, (u_int32_t)(src ^ 0x8000) << 16)
CONV_FUNC(16_32_end, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src))
CONV_FUNC(16_32_sign_end, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src ^ 0x8000))
CONV_FUNC(16_end_32, u_int16_t, u_int32_t, (u_int32_t)XMM_SWAP16(src) << 16)
CONV_FUNC(16_end_32_sign, u_int16_t, u_int32_t, (u_int32_t)(XMM_SWAP16(src) ^ 0x8000) << 16)
CONV_FUNC(16_end_32_end, u_int16_t, u_int32_t, (u_int32_t)src)
CONV_FUNC(16_end_32_sign_end, u_int16_t, u_int32_t, (u_int32_t)src ^ 0x80)

CONV_FUNC(24_8, u_int32_t, u_int8_t, src >> 16)
CONV_FUNC(24_end_8, u_int32_t, u_int8_t, src >> 8)
CONV_FUNC(24_8_sign, u_int32_t, u_int8_t, (src >> 16) ^ 0x80)
CONV_FUNC(24_end_8_sign, u_int32_t, u_int8_t, (src >> 8) ^ 0x80)

CONV_FUNC(24_16, u_int32_t, u_int16_t, src >> 8)
CONV_FUNC(24_16_sign, u_int32_t, u_int16_t, (src >> 8) ^ 0x8000)
CONV_FUNC(24_16_end, u_int32_t, u_int16_t, XMM_SWAP32(src >> 8))
CONV_FUNC(24_16_sign_end, u_int32_t, u_int16_t, XMM_SWAP32((src >> 8) ^ 0x8000))
CONV_FUNC(24_end_16, u_int32_t, u_int16_t, XMM_SWAP32(src) >> 8)
CONV_FUNC(24_end_16_sign, u_int32_t, u_int16_t, (XMM_SWAP32(src) >> 8) ^ 0x8000)
CONV_FUNC(24_end_16_end, u_int32_t, u_int16_t, src >> 8)
CONV_FUNC(24_end_16_sign_end, u_int32_t, u_int16_t, (src >> 8) ^ 0x80)

CONV_FUNC(24_sign, u_int32_t, u_int32_t, src ^ 0x800000)
CONV_FUNC(24_end, u_int32_t, u_int32_t, XMM_SWAP32(src))
CONV_FUNC(24_end_sign, u_int32_t, u_int32_t, XMM_SWAP32(src) ^ 0x800000)
CONV_FUNC(24_sign_end, u_int32_t, u_int32_t, XMM_SWAP32(src) ^ 0x80)
CONV_FUNC(24_end_sign_end, u_int32_t, u_int32_t, src ^ 0x80)

CONV_FUNC(24_32, u_int32_t, u_int32_t, src << 8)
CONV_FUNC(24_32_sign, u_int32_t, u_int32_t, (src << 8) ^ 0x80000000)
CONV_FUNC(24_32_end, u_int32_t, u_int32_t, XMM_SWAP32(src << 8))
CONV_FUNC(24_32_sign_end, u_int32_t, u_int32_t, XMM_SWAP32((src << 8) ^ 0x80000000))
CONV_FUNC(24_end_32, u_int32_t, u_int32_t, XMM_SWAP32(src) << 8)
CONV_FUNC(24_end_32_sign, u_int32_t, u_int32_t, (XMM_SWAP32(src) << 8) ^ 0x80000000)
CONV_FUNC(24_end_32_end, u_int32_t, u_int32_t, src >> 8)
CONV_FUNC(24_end_32_sign_end, u_int32_t, u_int32_t, (src >> 8) ^ 0x80)

CONV_FUNC(32_8, u_int32_t, u_int8_t, src >> 24)
CONV_FUNC(32_end_8, u_int32_t, u_int8_t, src)
CONV_FUNC(32_8_sign, u_int32_t, u_int8_t, (src >> 24) ^ 0x80)
CONV_FUNC(32_end_8_sign, u_int32_t, u_int8_t, src ^ 0x80)

CONV_FUNC(32_16, u_int32_t, u_int16_t, src >> 16)
CONV_FUNC(32_16_sign, u_int32_t, u_int16_t, (src >> 16) ^ 0x8000)
CONV_FUNC(32_16_end, u_int32_t, u_int16_t, XMM_SWAP16(src >> 16))
CONV_FUNC(32_16_sign_end, u_int32_t, u_int16_t, XMM_SWAP16((src >> 16) ^ 0x8000))
CONV_FUNC(32_end_16, u_int32_t, u_int16_t, XMM_SWAP16(src))
CONV_FUNC(32_end_16_sign, u_int32_t, u_int16_t, XMM_SWAP16(src) ^ 0x8000)
CONV_FUNC(32_end_16_end, u_int32_t, u_int16_t, src)
CONV_FUNC(32_end_16_sign_end, u_int32_t, u_int16_t, src ^ 0x80)

CONV_FUNC(32_24, u_int32_t, u_int32_t, src >> 8)
CONV_FUNC(32_24_sign, u_int32_t, u_int32_t, (src >> 8) ^ 0x800000)
CONV_FUNC(32_24_end, u_int32_t, u_int32_t, XMM_SWAP32(src >> 8))
CONV_FUNC(32_24_sign_end, u_int32_t, u_int32_t, XMM_SWAP32((src >> 8) ^ 0x800000))
CONV_FUNC(32_end_24, u_int32_t, u_int32_t, XMM_SWAP32(src) >> 8)
CONV_FUNC(32_end_24_sign, u_int32_t, u_int32_t, (XMM_SWAP32(src) >> 8) ^ 0x800000)
CONV_FUNC(32_end_24_end, u_int32_t, u_int32_t, src << 8)
CONV_FUNC(32_end_24_sign_end, u_int32_t, u_int32_t, (src << 8) ^ 0x80)

CONV_FUNC(32_sign, u_int32_t, u_int32_t, src ^ 0x80000000)
CONV_FUNC(32_end, u_int32_t, u_int32_t, XMM_SWAP32(src))
CONV_FUNC(32_end_sign, u_int32_t, u_int32_t, XMM_SWAP32(src) ^ 0x80000000)
CONV_FUNC(32_sign_end, u_int32_t, u_int32_t, XMM_SWAP32(src) ^ 0x80)
CONV_FUNC(32_end_sign_end, u_int32_t, u_int32_t, src ^ 0x80)

/* source size, dest size, source endian, dest endian, sign swap */
static convert_t convert_functions[128] =
{
 	NULL,			/* 8->8: Nothing to do */
	conv_8_sign,		/* 8->8 sign: conv_8_sign */
	NULL,			/* 8->8 dst_end: Nothing to do */
	conv_8_sign,		/* 8->8 dst_end sign: conv_8_sign */
	NULL,			/* 8->8 src_end: Nothing to do */
	conv_8_sign,		/* 8->8 src_end sign: conv_8_sign */
	NULL,			/* 8->8 src_end dst_end: Nothing to do */
	conv_8_sign,		/* 8->8 src_end dst_end sign: conv_8_sign */
	conv_8_16,		/* 8->16: conv_8_16 */
	conv_8_16_sign,		/* 8->16 sign: conv_8_16_sign */
	conv_8_16_end,		/* 8->16 dst_end: conv_8_16_end */
	conv_8_16_sign_end,	/* 8->16 dst_end sign: conv_8_16_sign_end */
	conv_8_16,		/* 8->16 src_end: conv_8_16 */
	conv_8_16_sign,		/* 8->16 src_end sign: conv_8_16_sign */
	conv_8_16_end,		/* 8->16 src_end dst_end: conv_8_16_end */
	conv_8_16_sign_end,	/* 8->16 src_end dst_end sign: conv_8_16_sign_end */
	conv_8_24,		/* 8->24: conv_8_24 */
	conv_8_24_sign,		/* 8->24 sign: conv_8_24_sign */
	conv_8_24_end,		/* 8->24 dst_end: conv_8_24_end */
	conv_8_24_sign_end,	/* 8->24 dst_end sign: conv_8_24_sign_end */
	conv_8_24,		/* 8->24 src_end: conv_8_24 */
	conv_8_24_sign,		/* 8->24 src_end sign: conv_8_24_sign */
	conv_8_24_end,		/* 8->24 src_end dst_end: conv_8_24_end */
	conv_8_24_sign_end,	/* 8->24 src_end dst_end sign: conv_8_24_sign_end */
	conv_8_32,		/* 8->32: conv_8_32 */
	conv_8_32_sign,		/* 8->32 sign: conv_8_32_sign */
	conv_8_32_end,		/* 8->32 dst_end: conv_8_32_end */
	conv_8_32_sign_end,	/* 8->32 dst_end sign: conv_8_32_sign_end */
	conv_8_32,		/* 8->32 src_end: conv_8_32 */
	conv_8_32_sign,		/* 8->32 src_end sign: conv_8_32_sign */
	conv_8_32_end,		/* 8->32 src_end dst_end: conv_8_32_end */
	conv_8_32_sign_end,	/* 8->32 src_end dst_end sign: conv_8_32_sign_end */
	conv_16_8,		/* 16->8: conv_16_8 */
	conv_16_8_sign,		/* 16->8 sign: conv_16_8_sign */
	conv_16_8,		/* 16->8 dst_end: conv_16_8 */
	conv_16_8_sign,		/* 16->8 dst_end sign: conv_16_8_sign */
	conv_16_end_8,		/* 16->8 src_end: conv_16_end_8 */
	conv_16_end_8_sign,	/* 16->8 src_end sign: conv_16_end_8_sign */
	conv_16_end_8,		/* 16->8 src_end dst_end: conv_16_end_8 */
	conv_16_end_8_sign,	/* 16->8 src_end dst_end sign: conv_16_end_8_sign */
	NULL,			/* 16->16: Nothing to do */
	conv_16_sign,		/* 16->16 sign: conv_16_sign */
	conv_16_end,		/* 16->16 dst_end: conv_16_end */
	conv_16_sign_end,	/* 16->16 dst_end sign: conv_16_sign_end */
	conv_16_end,		/* 16->16 src_end: conv_16_end */
	conv_16_end_sign,	/* 16->16 src_end sign: conv_16_end_sign */
	NULL,			/* 16->16 src_end dst_end: Nothing to do */
	conv_16_end_sign_end,	/* 16->16 src_end dst_end sign: conv_16_end_sign_end */
	conv_16_24,		/* 16->24: conv_16_24 */
	conv_16_24_sign,	/* 16->24 sign: conv_16_24_sign */
	conv_16_24_end,		/* 16->24 dst_end: conv_16_24_end */
	conv_16_24_sign_end,	/* 16->24 dst_end sign: conv_16_24_sign_end */
	conv_16_end_24,		/* 16->24 src_end: conv_16_end_24 */
	conv_16_end_24_sign,	/* 16->24 src_end sign: conv_16_end_24_sign */
	conv_16_end_24_end,	/* 16->24 src_end dst_end: conv_16_end_24_end */
	conv_16_end_24_sign_end,/* 16->24 src_end dst_end sign: conv_16_end_24_sign_end */
	conv_16_32,		/* 16->32: conv_16_32 */
	conv_16_32_sign,	/* 16->32 sign: conv_16_32_sign */
	conv_16_32_end,		/* 16->32 dst_end: conv_16_32_end */
	conv_16_32_sign_end,	/* 16->32 dst_end sign: conv_16_32_sign_end */
	conv_16_end_32,		/* 16->32 src_end: conv_16_end_32 */
	conv_16_end_32_sign,	/* 16->32 src_end sign: conv_16_end_32_sign */
	conv_16_end_32_end,	/* 16->32 src_end dst_end: conv_16_end_32_end */
	conv_16_end_32_sign_end,/* 16->32 src_end dst_end sign: conv_16_end_32_sign_end */
	conv_24_8,		/* 24->8: conv_24_8 */
	conv_24_8_sign,		/* 24->8 sign: conv_24_8_sign */
	conv_24_8,		/* 24->8 dst_end: conv_24_8 */
	conv_24_8_sign,		/* 24->8 dst_end sign: conv_24_8_sign */
	conv_24_end_8,		/* 24->8 src_end: conv_24_end_8 */
	conv_24_end_8_sign,	/* 24->8 src_end sign: conv_24_end_8_sign */
	conv_24_end_8,		/* 24->8 src_end dst_end: conv_24_end_8 */
	conv_24_end_8_sign,	/* 24->8 src_end dst_end sign: conv_24_end_8_sign */
	conv_24_16,		/* 24->16: conv_24_16 */
	conv_24_16_sign,	/* 24->16 sign: conv_24_16_sign */
	conv_24_16_end,		/* 24->16 dst_end: conv_24_16_end */
	conv_24_16_sign_end,	/* 24->16 dst_end sign: conv_24_16_sign_end */
	conv_24_end_16,		/* 24->16 src_end: conv_24_end_16 */
	conv_24_end_16_sign,	/* 24->16 src_end sign: conv_24_end_16_sign */
	conv_24_end_16_end,	/* 24->16 src_end dst_end: conv_24_end_16_end */
	conv_24_end_16_sign_end,/* 24->16 src_end dst_end sign: conv_24_end_16_sign_end */
	NULL,			/* 24->24: Nothing to do */
	conv_24_sign,		/* 24->24 sign: conv_24_sign */
	conv_24_end,		/* 24->24 dst_end: conv_24_end */
	conv_24_sign_end,	/* 24->24 dst_end sign: conv_24_sign_end */
	conv_24_end,		/* 24->24 src_end: conv_24_end */
	conv_24_end_sign,	/* 24->24 src_end sign: conv_24_end_sign */
	NULL,			/* 24->24 src_end dst_end: Nothing to do */
	conv_24_end_sign_end,	/* 24->24 src_end dst_end sign: conv_24_end_sign_end */
	conv_24_32,		/* 24->32: conv_24_32 */
	conv_24_32_sign,	/* 24->32 sign: conv_24_32_sign */
	conv_24_32_end,		/* 24->32 dst_end: conv_24_32_end */
	conv_24_32_sign_end,	/* 24->32 dst_end sign: conv_24_32_sign_end */
	conv_24_end_32,		/* 24->32 src_end: conv_24_end_32 */
	conv_24_end_32_sign,	/* 24->32 src_end sign: conv_24_end_32_sign */
	conv_24_end_32_end,	/* 24->32 src_end dst_end: conv_24_end_32_end */
	conv_24_end_32_sign_end,/* 24->32 src_end dst_end sign: conv_24_end_32_sign_end */
	conv_32_8,		/* 32->8: conv_32_8 */
	conv_32_8_sign,		/* 32->8 sign: conv_32_8_sign */
	conv_32_8,		/* 32->8 dst_end: conv_32_8 */
	conv_32_8_sign,		/* 32->8 dst_end sign: conv_32_8_sign */
	conv_32_end_8,		/* 32->8 src_end: conv_32_end_8 */
	conv_32_end_8_sign,	/* 32->8 src_end sign: conv_32_end_8_sign */
	conv_32_end_8,		/* 32->8 src_end dst_end: conv_32_end_8 */
	conv_32_end_8_sign,	/* 32->8 src_end dst_end sign: conv_32_end_8_sign */
	conv_32_16,		/* 32->16: conv_32_16 */
	conv_32_16_sign,	/* 32->16 sign: conv_32_16_sign */
	conv_32_16_end,		/* 32->16 dst_end: conv_32_16_end */
	conv_32_16_sign_end,	/* 32->16 dst_end sign: conv_32_16_sign_end */
	conv_32_end_16,		/* 32->16 src_end: conv_32_end_16 */
	conv_32_end_16_sign,	/* 32->16 src_end sign: conv_32_end_16_sign */
	conv_32_end_16_end,	/* 32->16 src_end dst_end: conv_32_end_16_end */
	conv_32_end_16_sign_end,/* 32->16 src_end dst_end sign: conv_32_end_16_sign_end */
	conv_32_24,		/* 32->24: conv_32_24 */
	conv_32_24_sign,	/* 32->24 sign: conv_32_24_sign */
	conv_32_24_end,		/* 32->24 dst_end: conv_32_24_end */
	conv_32_24_sign_end,	/* 32->24 dst_end sign: conv_32_24_sign_end */
	conv_32_end_24,		/* 32->24 src_end: conv_32_end_24 */
	conv_32_end_24_sign,	/* 32->24 src_end sign: conv_32_end_24_sign */
	conv_32_end_24_end,	/* 32->24 src_end dst_end: conv_32_end_24_end */
	conv_32_end_24_sign_end,/* 32->24 src_end dst_end sign: conv_32_end_24_sign_end */
	NULL,			/* 32->32: Nothing to do */
	conv_32_sign,		/* 32->32 sign: conv_32_sign */
	conv_32_end,		/* 32->32 dst_end: conv_32_end */
	conv_32_sign_end,	/* 32->32 dst_end sign: conv_32_sign_end */
	conv_32_end,		/* 32->32 src_end: conv_32_end */
	conv_32_end_sign,	/* 32->32 src_end sign: conv_32_end_sign */
	NULL,			/* 32->32 src_end dst_end: Nothing to do */
	conv_32_end_sign_end	/* 32->32 src_end dst_end sign: conv_32_end_sign_end */
};

/*
 * Types
 */

/*
 * Private type
 */

struct priv_cfmt_t
{
    convert_t			convert;

    XMM_AudioFormat		saf;
    XMM_AudioFormat		daf;
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info_cfmt;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *cfmt_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_cfmt_t	*priv;
  int			ret = 0, sSS, dSS;

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check audio format */
  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: 0x%x not supported ( only PCM supported )", saf->format );
	return (void *)ret;
  }

  if(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: 0x%x not supported ( only PCM supported )", daf->format );
	return (void *)ret;
  }

  if( saf->samprate != daf->samprate )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() samprate: source(%i) != dest(%i)", saf->samprate, daf->samprate );
	return (void *)ret;
  }

  if( saf->channels != daf->channels )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() channels: source(%i) != dest(%i)", saf->channels, daf->channels );
	return (void *)ret;
  }

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_cfmt, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_cfmt_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_cfmt_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Initialize data */
  memcpy( &priv->saf, saf, sizeof(XMM_AudioFormat));
  memcpy( &priv->daf, daf, sizeof(XMM_AudioFormat));

#ifdef DEBUG
  xmm_logging( 1, "FA-cfmt! saf: 0x%x, %i channel(s), %i Hz\n", saf->format, saf->channels, saf->samprate );
  xmm_logging( 1, "FA-cfmt! daf: 0x%x, %i channel(s), %i Hz\n", daf->format, daf->channels, daf->samprate );
#endif

  sSS = ( saf->format & XMM_AUDIO_MASK_SIZE );
  dSS = ( daf->format & XMM_AUDIO_MASK_SIZE );

  /* Set conversion function */
  priv->convert = ((convert_t(*)[4][2][2][2])convert_functions)
	    [( sSS >> 3 ) - 1 ]
	    [( dSS >> 3 ) - 1 ]
	    [( saf->format & XMM_AUDIO_MASK_BE ) ? 1 : 0 ]
	    [( daf->format & XMM_AUDIO_MASK_BE ) ? 1 : 0 ]
	    [( saf->format & XMM_AUDIO_MASK_SIGNED ) != ( daf->format & XMM_AUDIO_MASK_SIGNED )];

  return pAFilter;
}

/*
 * Free filter
 */
static int cfmt_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int cfmt_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_cfmt_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data ) = param * ( priv->saf.format & XMM_AUDIO_MASK_SIZE ) / ( priv->daf.format & XMM_AUDIO_MASK_SIZE );
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(FA-cfmt) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(FA-cfmt) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int cfmt_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_cfmt_t	*priv = (struct priv_cfmt_t *)filter->sys.priv;
  uint32_t		samples;

  /* Number of (mono) samples in all channels */
  samples = isize / (( priv->saf.format & XMM_AUDIO_MASK_SIZE ) / 8 );

#ifdef DEBUG
  xmm_logging( 1, "FA-cfmt! Input: %i bytes ( %i samples )\n", isize, samples / priv->saf.channels );
#endif

  priv->convert( src, dest, samples );

  *osize = isize * ( priv->daf.format & XMM_AUDIO_MASK_SIZE ) / ( priv->saf.format & XMM_AUDIO_MASK_SIZE );

#ifdef DEBUG
  xmm_logging( 1, "FA-cfmt! Output: %i bytes ( %i samples )\n", *osize, samples / priv->saf.channels );
#endif

  return XMM_RET_OK;
}

/*
 * Process data (BQ)
 */
static int cfmt_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio	plugin_info_cfmt = {{ NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-FORMAT",
				    "libxmm sample format converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    cfmt_Open, cfmt_Close, cfmt_Control,
				    cfmt_Process, cfmt_ProcessBQ };
