/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * achain_sr.c
 * Sample rate conversion
 */

#include <stdio.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/xmmctl.h"
#include "libxmm/achain.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Resampling ( ALSA: kernel/plugin/rate.c )
 */

/*
 * Definitions
 */

#define SHIFT			11
#define BITS			(1 << SHIFT)
#define MASK			(BITS - 1)

#define MAX_VOICES		6

/*
 * Types
 */

struct priv_alsa_csr_s
{
    unsigned int		pitch;
    unsigned int		pos;
    signed short		last_S1[MAX_VOICES];
    signed short		last_S2[MAX_VOICES];
};

typedef void (*resample16_t)( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size );
typedef void (*resample8_t)( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size );

static void resample16_shrink( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size );
static void resample16_expand( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size );
static void resample8_shrink( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size );
static void resample8_expand( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size );

/*
 * Types
 */

/*
 * Private type
 */

struct priv_csr_t
{
    struct priv_alsa_csr_s	alsa;
    resample16_t		resample16;
    resample8_t			resample8;

    XMM_AudioFormat		saf;
    XMM_AudioFormat		daf;
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info_csr;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *csr_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pAFilter;
  struct priv_csr_t	*priv;
  int			ch, ret = 0;

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check audio format */
  if( saf->format != daf->format )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: source(0x%x) != dest(0x%x)", saf->format, daf->format );
	return (void *)ret;
  }

  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() format: 0x%x not supported ( only PCM supported )", saf->format );
	return (void *)ret;
  }

  if( saf->channels != daf->channels )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() channels: source(%i) != dest(%i)", saf->channels, daf->channels );
	return (void *)ret;
  }

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate plugin data */
  if(( pAFilter = xmm_memdup_x( &plugin_info_csr, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_csr_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_csr_t *) &pAFilter[1];
  pAFilter->sys.priv = (void *) priv;
  pAFilter->sys.xmm = xmm;

  /* Initialize data */
  memcpy( &priv->saf, saf, sizeof(XMM_AudioFormat));
  memcpy( &priv->daf, daf, sizeof(XMM_AudioFormat));

#ifdef DEBUG
  xmm_logging( 1, "FA-csr! saf: 0x%x, %i channel(s), %i Hz\n", saf->format, saf->channels, saf->samprate );
  xmm_logging( 1, "FA-csr! daf: 0x%x, %i channel(s), %i Hz\n", daf->format, daf->channels, daf->samprate );
#endif

  /* Initialize conversion */
  if( saf->samprate < daf->samprate )
  {
	priv->resample16 = resample16_expand;
	priv->resample8 = resample8_expand;
	priv->alsa.pitch = (( saf->samprate << SHIFT ) + ( daf->samprate >> 1 )) / daf->samprate;
  }
  else
  {
	priv->resample16 = resample16_shrink;
	priv->resample8 = resample8_shrink;
	priv->alsa.pitch = (( daf->samprate << SHIFT ) + ( saf->samprate >> 1 )) / saf->samprate;
  }

  priv->alsa.pos = 0;

  for( ch = 0; ch < daf->channels; ch++ )
	priv->alsa.last_S1[ch] = priv->alsa.last_S2[ch] = 0;

  return pAFilter;
}

/*
 * Free filter
 */
static int csr_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int csr_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_csr_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data ) = param * priv->saf.samprate / priv->daf.samprate;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(FA-csr) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(FA-csr) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int csr_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_csr_t	*priv = (struct priv_csr_t *)filter->sys.priv;
  uint32_t		isamples;
  uint32_t		osamples;

  isamples = isize / ( priv->saf.channels * (( priv->saf.format & XMM_AUDIO_MASK_SIZE ) / 8 ));
  osamples = ( isize * priv->daf.samprate / priv->saf.samprate ) / ( priv->daf.channels * (( priv->daf.format & XMM_AUDIO_MASK_SIZE ) / 8 ));

#ifdef DEBUG
  xmm_logging( 1, "FA-csr: Input: %i bytes ( %i samples )\n", isize, isamples );
#endif

  if(( priv->daf.format & XMM_AUDIO_MASK_SIZE ) == 16 )
  {
	priv->resample16( &priv->alsa, priv->daf.channels, (signed short *)src,
			isamples, (signed short *)dest, osamples );
  }
  else
  {
	priv->resample8( &priv->alsa, priv->daf.channels, src,
			isamples, dest, osamples );
  }

  *osize = osamples * priv->daf.channels * ( priv->daf.format & XMM_AUDIO_MASK_SIZE ) / 8;

#ifdef DEBUG
  xmm_logging( 1, "FA-csr! Output: %i bytes ( %i samples )\n", *osize, osamples );
#endif

  return XMM_RET_OK;
}

/*
 * Process data (BQ)
 */
static int csr_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio	plugin_info_csr = {{ NULL,
				    XMM_PLUGIN_ID,
				    XMM_PLUGIN_TYPE_AFILTER,
				    0,
				    XMM_VERSION_NUM,
				    "",
				    "STDACONV-SAMPLERATE",
				    "libxmm sample rate converter",
				    "Copyright (c) 2001 by Arthur Kleer",
				    NULL, NULL },
				    csr_Open, csr_Close, csr_Control,
				    csr_Process, csr_ProcessBQ };

/*
 * Internal code ( ALSA: kernel/plugin/rate.c )
 */


/*
 * Code
 */


/*
 * Shrink ( 16 bit )
 */
static void resample16_shrink( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  signed short *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	while( size > 0 )
	{
		S1 = S2;

		if(( src - src_ptr ) < ( src_size * voices ))
		{
			S2 = *src;
			src += voices;
		}

		if( pos >> SHIFT )
		{
			pos &= MASK;
			val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

			if( val < -32768 )	val = -32768;
			else if( val > 32767 )	val = 32767;

    			*dst = val;
			dst += voices;
			size--;
		}
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Expand ( 16 bit )
 */
static void resample16_expand( struct priv_alsa_csr_s *data, int voices,
					signed short *src_ptr, int src_size,
					signed short *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  signed short *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	if( pos >> SHIFT )
	{
		pos &= MASK;
		S1 = S2;
		S2 = *src;
	}

	while( size-- > 0 )
	{
		if( pos >> SHIFT )
		{
			src += voices;
			pos &= MASK;
			S1 = S2;
			if(( src - src_ptr ) < src_size * voices)	S2 = *src;
		}

		val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

		if( val < -32768 )	val = -32768;
		else if( val > 32767 )	val = 32767;

		*dst = val;
		dst += voices;
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Shrink ( 8 bit )
 */
static void resample8_shrink( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  unsigned char *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	while( size > 0 )
	{
		S1 = S2;

		if(( src - src_ptr ) < ( src_size * voices ))
		{
			S2 = (*src << 8) ^ 0x8000;
			src += voices;
		}

		if( pos >> SHIFT )
		{
			pos &= MASK;
			val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

			if( val < -32768 )	val = -32768;
			else if( val > 32767 )	val = 32767;

			*dst = (val >> 8) ^ 0x0080;
			dst += voices;
			size--;
		}
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}

/*
 * Expand ( 8 bit )
 */
static void resample8_expand( struct priv_alsa_csr_s *data, int voices,
					unsigned char *src_ptr, int src_size,
					unsigned char *dst_ptr, int dst_size )
{
  unsigned int pos;
  signed int val;
  signed short S1, S2;
  int voice;
  unsigned char *src, *dst;
  int size;

  for( voice = 0; voice < voices; voice++ )
  {
	pos = data->pos;
	S1 = data->last_S1[voice];
	S2 = data->last_S2[voice];
	src = src_ptr + voice;
	dst = dst_ptr + voice;
	size = dst_size;

	if( pos >> SHIFT )
	{
		pos &= MASK;
		S1 = S2;
		S2 = (*src << 8) ^ 0x8000;
	}

	while( size-- > 0 )
	{
		if( pos >> SHIFT )
		{
			src += voices;
			pos &= MASK;
			S1 = S2;
			if(( src - src_ptr ) < src_size * voices )
				S2 = (*src << 8) ^ 0x8000;
		}

		val = S1 + (( S2 - S1 ) * (signed int)pos) / BITS;

		if( val < -32768 )	val = -32768;
		else if( val > 32767 )	val = 32767;

		*dst = (val >> 8) ^ 0x0080;
		dst += voices;
		pos += data->pitch;
	}

	data->last_S1[voice] = S1;
	data->last_S2[voice] = S2;
	data->pos = pos;
  }
}
