/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * afilter.c
 * Audio Filter Plugin manager/wrapper
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/version.h"
#include "libxmm/xmmctl.h"
#include "libxmm/error.h"
#include "libxmm/afilter.h"
#include "libxmm/util/utils.h"
#include "../xmmpriv.h"

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/*
 * Load filter
 */
XMM_PluginFilterAudio *xmm_FilterAudioLoad( char *filename, void *_xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFA = NULL, *tFA;
  XMM			*xmm = (XMM *) _xmm;

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() filename == NULL\n" );
	return NULL;
  }

  /* Register plugin */
  if(( tFA = (XMM_PluginFilterAudio *)xmm_PluginRegister( filename )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to load plugin '%s'\n", filename );
	return NULL;
  }

  /* Initialize plugin */
  pFA = tFA->Open( xmm, saf, daf, 0 );
  if( pFA == NULL )	return NULL;

  return pFA;
}

/*
 * Find filter
 */
XMM_PluginFilterAudio *xmm_FilterAudioFind( void *_xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_List		*le;
  XMM_PluginFilterAudio	*pAFilter = NULL, *tAFilter = NULL;
  XMM			*xmm = (XMM *) _xmm;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( saf == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() saf = NULL\n" );
	return NULL;
  }

  if( daf == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() daf = NULL\n" );
	return NULL;
  }

  if( xmm->pACodecList == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() pACodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

#ifdef VERBOSE
  xmm_logging( 1, __FUNCTION__ "() source audio format:\n" );
  xmm_logging( 1, "\t format = 0x%x\n", saf->format );
  xmm_logging( 1, "\t samprate = %i\n", saf->samprate );
  xmm_logging( 1, "\t channels = %i\n", saf->channels );
  xmm_logging( 1, "\t bitrate = %i\n", saf->bitrate );
  xmm_logging( 1, "\t blockSize = %i\n", saf->blockSize );
  xmm_logging( 1, "\t extraSize = %i\n", saf->extraSize );
  xmm_logging( 1, "\t extraType = 0x%x\n", saf->extraType );
  xmm_logging( 1, "\t desc = %s\n", saf->desc );

  xmm_logging( 1, __FUNCTION__ "() destination audio format:\n" );
  xmm_logging( 1, "\t format = 0x%x\n", daf->format );
  xmm_logging( 1, "\t samprate = %i\n", daf->samprate );
  xmm_logging( 1, "\t channels = %i\n", daf->channels );
  xmm_logging( 1, "\t bitrate = %i\n", daf->bitrate );
  xmm_logging( 1, "\t blockSize = %i\n", daf->blockSize );
  xmm_logging( 1, "\t extraSize = %i\n", daf->extraSize );
  xmm_logging( 1, "\t extraType = 0x%x\n", daf->extraType );
  xmm_logging( 1, "\t desc = %s\n", daf->desc );
#endif

  /* Find plugin */
  for( le = xmm->pACodecList; le; le = le->next )
  {
	/* Only audio codecs */
	if(((XMM_Plugin *)le->data)->Flags != XMM_FILTER_FLAG_ACODEC )
		continue;

	/* Only audio codecs */
	if(((XMM_PluginFilterAudio *)le->data)->Open( xmm, saf, daf, flags | XMM_FILTER_AOF_QUERY ) == NULL )
	{
		tAFilter = (XMM_PluginFilterAudio *)le->data;
		break;
	}
  }

  if( tAFilter == NULL || ( le == NULL ))
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() No matching audio codec plugin found ( formatID = 0x%x -> 0x%x, flags %x )", saf->format, daf->format, flags );
	return NULL;
  }

  /* Initialize  */
  if(( pAFilter = tAFilter->Open( xmm, saf, daf, flags )) == NULL )	
	return NULL;

  return pAFilter;
}

/*
 * (Re)open filter
 */
XMM_PluginFilterAudio *xmm_FilterAudioOpen( XMM_PluginFilterAudio *filter, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pAFilter;

  /* Initialize  */
  if(( pAFilter = filter->Open( filter->sys.xmm, saf, daf, flags )) == NULL )	
	return NULL;

  return pAFilter;

}

/*
 * Close filter / Free plugin.
 */
int xmm_FilterAudioCloseFree( XMM_PluginFilterAudio *filter )
{
  XMM_Plugin	*plugin;

  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  /* Get plugin ID */
  plugin = xmm_memdup( &filter->sys, sizeof( XMM_Plugin ));

  /* Free plugin */
  filter->Close( filter );

  /* unload plugin */
  if( plugin )
  {
	xmm_PluginRemove( plugin );
	free( plugin );
  }

  return XMM_RET_OK;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ----------------------- Wrapper functions ----------------------- * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Close filter
 */
int xmm_FilterAudioClose( XMM_PluginFilterAudio *filter )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Close( filter );
}

/*
 * Control call.
 */
int xmm_FilterAudioControl( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Control( filter, cmd, param, data );
}

/*
 * Process data
 */
int xmm_FilterAudioProcess( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->Process( filter, src, isize, dest, osize, flags );
}

/*
 * Process data (BQ)
 */
int xmm_FilterAudioProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  if( filter == NULL )	return XMM_RET_INVALID_ARG;

  return filter->ProcessBQ( filter, bq, dest, osize, flags );
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * ------------------------- Misc functions ------------------------ * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Get Information about installed audio codec filters
 */
XMM_List *xmm_FilterAudioInfoList( void *_xmm, uint32_t flags )
{
  XMM_List		*le, *ailist = NULL;
  XMM_FilterAudioInfo	*ai;
  XMM			*xmm = (XMM *) _xmm;
  char			*ptr, *tmp;

  if( xmm == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() xmm = NULL\n" );
	return NULL;
  }

  if( xmm->pACodecList == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() pACodecList: libXMM not initialized or no codec plugins found ?\n" );
	return NULL;
  }

  /* Find plugins */
  for( le = xmm->pACodecList; le; le = le->next )
  {
	/* Only audio codecs */
	if( !(((XMM_Plugin *)le->data)->Flags & XMM_FILTER_FLAG_ACODEC ))
		continue;

	/* Get codec info */
	((XMM_PluginFilterAudio *)le->data)->Control( le->data, XMM_CTLGET_FILTER_INFO, 0, &ai );

	/* If codec according to flags, fill file member and appene codec */
	if( ai && (( ai->caps & flags ) == flags ))
	{
		tmp = strdup( ((XMM_Plugin *)le->data)->Filename );

		ptr = strrchr( tmp, '/' );
		if( ptr )
		{
			if( strlen( ptr + 1 ) > 63 )	ptr[63] = '\0';
			strcpy( ai->file, ptr + 1 );
		}
		else
		{
			if( strlen( tmp ) > 63 )	tmp[63] = '\0';
			strcpy( ai->file, tmp );
		}

		free( tmp );

		ailist = xmmList_Append( ailist, ai );
	}
  }

  return ailist;
}

/*
 * Get audio filter info
 */
XMM_FilterAudioInfo *xmm_FilterAudioInfo( void *_xmm, char *filename )
{
  XMM_FilterAudioInfo	*fai, *faidup;
  XMM_PluginFilterAudio	*pSound;
  int			ret;

  /* load sound output plugin */
  pSound = (XMM_PluginFilterAudio *)xmm_PluginRegister( filename );
  if( pSound == NULL )	
  {
	xmm_SetError( _xmm, XMM_RET_ERROR, __FUNCTION__ "() ERROR: Unable to register '%s'", filename );
	return NULL;
  }

  /* Get audio formats */
  ret = xmm_FilterAudioControl( pSound, XMM_CTLGET_FILTER_INFO, 0, &fai );
  if( ret < XMM_RET_OK )	return NULL;

  /* Duplicate audio filter info */
  if(( faidup = xmm_memdup_x( fai, sizeof( XMM_FilterAudioInfo ), sizeof( XMM_AudioFormat ) * fai->nfmt )) == NULL )
  {
	xmm_SetError( _xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to duplicate audio filter info" );
	return NULL;
  }

  faidup->fmt = (XMM_AudioFormat *) &faidup[1];
  memcpy( faidup->fmt, fai->fmt, sizeof( XMM_AudioFormat ) * fai->nfmt );

  /* free plugin */
  xmm_PluginRemove( (XMM_Plugin *)pSound );

  return faidup;
}
