/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * event.c
 * Event handling
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "libxmm/xmmp.h"
#include "libxmm/event.h"
#include "libxmm/event_ksym.h"
#include "libxmm/util/buffer.h"
#include "libxmm/util/utils.h"
#include "libxmm/error.h"
#include "../xmmpriv.h"
#include "event.h"

/*
 * Prototypes
 */

static int ReadKeymap( char *file, struct KeymapEntry_t	*mKeymap );

/*
 * Initialize Event handling
 */
int xmm_InitEvent( void *_xmm )
{
  char				*ptr;
  int				ret, i;
  struct event_queue_private	*priv;
  XMM				*xmm = (XMM *) _xmm;

  /* Allocate private event queue data */
  if( xmm->pEQpriv != NULL )
	return xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Event handling already initialized." );

  if(( xmm->pEQpriv = malloc( sizeof( struct event_queue_private ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() private EQ data" );

  priv = xmm->pEQpriv;

  /* Initialize private data */
  xmmBQ_Init( &priv->bq );
  priv->mod = 0;  

  /*
   * Load keymap.
   * First try user xmmp directory, then global xmmp directory
   */

  ptr = xmm_gethome_filename( XMM_FILE_KEYMAP );
  ret = ReadKeymap( ptr, priv->mKeymap );
  free( ptr );

  if( ret < 0 )
  {
	ptr = xmm_gethome_filename( LIBDIR XMM_FILE_KEYMAP );
	ret = ReadKeymap( ptr, priv->mKeymap );
	free( ptr );
  }
  
  if( ret < 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to read keymap file." );

  /* Clear key buffer */
  for( i = 0; i < KEYB_BUFFERSIZE; i++ )	priv->mBuffer[i] = 0;

  return XMM_RET_OK;
}

/*
 * Free event stuff
 */
int xmm_ExitEvent( void *_xmm )
{
  struct event_queue_private	*priv = (((XMM *)_xmm)->pEQpriv);
  XMM				*xmm = (XMM *) _xmm;

  if( priv )
  {
	xmmBQ_Free( &priv->bq );
	free( priv );
  }

  xmm->pEQpriv = NULL;

  return XMM_RET_OK;
}

/*
 * Add event to queue ( and do some event dependent stuff )
 *
 * Key event:
 *	* type, scode, state have to be initialized
 *	* sym, mod will be initialized
 */
int xmm_PushEvent( void *_xmm, XMM_Event *event, int queue )
{
  int				mod;
  XMM_Event_Key			*key;
  struct event_queue_private	*priv = ((XMM *)_xmm)->pEQpriv;

  if( priv == NULL )
	return xmm_SetError( _xmm, XMM_RET_NULL, __FUNCTION__ "() priv = NULL. Event subsystem not initialized ?" );

  switch( event->type )
  {
	case	XMM_EVENT_KEY:
		    key = (XMM_Event_Key *) event;

		    /* update scancode table */
		    if( key->state == XMM_KEY_PRESSED )
			priv->mBuffer[ key->scode ] = 1;

		    if( key->state == XMM_KEY_RELEASED )
			priv->mBuffer[ key->scode ] = 0;

		    /* Check / Update key modifiers */
		    key->sym = priv->mKeymap[ key->scode ].n;
		    if(( key->sym & XMM_KEYSYM_MASK_MOD ) == XMM_KEYSYM_MASK_MOD )
		    {
			mod = XMM_KEYSYM2KEYMOD( key->sym );

			if( key->state == XMM_KEY_PRESSED )
			    priv->mod |= mod;

			if( key->state == XMM_KEY_RELEASED )
			    priv->mod &= ~mod;

			return XMM_RET_OK;
		    }

		    /* Check character modificators */
		    if(( priv->mod & XMM_KEYMOD_SHIFT ) && priv->mKeymap[ key->scode ].s )
			key->sym = priv->mKeymap[ key->scode ].s;

		    if(( priv->mod & XMM_KEYMOD_ALT ) && ( priv->mod & XMM_KEYMOD_CTRL ) && priv->mKeymap[ key->scode ].s )
			key->sym = priv->mKeymap[ key->scode ].ca;

		    /* Set modificators */
		    key->mod = priv->mod;

		    break;
  }

  if( queue )
  {
	key = (XMM_Event_Key *) event;

	xmmBQ_Add( &priv->bq, event, sizeof( XMM_Event ));
  }
  else
  {
	if( priv->EventCB )	return	priv->EventCB( priv->priv_data, event );
  }

  return XMM_RET_OK;
}

/*
 * Get event from queue
 */
int xmm_PollEvent( void *_xmm, int mask, XMM_Event *event )
{
  struct event_queue_private	*priv = ((XMM *) _xmm)->pEQpriv;
  int				ret;

  if( priv == NULL )
	return xmm_SetError( _xmm, XMM_RET_NULL, __FUNCTION__ "() priv = NULL. Event subsystem not initialized ?" );

  ret = xmmBQ_HeadRead( &priv->bq, event, sizeof( XMM_Event ));
  if( ret == 0 )	return 0;

  return 1;
}

/*
 * Set event handler
 */
int xmm_SetEventCB( void *_xmm, int (*eventCB)( void *priv, XMM_Event *event ), void *priv_data, int onlyCB )
{
  struct event_queue_private	*priv = ((XMM *) _xmm)->pEQpriv;

  if( priv == NULL )
	return xmm_SetError( _xmm, XMM_RET_NULL, __FUNCTION__ "() priv = NULL. Event subsystem not initialized ?" );

  priv->EventCB = eventCB;
  priv->priv_data = priv_data;
  priv->onlyCB = onlyCB;

  return XMM_RET_OK;
}

/*
 * Private code
 */

static int ReadKeymap( char *file, struct KeymapEntry_t	*mKeymap )
{
  FILE *stream;
  char buffer[256], *end, token[32], *ptr;
  int code, asc1, asc2, asc3, line = 0, x;

  /* Open file */
  if(( stream = fopen( file, "rb" )) == NULL )	return -1;

  /* Parse all lines in file */
  while(( end = fgets( buffer, 255, stream )))
  {
	line++;
	asc1 = asc2 = asc3 = 0;
	
	while( isspace( *end ))	end++;
	if(( *end == 10 ) || ( *end == '#' ))	continue;

	code = strtoul( end, &end, 10 );

	while( isspace( *end ))	end++;
	if(( *end != 10 ) && ( *end != '#' ))
	{
		if(( *end == '@' ) && isupper( *(end+1)))
		{
			for( end++, ptr = token; !isspace( *end ); )	*ptr++ = *end++;
			for( *ptr = '\0', x = 0; table_keysym[x].name[0]; x++ )
			{
				if( strcmp( token, table_keysym[x].name ))	continue;

				mKeymap[code].n = table_keysym[x].code;
				mKeymap[code].s = 0;
				mKeymap[code].ca = 0;
				break;
			}
			if( !table_keysym[x].name[0] )	xmm_logging( 1, "WARNING: Unknown keysym string. Ignoring line %i.\n", line );
			continue;
		}

		if(( *end == '@' ) && isdigit( *(end+1)))	asc1 = strtoul( end + 1, &end, 10 );
		else	asc1 = *end++;
	}

	while( isspace( *end ))	end++;
	if(( *end != 10 ) && ( *end != '#' ))
	{
		if(( *end == '@' ) && isdigit( *(end+1)))	asc2 = strtoul( end + 1, &end, 10 );
		else	asc2 = *end++;
	}

	while( isspace( *end ))	end++;
	if(( *end != 10 ) && ( *end != '#' ))
	{
		if(( *end == '@' ) && isdigit( *(end+1)))	asc3 = strtoul( end + 1, &end, 10 );
		else	asc3 = *end++;
	}
	
	mKeymap[code].n  = asc1;
	mKeymap[code].s  = asc2;
	mKeymap[code].ca = asc3;
  }

  fclose( stream );

  return 0;
}
