/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lpgraph.c
 * Graph manager/wrapper
 */

#include <stdio.h>
#include <stdlib.h>
#include "libxmm/xmmp.h"
#include "libxmm/lpgraph.h"
#include "libxmm/error.h"
#include "libxmm/util/utils.h"
#include "../xmmpriv.h"

/*
 * Open graph
 */

XMM_PluginGraph *xmm_GraphOpen( void *_xmm, char *filename )
{
  XMM_PluginGraph	*pGraph = NULL, *tGraph;
  XMM			*xmm = (XMM *) _xmm;

  if( filename == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() filename == NULL\n" );
	return NULL;
  }

  /* Register plugin */
  if(( tGraph = (XMM_PluginGraph *)xmm_PluginRegister( filename )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to load plugin '%s'\n", filename );
	return NULL;
  }

  /* Initialize plugin */
  pGraph = tGraph->Init( xmm );
  if( pGraph == NULL )	return NULL;

  return pGraph;
}

/*
 * Open graph from memory plugin
 */
XMM_PluginGraph *xmm_GraphOpenMP( void *xmm, XMM_PluginGraph *tGraph )
{
  if( tGraph == NULL )
  {
	xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() tGraph == NULL" );
	return NULL;
  }

  /* Initialize plugin */
  return tGraph->Init( xmm );
}

/*
 * Free plugin.
 */
int xmm_GraphClose( XMM_PluginGraph *graph )
{
  XMM_Plugin		*plugin;

  if( graph == NULL )	return XMM_RET_INVALID_ARG;

  /* Get plugin ID */
  plugin = xmm_memdup( &graph->sys, sizeof( XMM_Plugin ));

  /* Free plugin */
  graph->Close( graph );

  /* unload plugin */
  if( plugin )
  {
	xmm_PluginRemove( plugin );
	free( plugin );
  }

  return XMM_RET_OK;
}

/* ************************************************************************** *
 * * * * * * * * * * * * * * * * * * wrappers * * * * * * * * * * * * * * * * *
 * ************************************************************************** */

/*
 * Control call.
 */
int xmm_GraphControl( XMM_PluginGraph *graph, int cmd, uint32_t param, void *data )
{
  if( graph == NULL )	return XMM_RET_INVALID_ARG;

  return graph->Control( graph, cmd, param, data );
}

/*
 * Start graph output.
 */
int xmm_GraphStart( XMM_PluginGraph *graph, int width, int height, uint32_t format, int flags )
{
  if( graph == NULL )	return XMM_RET_INVALID_ARG;

  return graph->Start( graph, width, height, format, flags );
}

/*
 * Output some graph data.
 */
int xmm_GraphDraw( XMM_PluginGraph *graph, uint8_t *data[], int stride[], int x, int y, int width, int height, int flags )
{
  if( graph == NULL )	return XMM_RET_INVALID_ARG;

  graph->Draw( graph, data, stride, x, y, width, height, flags );

  return XMM_RET_OK;
}

/*
 * Blit image to screen.
 */
int xmm_GraphBlit( XMM_PluginGraph *graph )
{
  if( graph == NULL )	return XMM_RET_INVALID_ARG;

  graph->Blit( graph );

  return XMM_RET_OK;
}
