/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "libxmm/xmmp.h"
#include "libxmm/config.h"
#include "libxmm/error.h"
#include "libxmm/util/config.h"
#include "libxmm/util/utils.h"
#include "xmmpriv.h"

/*
 * Get codec properties 
 */
int xmm_AcCodecFile( void *xmm, uint32_t format, int type, XMM_AcCodecProperties *cp )
{
  char			id[10], *section, cmap[8], guid[256], *ptr;
  int			i, ret;
  XMM_AcCodecProperties	icp = { "", "", "", 0, "", 0, {0,0,0,{0,0,0,0,0,0,0,0}}};
  XMM_ConfigBlock	ccfg[] =
  {
    { icp.info,		"xxxx.info",	XMM_CFG_TYPE_STRING,	XMM_CFG_CPL_INFO,	"" },
    { icp.file,		"xxxx.file",	XMM_CFG_TYPE_STRING,	XMM_CFG_CPL_FILE,	"" },
    { icp.stat,		"xxxx.stat",	XMM_CFG_TYPE_STRING,	XMM_CFG_CPL_STAT,	"" },
    { &icp.flags,	"xxxx.flag",	XMM_CFG_TYPE_INT,	4,			"" },
    { icp.ofmt,		"xxxx.ofmt",	XMM_CFG_TYPE_STRING,	XMM_CFG_CPL_OFMT,	"" },
    { cmap,		"xxxx.cmap",	XMM_CFG_TYPE_STRING,	8,			"" },
    { &icp.msid,	"xxxx.msid",	XMM_CFG_TYPE_INT,	4,			"" },
    { guid,		"xxxx.guid",	XMM_CFG_TYPE_STRING,	256,			"" },
    { NULL, "", 0 }
  };

  /* Build section and line name */
  switch( type )
  {
	case XMM_CFG_CODEC_VFW:
		section = "vfw";
		strcpy( id, xmm_FOURCC_string( format ));
		break;

	case XMM_CFG_CODEC_ACM:
		section = "acm";
		sprintf( id, "%4.4x", format & 0xFFFF );
		break;

	case XMM_CFG_CODEC_DSHOW_VIDEO:
		section = "dshow-video";
		strcpy( id, xmm_FOURCC_string( format ));
		break;

	case XMM_CFG_CODEC_DSHOW_AUDIO:
		section = "dshow-audio";
		sprintf( id, "%4.4x", format & 0xFFFF );
		break;

	default:
		return xmm_SetError( xmm, XMM_RET_INVALID_ARG, __FUNCTION__ "() Unknown type" );
  }

  while( id[0] != '\0' )
  {
	cmap[0] = '\0';
	guid[0] = '\0';
	/* Initialize properties */
	icp.info[0] = '\0';
	icp.file[0] = '\0';
	icp.stat[0] = '\0';
	icp.flags = 0;
	icp.ofmt[0] = '\0';
	icp.msid = 0;

	/* Set format ID */
	for( i = 0; ccfg[i].ptr; i++ )	strncpy( ccfg[i].name, id, 4 );

	/* Load codec info */
	ret = xmmCfg_BlockLoad( XMM_FILE_CODEC_CFGW32, section, ccfg );
	if( ret < 0 )
	{
	    ret = xmmCfg_BlockLoad( LIBDIR XMM_FILE_CODEC_CFGW32, section, ccfg );
	}
  
	if( ret < 0 )
	    return xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to read codec config." );

	/* Do mapping */
	strcpy( id, cmap );
  }

  if( icp.file[0] == '\0' )
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, __FUNCTION__ "() Entry not found ( for ID 0x%x, type %s )", format, section );

  /* Parse GUID */
  if( guid[0] )
  {
	ptr = guid;
	if( ptr )	icp.guid.v1 = strtol( ptr, &ptr, 0 );
	if( ptr )	icp.guid.v2 = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v3 = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[0] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[1] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[2] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[3] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[4] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[5] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[6] = strtol( ptr + 1, &ptr, 0 );
	if( ptr )	icp.guid.v4[7] = strtol( ptr + 1, &ptr, 0 );
  }

  /* Copy properties */
  if( cp )	memcpy( cp, &icp, sizeof( XMM_AcCodecProperties ));
  
  return XMM_RET_OK;
}

/*
 * Get/Set name of sound plugin
 */
const char *xmm_AcSoundPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )
  {
	ptr = xmm_getsys_filename( "Plugins/Filter/Output/Audio", name );
	if( ptr == NULL )	return NULL;

	strcpy( xmm->sound_plugin, ptr );
	free( ptr );

	if( strstr( xmm->sound_plugin, ".so" ) == NULL )
	    strcat( xmm->sound_plugin, ".so" );
  }

  return xmm->sound_plugin;
}

/*
 * Get/Set name of graph plugin
 */
const char *xmm_AcGraphPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )
  {
	ptr = xmm_getsys_filename( "Plugins/Graph", name );
	if( ptr == NULL )	return NULL;

	strcpy( xmm->graph_plugin, ptr );
	free( ptr );

	if( strstr( xmm->graph_plugin, ".so" ) == NULL )
	    strcat( xmm->graph_plugin, ".so" );
  }

  return xmm->graph_plugin;
}

/*
 * Get/Set name of output plugin
 */
const char *xmm_AcOutputPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )
  {
	ptr = xmm_getsys_filename( "Plugins/Output", name );
	if( ptr == NULL )	return NULL;

	strcpy( xmm->output_plugin, ptr );
	free( ptr );

	if( strstr( xmm->output_plugin, ".so" ) == NULL )
	    strcat( xmm->output_plugin, ".so" );
  }

  return xmm->output_plugin;
}

/*
 * Get/Set name of graph conversion filter plugin
 */
const char *xmm_AcGConvPluginName( void *_xmm, char *name )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  if( name )
  {
	ptr = xmm_getsys_filename( "Plugins/Filter/Video", name );
	if( ptr == NULL )	return NULL;

	strcpy( xmm->gconv_plugin, ptr );
	free( ptr );

	if( strstr( xmm->gconv_plugin, ".so" ) == NULL )
	    strcat( xmm->gconv_plugin, ".so" );
  }

  return xmm->gconv_plugin;
}

/*
 * Get/Set name of output file
 */
const char *xmm_AcOutputFilename( void *_xmm, char *name, char *ext )
{
  XMM	*xmm = (XMM *) _xmm;
  char	*ptr, *ifn = NULL;
  int	len;

  /* Check arguments */
  if( xmm == NULL )	return NULL;

  /* Set file name */
  if( name )	ifn = strdup( name );

  if( ifn == NULL )	/* outputFilename set ??? */
  {
	if( xmm->oFilename )
	{
	    if( xmm->oFilename[0] != '\0' )
		ifn = strdup( xmm->oFilename );
	}
  }

  if( ifn == NULL )	/* Use input file name */
  {
	if( xmm->iFilename )	ifn = strdup( xmm->iFilename );
  }

  if( ifn == NULL )	/* default name */
  {
	ifn = strdup( "xmmp-output" );
  }

  /* Remove extension */
  if(( ptr = strrchr( ifn, '.' )) == NULL )	len = strlen( ifn );
  else	len = ptr - ifn;

  /* Free output file name */
  if( xmm->oFilename != NULL )	free( xmm->oFilename );

  /* Allocate output file name */
  if(( xmm->oFilename = malloc( len + 7 )) == NULL )
  {
	free( ifn );
	xmm_SetError( xmm, XMM_RET_ALLOC, __FUNCTION__ "() memory for output filename." );
	return NULL;
  }

  memcpy( xmm->oFilename, ifn, len );
  xmm->oFilename[len] = '\0';

  if( ext )
  {
	if( !strcmp( &ifn[len], ext ))	strcat( xmm->oFilename, ".2" );
	strcat( xmm->oFilename, ext );
  }

  free( ifn );
  return xmm->oFilename;
}
