/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * libxmm.c
 * Initialize and free libxmm (object)
 */

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "libxmm/xmmp.h"
#include "libxmm/plugin.h"
#include "libxmm/error.h"
#include "libxmm/event.h"
#include "libxmm/lpfilterv.h"
#include "libxmm/achain.h"
#include "libxmm/util/timer.h"
#include "libxmm/util/utils.h"
#include "libxmm/util/mmaccel.h"
#include "libxmm/util/config.h"
#include "xmmpriv.h"

/*
 * Globyal data
 */

/*
 * Init library
 */
void *xmm_Init( void )
{
  XMM			*xmm;
  XMM_ConfigBlock	xmm_config[] =
  {
    { NULL,	"GraphPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"SoundPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"OutputPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"GConvPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL, "", 0 }
  };
  char			*ptr;

  /* Allocate object */
  if(( xmm = malloc( sizeof( XMM ))) == NULL )
  {
	xmm_logging( 1, __FUNCTION__ "() Cannot allocate XMM object\n" );
	return NULL;
  }
  memset( xmm, 0, sizeof( XMM ));

  /* Load config */
  strcpy( xmm->graph_plugin, LIBDIR "Plugins/Graph/x11.so" );
  strcpy( xmm->sound_plugin, LIBDIR "Plugins/Filter/Output/Audio/oss.so" );
  strcpy( xmm->output_plugin, "" ); 
  strcpy( xmm->gconv_plugin, "" ); 

  xmm_config[0].ptr  = xmm->graph_plugin;
  xmm_config[1].ptr  = xmm->sound_plugin;
  xmm_config[2].ptr  = xmm->output_plugin;
  xmm_config[3].ptr  = xmm->gconv_plugin;

  xmmCfg_BlockLoad( XMM_FILE_CONFIG, "libxmm", xmm_config );

#if 1

  /* Initialize timer: 500 ms interval */
  if( xmmTimer_Init( 500 ) < 0 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, __FUNCTION__ "() Unable to initialize timer" );
	free( xmm );
	return NULL;
  }

#endif

  /* Init event handling */
  if( xmm_InitEvent( xmm ) < 0 )
  {
	free( xmm );
	return NULL;
  }

  /* Initialize Filter chain/subsystem */
  xmm_FilterAChainInit( xmm );
  xmm_FilterVChainInit( xmm );

  /* Codec Plugins */
  xmm->pVCodecList = xmm_PluginRegisterDir( LIBDIR "Plugins/Codec", XMM_PLUGIN_TYPE_CODEC );

  /* Audio filter (codec) plugins */
  xmm->pACodecList = xmm_PluginRegisterDir( LIBDIR "Plugins/Filter/Codec/Audio", XMM_PLUGIN_TYPE_AFILTER );

  /* Input Plugins */
  xmm->pInputList = xmm_PluginRegisterDir( LIBDIR "Plugins/Input", XMM_PLUGIN_TYPE_INPUT );

  /* I/O Plugins */
  xmm->pIOList = xmm_PluginRegisterDir( LIBDIR "Plugins/IO", XMM_PLUGIN_TYPE_IO );

  /* Create dirs */
  ptr = xmm_gethome_filename( "" );
  mkdir( ptr, 0700 );
  free( ptr );

  return xmm;
}

/*
 * Exit library
 */
void xmm_Exit( void *_xmm )
{
  XMM			*xmm = (XMM *) _xmm;
  XMM_ConfigBlock	xmm_config[] =
  {
    { NULL,	"GraphPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"SoundPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"OutputPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL,	"GConvPlugin",	XMM_CFG_TYPE_STRING, XMM_MAXPATHLEN, "" },
    { NULL, "", 0 }
  };

  /* Check arguments */
  if( xmm == NULL )	return;

  /* Free Filter chain/subsystem */
  xmm_FilterAChainExit( xmm );
  xmm_FilterVChainExit( xmm );

  /* DeInitialize timer */
  xmmTimer_Exit();  

  /* DeInitialize event handling */
  xmm_ExitEvent( xmm );

  /* Free Codec list */
  if( xmm->pVCodecList )	xmm_PluginRemoveList( xmm->pVCodecList );
  xmm->pVCodecList = NULL;

  /* Free Codec list */
  if( xmm->pACodecList )	xmm_PluginRemoveList( xmm->pACodecList );
  xmm->pACodecList = NULL;

  /* Free Input List */
  if( xmm->pInputList )		xmm_PluginRemoveList( xmm->pInputList );
  xmm->pInputList = NULL;

  /* Free I/O List */
  if( xmm->pIOList )		xmm_PluginRemoveList( xmm->pIOList );
  xmm->pIOList = NULL;

  /* Save config */
  xmm_config[0].ptr  = xmm->graph_plugin;
  xmm_config[1].ptr  = xmm->sound_plugin;
  xmm_config[2].ptr  = xmm->output_plugin;
  xmm_config[3].ptr  = xmm->gconv_plugin;

  xmmCfg_BlockSave( XMM_FILE_CONFIG, "libxmm", xmm_config );

  /* Free Object */
  free( xmm );
}
