/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * list.c
 * List structure
 */

#include <stdio.h>
#include <stdlib.h>
#include "libxmm/util/list.h"
#include "libxmm/util/utils.h"

/*
 * Free complete list
 */
void xmmList_Free( XMM_List *list )
{
  if( list )
  {
	if( list->prev )	list->prev->next = NULL;

	xmmList_Free( list->next );
	free( list );
  }
}

/*
 * Concat two lists
 */
XMM_List *xmmList_Concat( XMM_List *list, XMM_List *list2 )
{
  XMM_List *le;

  if( list == NULL )	return list2;	/* Only seconds list */
  if( list2 == NULL )	return list;	/* Only first list */

  for( le = list; le->next; le = le->next );

  le->next = list2;
  list2->prev = le;
  return list;
}

/*
 * Reverse the complete list
 */
XMM_List *xmmList_Reverse( XMM_List *list )
{
  XMM_List	*le = NULL;

  while( list )
  {
	le = list;
	list = list->next;
	le->next = le->prev;
	le->prev = list;
  }

  return le;
}

/*
 * Length of the list
 */
int xmmList_Length( XMM_List *list )
{
  int		len;

  for( len = 0; list; list = list->next )	len++;

  return len;
}

/* * * * * * * * * * * * * * * * * *
 *-*-*-*-* get/find nodes  *-*-*-*-*
 * * * * * * * * * * * * * * * * * */

/*
 * Get node at position 'pos' in list
 */
XMM_List *xmmList_GetNodePos( XMM_List *list, int pos )
{
  int		i;

  for( i = 0; list && ( i < pos ); i++ )	list = list->next;

  return list;
}

/*
 * Get node with key 'data'
 */
XMM_List *xmmList_GetNodeData( XMM_List *list, void *data )
{
  for( ; list && ( list->data != data ); )	list = list->next;

  return list;
}

/* * * * * * * * * * * * * * * * *
 *-*-*-*-* add elements  *-*-*-*-*
 * * * * * * * * * * * * * * * * */

/*
 * Append element to list
 */
XMM_List *xmmList_Append( XMM_List *list, void *data )
{
  XMM_List	*new = NULL, *last;

  if(( new = malloc( sizeof( XMM_List ))) == NULL )
  {
	xmm_logging( 1, "xmmList_Append: error allocating memory for new node\n" );
	return NULL;
  }

  new->data = data;
  new->next = NULL;
  new->prev = NULL;

  if( list == NULL )	return new;
  else
  {
	for( last = list; last->next; last = last->next );
	last->next = new;
	new->prev = last;
	return list;
  }
}

/*
 * Prepend element to list
 */
XMM_List *xmmList_Prepend( XMM_List *list, void *data )
{
  XMM_List	*new = NULL;

  if(( new = malloc( sizeof( XMM_List ))) == NULL )
  {
	xmm_logging( 1, "xmmList_Prepend: error allocating memory for new node\n" );
	return NULL;
  }

  new->data = data;
  new->next = NULL;
  new->prev = NULL;

  if( list != NULL )
  {
	if( list->prev )
	{

	    list->prev->next = new;
	    new->prev = list->prev;
	}
	list->prev = new;
	new->next = list;
  }

  return new;
}

/*
 * Insert element into list at position 'pos'
 */
XMM_List *xmmList_Insert( XMM_List *list, void *data, int pos )
{
  XMM_List	*le, *new = NULL;

  if( list == NULL )	return NULL;

  le = xmmList_GetNodePos( list, pos );
  if( le == NULL )	return xmmList_Append( list, data );	/* Append */

  /* insert node / prepend */
  if(( new = malloc( sizeof( XMM_List ))) == NULL )
  {
	xmm_logging( 1, "xmmList_Insert: error allocating memory for new node\n" );
	return NULL;
  }

  new->data = data;
  new->next = NULL;
  new->prev = NULL;

  if( le->prev )
  {
	le->prev->next = new;
	new->prev = le->prev;
  }

  le->prev = new;
  new->next = le;

  if( list == le )	return new;	/* New first element */
  return list;
}

/* * * * * * * * * * * * * * * * *
 *-*-*-*-* remove nodes  *-*-*-*-*
 * * * * * * * * * * * * * * * * */

/*
 * Remove node from list
 */
XMM_List *xmmList_RemoveNode( XMM_List *list, XMM_List *rem )
{
  if( list == NULL )	return NULL;
  if( rem == NULL )	return list;	/* Nothing to remove */

  if( list == rem )	list = list->next;

  if( rem->next )	rem->next->prev = rem->prev;
  if( rem->prev )	rem->prev->next = rem->next;

  rem->next = NULL;
  rem->prev = NULL;
  return list;
}

/*
 * Remove node at position 'pos' in list
 */
XMM_List *xmmList_RemoveNodePos( XMM_List *list, int pos )
{
  XMM_List	*rem;

  if( list == NULL )	return NULL;

  rem = xmmList_GetNodePos( list, pos );
  list = xmmList_RemoveNode( list, rem );

  free( rem );
  return list;
}

/*
 * Remove node with key 'data'
 */
XMM_List *xmmList_RemoveNodeData( XMM_List *list, void *data )
{
  XMM_List	*rem;

  if( list == NULL )	return NULL;

  rem = xmmList_GetNodeData( list, data );
  list = xmmList_RemoveNode( list, rem );

  free( rem );
  return list;
}
