/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * system-x86-unix.c
 * Code for x86 linux
 */

#include <inttypes.h>
#include <unistd.h>
#include <sys/time.h>
#include <libxmm/util/system.h>
#include <libxmm/util/mmaccel.h>
#include "config.h"

/*
 * Prototypes
 */

static uint32_t xmmSYS_count_init( void );
static uint64_t xmmSYS_longcount_init( void );

/*
 * Global data
 */

uint32_t (*xmmSYS_count)( void ) = xmmSYS_count_init;
uint64_t (*xmmSYS_longcount)( void ) = xmmSYS_longcount_init;

/* CPU clock */
static uint32_t cpu_freq = (uint32_t) -1;

/*
 * Sleep
 */
void xmmSYS_usleep( uint32_t usecs )
{
  usleep( usecs );
}

/*
 * Check file access
 */
int xmmSYS_access( const char *path, int mode )
{
  if( access( path, F_OK ) == 0 )	return 1;
  else	return 0;
}

/*
 * Internal data:
 * tick count
 */

/*
 * TSC functions
 */

#ifdef HAVE_CPU_IX86

static uint32_t xmmSYS_count_tsc( void )
{
  uint32_t tick;

  __asm__ __volatile__ ( "rdtsc\n" : "=a"(tick) : : "edx" );

  return tick;
}

static uint64_t xmmSYS_longcount_tsc( void )
{
  uint32_t tickl, tickh;

  __asm__ __volatile__ ("rdtsc\n" : "=a"(tickl), "=d"(tickh): );

  return ((uint64_t)tickh << 32 ) | (uint64_t) tickl;
}

#endif

/*
 * Kernel fallback functions
 */

static uint32_t xmmSYS_count_sys( void )
{
  struct timeval tv;

  gettimeofday( &tv, 0 );

  return (uint32_t) tv.tv_usec * ( cpu_freq / 1000000 );
}

static uint64_t xmmSYS_longcount_sys( void )
{
  struct timeval tv;

  gettimeofday( &tv, 0 );

  return (uint64_t) tv.tv_sec * cpu_freq + tv.tv_usec * ( cpu_freq / 1000000 );
}

/*
 * Init functions
 */

static uint32_t xmmSYS_count_init( void )
{
#ifdef HAVE_CPU_IX86
  if( xmmCPU_mmsupport() & XMM_MMACCEL_X86_TSC )
  {
	xmmSYS_count = xmmSYS_count_tsc;
  }
  else
  {
#endif
	cpu_freq = xmmCPU_clockfreq();
	xmmSYS_count = xmmSYS_count_sys;
#ifdef HAVE_CPU_IX86
  }
#endif

  return xmmSYS_count();
}

static uint64_t xmmSYS_longcount_init( void )
{
#ifdef HAVE_CPU_IX86
  if( 0 && xmmCPU_mmsupport() & XMM_MMACCEL_X86_TSC )
  {
	xmmSYS_longcount = xmmSYS_longcount_tsc;
  }
  else
  {
#endif
	cpu_freq = xmmCPU_clockfreq();
	xmmSYS_longcount = xmmSYS_longcount_sys;
#ifdef HAVE_CPU_IX86
  }
#endif

  return xmmSYS_longcount();
}
