/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * timer.c
 * Timer code
 */

#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include "libxmm/util/timer.h"

/*
 * Definitions
 */

#define TIMER_MAXNUM		5

/*
 * Types
 */

struct xmm_timer_s
{
    XMM_TimerProc	Handler;
    void		*priv;
};

/*
 * Global data
 */

struct xmm_timer_s	aTimer[TIMER_MAXNUM];

/*
 * Prototypes
 */

static void TimeHandler( int value );

/*
 * Initialize timer
 */

int xmmTimer_Init( int interval )
{
  struct itimerval	itv;
  struct sigaction	sa;

  /* Initialize timer data */
  memset( aTimer, 0, sizeof( struct xmm_timer_s ) * TIMER_MAXNUM );

  /* Set timer interval */
  itv.it_interval.tv_sec = interval / 1000;
  itv.it_interval.tv_usec = ( interval % 1000 ) * 1000;
  itv.it_value.tv_sec = interval / 1000;
  itv.it_value.tv_usec = ( interval % 1000 ) * 1000;

  /* Initialize signal ( SIGALRM ) handler */
  sa.sa_handler = TimeHandler;
  sigemptyset( &sa.sa_mask );
  sa.sa_flags = SA_RESTART;
  sa.sa_restorer = 0;

  /* Set timer */
  setitimer( ITIMER_REAL, &itv, NULL );

  /* set handler */
  if( sigaction( SIGALRM, &sa, NULL ) == -1 )	return -1;

  return 0;
}

/*
 * Deinitialize timer
 */

void xmmTimer_Exit( void )
{
}

/*
 * Add timer
 */

int xmmTimer_Add( XMM_TimerProc Handler, void *priv )
{
  int x;

  for( x = 0; x < TIMER_MAXNUM; x++ )
	if( aTimer[x].Handler == NULL )
	{
		aTimer[x].Handler = Handler;
		aTimer[x].priv = priv;
		break;
	}

  if( x == TIMER_MAXNUM )	return -1;
  return x;
}

/*
 * Remove timer
 */

void xmmTimer_Remove( int timer )
{
  aTimer[timer].Handler = NULL;
}

/*
 * Remove all timers
 */

void xmmTimer_RemoveAll( void )
{
  int x;

  for( x = 0; x < TIMER_MAXNUM; x++ )	aTimer[x].Handler = NULL;
}

/*
 * Internal code ( signal handler )
 */

static void TimeHandler( int value )
{
  int x;

  for( x = 0; x < TIMER_MAXNUM; x++ )
	if( aTimer[x].Handler )	aTimer[x].Handler( aTimer[x].priv );
}
