/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <inttypes.h>

#include "libxmm/xmmp.h"

/*
 * If filename is not an absolute path,
 * prepends the homedir and XMM_DIR_HOME dir
 */
char *xmm_gethome_filename( char *filename )
{
  char *newfn, *home;

  if( filename == NULL )	return NULL;
  if( filename[0] == '/' )	return strdup( filename );

  home = getenv( "HOME" );
  if( home == NULL )	return 0;

  newfn = malloc( strlen( home ) + strlen( "/" XMM_DIR_HOME "/" ) + strlen( filename ) + 1 );
  if( newfn == NULL )	return NULL;

  strcpy( newfn, home );
  strcat( newfn, "/" XMM_DIR_HOME "/" );
  strcat( newfn, filename );

  return newfn;
}

/*
 * If filename is not an absolute path,
 * prepends the system library prefix and the relative subdir
 */
char *xmm_getsys_filename( char *prefix, char *filename )
{
  char *newfn;

  if( filename == NULL )	return NULL;
  if( filename[0] == '/' )	return strdup( filename );

  newfn = malloc( strlen( LIBDIR ) + strlen( prefix ) + strlen( filename ) + 3 );
  if( newfn == NULL )	return NULL;

  strcpy( newfn, LIBDIR );
  strcat( newfn, "/" );
  strcat( newfn, prefix );
  strcat( newfn, "/" );
  strcat( newfn, filename );

  return newfn;
}

/*
 * Functions for logging information
 */

static int xmm_logging_level = 1;

int xmm_logging_get_level( void )
{
  return xmm_logging_level;
}

void xmm_logging_set_level( int level )
{
  xmm_logging_level = level;
}

void xmm_logging( int level, char *format, ... )
{
  va_list	ap;
  FILE		*stream = NULL;
  char		*ptr;

  if( xmm_logging_level >= 1 )
  {
	ptr = xmm_gethome_filename( XMM_FILE_LOG );
	if( ptr )
	{
	    stream = fopen( ptr, "a" );
	    free( ptr );
	}
  }

  va_start( ap, format );
  if( level <= xmm_logging_level )
  {
	vprintf( format, ap );
	fflush( stdout );
  }
  if( stream )  vfprintf( stream, format, ap );
  va_end( ap );

  if( stream )	fclose( stream );
}

/*
 * Like strdup
 */
void *xmm_memdup( void *source, int size )
{
  void *ptr;

  if(( ptr = malloc( size )) == NULL )	return NULL;

  memcpy( ptr, source, size );

  return ptr;
}

/*
 * Like strdup, but allocates some extraspace
 */
void *xmm_memdup_x( void *source, int size, int extra )
{
  void *ptr;

  if(( ptr = malloc( size + extra )) == NULL )	return NULL;

  memcpy( ptr, source, size );
  memset( ptr + size, 0, extra );

  return ptr;
}

/*
 * Create zero-terminated string from the FOURCC, (expects big endian value)
 */
char *xmm_FOURCC_string( uint32_t val )
{
  static char tmp[5]="    ";

  sprintf( tmp, "%c%c%c%c",	(char)( val ) & 0xFF,
				(char)( val >> 8 ) & 0xFF,
                                (char)( val >> 16 ) & 0xFF,
				(char)( val >> 24 ) & 0xFF );

  return tmp;
}
