/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * xmmplay.c
 * Player interface to XMMP
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <sys/time.h>

#include <libxmm/libxmm.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/thread.h>
#include <libxmm/util/mutex.h>
#include <libxmmplay/xmmplay.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#define	DEBUG			1
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif

/* Change 'undef' in 'define' to get sync info */
#ifndef DEBUG_SYNC
#define	DEBUG_SYNC
#endif

/* audio / video thread */
#define	PLAYING_AUDIO		1
#define	PLAYING_VIDEO		2


/*
 * xmmplay object
 */

struct XMM_Play
{
    void			*xmm;
    XMM_FileInfo		fi;
//    XMM_AudioFormat		*af;

    /**/
    XMM_FilterAudioInfo		*fai;

    /* Plugins */
    XMM_PluginFilterAudio	*pSound;
    XMM_PluginGraph		*pGraph;
    XMM_PluginInput		*pInput;
    XMM_PluginCodecVideo	*pVCodec;

    /* Play control */
    int				paused;
    int				playing;

    /* Audio */
    uint32_t			cSample;

    char			*rbuffer;		/* input (o) / audio chain (i) audio buffer */
    int				rbuffersize;
    uint32_t			fragsize;		/* sound (i) audio buffer size */

    /* Video buffer */
    char			*vbuffer;
    int				vbuffersize;
    uint8_t			*image[3];
    uint8_t			*vrbuffer[3];
    char			*vrbuffersize;

    /* Plugin caps */
    uint32_t			mode;

    /* Player thread stuff */
    XMM_Thread			*player_thread_audio;
    XMM_Thread			*player_thread_video;

    /* Mutex */
    XMM_Mutex			*mutex_graph;
    XMM_Mutex			*mutex_ctrl;

    uint32_t			audio_fix_flags;
};

/*
 * Prototypes
 */

static int player_thread_audio( void *arg );
static int player_thread_video( void *arg );
static int player_thread_video_async( void *arg );

/*
 * Init library
 */
XMM_Play *xmmplay_Init( void )
{
  XMM_Play	*xmmplay;

  /* Allocate object */
  if(( xmmplay = malloc( sizeof( XMM_Play ))) == NULL )
  {
	xmm_logging( 1, __FUNCTION__ "() Cannot allocate XMM_Play object\n" );
	return NULL;
  }
  memset( xmmplay, 0, sizeof( XMM_Play ));

  /* Initialize libxmm */
  if(( xmmplay->xmm = xmm_Init()) == NULL )
  {
	return NULL;
  }

  /* Create Mutex */
  xmmplay->mutex_graph = xmmMutex_Create();
  xmmplay->mutex_ctrl = xmmMutex_Create();

  return xmmplay;
}

/*
 * Exit library
 */
void xmmplay_Exit( XMM_Play *xmmplay )
{
  /* Destroy mutex */
  xmmMutex_Destroy( xmmplay->mutex_graph );
  xmmMutex_Destroy( xmmplay->mutex_ctrl );

  /* Exit libxmm */
  xmm_Exit( xmmplay->xmm );

  free( xmmplay );
}

/*
 * Set audio output format
 */
int xmmplay_SetAudioFormat( XMM_Play *xmmplay, XMM_FilterAudioInfo *fai, int afidx )
{

  /* Free old information */
  if( xmmplay->fai )	free( xmmplay->fai );
  xmmplay->fai = NULL;

  /* Autoselection */
  if(( fai == NULL ) || ( afidx == -1 ))	return XMM_RET_OK;

  /* Duplicate audio filter info */
  if(( xmmplay->fai = xmm_memdup_x( fai, sizeof( XMM_FilterAudioInfo ), sizeof( XMM_AudioFormat ))) == NULL )
	return xmm_SetError( xmmplay->xmm, XMM_RET_ALLOC, __FUNCTION__ "() Unable to duplicate audio filter info" );

  xmmplay->fai->nfmt = 1;
  xmmplay->fai->fmt = (XMM_AudioFormat *) &xmmplay->fai[1];
  memcpy( xmmplay->fai->fmt, &fai->fmt[afidx], sizeof( XMM_AudioFormat ));

  return XMM_RET_OK;
}

/*
 * Open file
 */
int xmmplay_Open( XMM_Play *xmmplay, char *filename, int flags )
{
  uint32_t		iformat, gformat, yflip, csize;
  int			ret;
  XMM_VideoFormat	*vf;

  int			afidx = -1;
  XMM_AudioFormat	*iaf, *coaf;	/* input (o) codec (o) */
  XMM_FilterAudioInfo	*fai;
  XMM_PluginFilterAudio	*pSound;

  /* Open File */
  xmmplay->pInput = xmm_InputOpen( xmmplay->xmm, filename, flags, &xmmplay->mode );
  if( xmmplay->pInput == NULL )	return XMM_RET_ERROR;

  /* Initialize Audio */
  xmmplay->fi.astreams = xmm_InputAudioStreams( xmmplay->pInput );
  if( xmmplay->fi.astreams )
  {
    /* Get audio stream information */
    xmm_InputAudioInfo( xmmplay->pInput, 0, &xmmplay->fi.ai[0] );
    if( xmmplay->fi.ai[0].fmt.desc[0] == '\0' )
	    xmm_aformat_codec_desc( xmmplay->fi.ai[0].fmt.format, xmmplay->fi.ai[0].fmt.desc );

    if(( xmmplay->mode & XMM_INPUT_CF_MODE_READ ) && (( flags & XMMPLAY_OFLAG_AUDIO_OFF ) == 0 ))
    {
	xmm_InputControl( xmmplay->pInput, XMM_CTLGET_AFORMAT_PTR, 0, &iaf );

	/* Initialize sound output */
	pSound = (XMM_PluginFilterAudio *)xmm_PluginRegister((char *)xmm_AcSoundPluginName( xmmplay->xmm, NULL ));
	if( pSound == NULL )	
	    return xmm_SetError( xmmplay->xmm, XMM_RET_ERROR, __FUNCTION__ "() ERROR: Unable to register '%s'", xmm_AcSoundPluginName( xmmplay->xmm, NULL ));

	/* Get audio formats */
	ret = xmm_FilterAudioControl( pSound, XMM_CTLGET_FILTER_INFO, 0, &fai );
	if( ret < XMM_RET_OK )	return XMM_RET_ERROR;

	/* format selected manual */
	if( xmmplay->fai != NULL )
	{
		fai = xmmplay->fai;
		afidx = 0;
	}

	/* Prepare audio filter chain */
	ret = xmm_FilterAChainPrepare( xmmplay->xmm, iaf, fai, afidx, &coaf );
	if( ret < XMM_RET_OK )	return XMM_RET_ERROR;

	/* Check */
	ret = xmm_FilterAudioControl( pSound, XMM_CTLQUERY_AFORMAT, 0, coaf );
	if( ret < 0 )	xmm_logging( 2, __FUNCTION__ "() WARNING: XMM_CTLQUERY_AFORMAT failed (%i) [%x, %i Hz, %i] )\n", ret, coaf->format, coaf->samprate, coaf->channels );

	/* Init sound output */
	xmmplay->pSound = xmm_FilterAudioOpen( pSound, coaf, NULL, 0 );
	if( xmmplay->pSound == NULL )	return XMM_RET_ERROR;

	/* Get buffer size */
	ret = xmm_FilterAudioControl( xmmplay->pSound, XMM_CTLGET_BUFFER_SIZE, 0, &xmmplay->fragsize );
	if( ret < 0 )	return XMM_RET_ERROR;

	/* Start audio filter chain */
	ret = xmm_FilterAChainStart( xmmplay->xmm, xmmplay->fragsize );
	if( ret < XMM_RET_OK )	return XMM_RET_ERROR;

	/**/
	xmmplay->rbuffer = NULL;
	xmmplay->rbuffersize = -1;
	xmmplay->cSample = 0;
    }
  }

  /* Initialize Video */
  xmmplay->fi.vstreams = xmm_InputVideoStreams( xmmplay->pInput );
  if( xmmplay->fi.vstreams )
  {
    /* Get video stream information */
    xmm_InputVideoInfo( xmmplay->pInput, 0, &xmmplay->fi.vi[0], NULL );

    if( xmmplay->mode & XMM_INPUT_CF_MODE_READ && (( flags & XMMPLAY_OFLAG_VIDEO_OFF ) == 0 ))
    {
	xmm_InputControl( xmmplay->pInput, XMM_CTLGET_GFORMAT_PTR, 0, &vf );

	/* Initialize video codec */
	xmmplay->pVCodec = (XMM_PluginCodecVideo *)xmm_CodecVideoOpen( xmmplay->xmm, XMM_CODEC_MODE_DECODE, vf );
	if( xmmplay->pVCodec == NULL )	return XMM_RET_ERROR;

	strcpy( xmmplay->fi.vi[0].fmt.desc, vf->desc );

	/* Initialize graph output */
	xmmplay->pGraph = xmm_GraphOpen( xmmplay->xmm, (char *)xmm_AcGraphPluginName( xmmplay->xmm, NULL ));
	if( xmmplay->pGraph == NULL )	return XMM_RET_ERROR;

	/* Find format */
	xmm_GraphControl( xmmplay->pGraph, XMM_CTLGET_GFORMAT, 0, &gformat );
	ret = xmmplay->pVCodec->Control( xmmplay->pVCodec, XMM_CTLQUERY_GFORMAT, (void *)gformat );
	if( ret != XMM_CTLRET_TRUE )
	{
		ret = xmmplay->pVCodec->Control( xmmplay->pVCodec, XMM_CTLGET_GFORMAT, &iformat );
		gformat = iformat;
	}
	else	iformat = gformat;

	/* Need Y - Flipping ? */
	ret = xmmplay->pVCodec->Control( xmmplay->pVCodec, XMM_CTLQUERY_YFLIP, &yflip );
	if( ret != XMM_CTLRET_ARG )	yflip = 0;

	/* Init graph output */
	if( xmm_GraphStart( xmmplay->pGraph, xmmplay->fi.vi[0].fmt.width,
					xmmplay->fi.vi[0].fmt.height, gformat,
					yflip ? XMM_GRAPH_FLAG_YFLIP : 0 ) < 0 )
						return XMM_RET_ERROR;

	/* Allocate video buffer */
	xmmplay->vbuffersize = xmmplay->fi.vi[0].fmt.width * xmmplay->fi.vi[0].fmt.height * 4;
	if(( xmmplay->vbuffer = malloc( xmmplay->vbuffersize )) == NULL )
	{
	    xmm_GraphClose( xmmplay->pGraph );
	    return xmm_SetError( xmmplay->xmm, XMM_RET_ALLOC, __FUNCTION__ "() Video buffer" );
	}

	/* Set frame rate */
	xmm_GraphControl( xmmplay->pGraph, XMM_CTLSET_FRAMERATE, (uint32_t) xmmplay->fi.vi[0].fmt.framerate, NULL );

	/* Set input format */
	ret = xmmplay->pVCodec->Control( xmmplay->pVCodec, XMM_CTLSET_GFORMAT, (void *)iformat );
	if( ret == XMM_CTLRET_FALSE )
	{
	    xmm_GraphClose( xmmplay->pGraph );
	    return xmm_SetError( xmmplay->xmm, XMM_RET_ERROR, __FUNCTION__ "() Format %s ( 0x%x ) not supported by codec", xmm_FOURCC_string( iformat ), iformat );
	}

	/* Set image pointer(s) */
	xmmplay->image[0] = xmmplay->vbuffer;

	if( gformat == xmmFOURCC( 'Y','V','1','2' ))
	{
		csize = xmmplay->fi.vi[0].fmt.width * xmmplay->fi.vi[0].fmt.height / 4;
		xmmplay->image[1] = xmmplay->vbuffer + csize * 4;
		xmmplay->image[2] = xmmplay->vbuffer + csize * 5;
	}

	/* video read buffers */
	if( xmmplay->vrbuffer[0] == NULL )
		xmmplay->vrbuffer[0] = malloc( 1000000 );
    }
  }

  return XMM_RET_OK;
}

/*
 * Close file
 */
int xmmplay_Close( XMM_Play *xmmplay )
{
  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  /* Stop playback */
  xmmplay_Stop( xmmplay );

  /* Close input plugin */
  if( xmmplay->pInput )	xmm_InputClose( xmmplay->pInput );
  xmmplay->pInput = NULL;

  /* Close sound plugin */
  if( xmmplay->pSound )	xmm_FilterAudioCloseFree( xmmplay->pSound );
  xmmplay->pSound = NULL;

  /* Stop audio chain */
  xmm_FilterAChainStop( xmmplay->xmm );

  /* Close graph plugin */
  if( xmmplay->pGraph )	xmm_GraphClose( xmmplay->pGraph );
  xmmplay->pGraph = NULL;

  /* Codec */
  if( xmmplay->pVCodec )	xmmplay->pVCodec->Close( xmmplay->pVCodec );
  xmmplay->pVCodec = NULL;

  /* Audio and read buffer */
  if( xmmplay->rbuffer )	free( xmmplay->rbuffer );
  xmmplay->rbuffer = NULL;

  return XMM_RET_OK;
}

/*
 * Play control
 */
int xmmplay_Play( XMM_Play *xmmplay )
{
  int		ret;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	if( xmmplay->playing )
	{
	    xmm_SetError( xmmplay->xmm, XMM_RET_ERROR, __FUNCTION__ "() Already playing." );
	    return XMM_RET_ERROR;
	}

	xmmMutex_Lock( xmmplay->mutex_ctrl );

	xmmplay->playing = 0;
	xmmplay->paused = 0;

#if 0
	if( xmmplay->pSound )
#else
	if( xmmplay->fi.astreams > 0 )
#endif
	{
	    xmmplay->playing |= PLAYING_AUDIO;
	    xmmplay->player_thread_audio = xmmThread_Create( player_thread_audio, (void *)xmmplay );
	}

#if 0
	if( xmmplay->pGraph )
#else
	if( xmmplay->fi.vstreams > 0 )
#endif
	{
	    xmmplay->playing |= PLAYING_VIDEO;

#if 1
#if 0
	    if( xmmplay->pSound )
#else
	    if( xmmplay->fi.astreams > 0 )
#endif
		xmmplay->player_thread_video = xmmThread_Create( player_thread_video_async, (void *)xmmplay );
	    else
#endif
		xmmplay->player_thread_video = xmmThread_Create( player_thread_video, (void *)xmmplay );
	}

	xmmMutex_Unlock( xmmplay->mutex_ctrl );
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_PLAY, 0, NULL );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Stop
 */
int xmmplay_Stop( XMM_Play *xmmplay )
{
  int	ret;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	xmmMutex_Lock( xmmplay->mutex_ctrl );
	xmmplay->paused = 0;
	xmmplay->playing = 0;
	xmmMutex_Unlock( xmmplay->mutex_ctrl );
#if 1
	/* Stop player thread */
	if( xmmplay->player_thread_audio )
	{
		xmmThread_Wait( xmmplay->player_thread_audio );
		xmmplay->player_thread_audio = NULL;
	}

	if( xmmplay->player_thread_video )
	{
		xmmThread_Wait( xmmplay->player_thread_video );
		xmmplay->player_thread_audio = NULL;
	}
#endif

	xmmplay->pInput->Seek( xmmplay->pInput, 0, 0, 0.0 );
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_STOP, 0, NULL );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Pause ( toggle )
 */
int xmmplay_Pause( XMM_Play *xmmplay )
{
  int		ret;
  uint32_t	paused;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
#if 0
	if(( xmmplay->playing == 0 ) && ( xmmplay->paused == 0 ))
	{
	    xmm_SetError( xmmplay->xmm, XMM_RET_ERROR, __FUNCTION__ "() Playback not paused / not playing." );
	    return XMM_RET_ERROR;
	}
#endif

	xmm_SetError( xmmplay->xmm, XMM_RET_ERROR, __FUNCTION__ "()" );
	return XMM_RET_NOTSUPPORTED;
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_PAUSE, 0, &paused );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
	return paused;
  }
}

/*
 * Get play status
 */
int xmmplay_Status( XMM_Play *xmmplay )
{
  int		ret;
  uint32_t	status;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	if( xmmplay->playing )	ret = XMM_PBS_PLAYING;
	else	ret = XMM_PBS_STOPPED;

	if(( ret & XMM_PBS_STOPPED ) && xmmplay->paused )
		ret = ( ret & ~XMM_PBS_STOPPED ) | XMM_PBS_PAUSED;

	return ret;
  }
  else
  {
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLINP_STATUS, 0, &status );
	if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;
	return (int)status;
  }
}

/*
 * Seek
 */
int xmmplay_Seek( XMM_Play *xmmplay, double percent )
{
  int status;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	status = xmmplay->playing;
	if( status )	xmmplay_Stop( xmmplay );

	xmmplay->pInput->Seek( xmmplay->pInput, 0, 0, percent );

	/* Empty audio buffer */
	if( xmmplay->pSound )	xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLSET_FLUSH, 0, NULL );

	if( status )	xmmplay_Play( xmmplay );

	return XMM_RET_OK;
  }
  else
  {
	return xmmplay->pInput->Seek( xmmplay->pInput, 0, 0, percent );
  }
}

/*
 * Scale
 */
int xmmplay_Scale( XMM_Play *xmmplay, int width, int height, int flags )
{
  XMM_ControlScale	scale;
  int			ret;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  /* If no video, we return OK */
  if( xmmplay->fi.vstreams == 0 )	return XMM_RET_OK;

  /* Scale */
  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
	if( xmmplay->pGraph == NULL )	return XMM_RET_ERROR;

	xmmMutex_Lock( xmmplay->mutex_graph );

	/* fill XMM_ControlScale structure */
	scale.width = width;
	scale.height = height;
	scale.flags = flags;

	/* Resize using graph plugin */
	ret = xmmplay->pGraph->Control( xmmplay->pGraph, XMM_CTLSET_SCALE, 0, &scale );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;

	xmmMutex_Unlock( xmmplay->mutex_graph );
  }
  else
  {
	/* fill XMM_ControlScale structure */
	scale.width = width;
	scale.height = height;
	scale.flags = flags;

	/* Resize using input plugin */
	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLSET_SCALE, 0, &scale );
	if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;
  }

  return XMM_RET_OK;
}

/*
 * Status & Info
 */
uint32_t xmmplay_Info( XMM_Play *xmmplay, XMM_FileInfo *fi, XMM_ClipInfo *ci, double *percent )
{
  double	delay = 0, seek = 0.0;
  uint32_t	pts = 0;
  XMM_ClipInfo	lci;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if( fi )	memcpy( fi, &xmmplay->fi, sizeof( XMM_FileInfo ));

  if( ci == NULL )	ci = &lci;		/* We need playtime */

  /* clipinfo, avidiff, seek */
  xmmplay->pInput->Info( xmmplay->pInput, ci, &seek );

  /* timestamps */
  if( xmmplay->pSound != NULL )
  {
	xmmplay->pInput->AudioPTS( xmmplay->pInput, 0, &pts );

	if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
	    xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_DELAY, 0, &delay );

	pts -= (uint32_t)( delay * 1000 );
  }
  else	if( xmmplay->player_thread_video != NULL )
  		xmmplay->pInput->VideoPTS( xmmplay->pInput, 0, &pts );

  if( percent )			/* calculate percent value */
  {
	*percent = seek;

	if(( seek == 0.0 ) && ( ci != NULL ))
	    if( ci->playtime != 0 )
		*percent = ((double)pts / 1000 ) / ci->playtime;
  }

  return pts;
}


/*
 * Set Volume
 */
int xmmplay_SetVolume( XMM_Play *xmmplay, int left, int right )
{
  uint32_t	volume = ( right << 16 ) | left;
  int		ret = XMM_CTLRET_FALSE;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
    if( xmmplay->pSound )
	 ret = xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLSET_VOLUME, volume, NULL );
  }
  else	 ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLSET_VOLUME, volume, NULL );

  if( ret != XMM_CTLRET_TRUE )	return XMM_RET_ERROR;

  return XMM_RET_OK;
}

/*
 * Get Volume
 */
int xmmplay_GetVolume( XMM_Play *xmmplay, int *left, int *right )
{
  uint32_t	volume;
  int		ret = XMM_CTLRET_FALSE;

  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  if(( xmmplay->mode & XMM_INPUT_CF_MODE_DOUT ) == 0 )
  {
    if( xmmplay->pSound )
	ret = xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_VOLUME, 0, &volume );
  }
  else	ret = xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLGET_VOLUME, 0, &volume );

  if( ret != XMM_CTLRET_ARG )	return XMM_RET_ERROR;

  *left = volume & 0xFFFF;
  *right = volume >> 16;

  return XMM_RET_OK;
}

/*
 * Input plugin dialogues
 */
int xmmplay_InputDialogue( XMM_Play *xmmplay, int dialog )
{
  /* Check arguments */
  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  /* */
  xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLDLG_DISPLAY, dialog, NULL );

  return XMM_RET_OK;
}

/*
 * Set input clip info
 */
int xmmplay_InputSetClipInfo( XMM_Play *xmmplay, XMM_ClipInfo *ci )
{
  /* Check arguments */
  if( xmmplay->pInput == NULL )
  {
	xmm_SetError( xmmplay->xmm, XMM_RET_NULL, __FUNCTION__ "() File not opened." );
	return XMM_RET_ERROR;
  }

  /* */
  return xmmplay->pInput->Control( xmmplay->pInput, XMM_CTLSET_CLIPINFO, 0, ci );
}


/*
 * Internal code ( player threads )
 */

#include <libxmm/util/system.h>
#include <libxmm/util/mmaccel.h>

/*
 * Player thread ( audio only )
 */
static int player_thread_audio( void *arg )
{
  XMM_Play		*xmmplay = (XMM_Play *) arg;
  XMM_PluginInput	*pInput	= xmmplay->pInput;
  XMM_PluginFilterAudio	*pSound = xmmplay->pSound;
  int			read, ret;
  uint32_t		total, avail, samples, decoded, need;
  uint8_t		*ptr;

  while( xmmplay->playing & PLAYING_AUDIO )
  {
	xmm_FilterAChainSourceSize( xmmplay->xmm, xmmplay->fragsize, &need );

#if DEBUG == 2
	xmm_logging( 1, __FUNCTION__ "() %i bytes input needed for %i bytes output\n", need, xmmplay->fragsize );
#endif

	if(( xmmplay->rbuffersize < need ) && xmmplay->rbuffer )
	{
	    free( xmmplay->rbuffer );
	    xmmplay->rbuffer = NULL;
	}

	if( xmmplay->rbuffer == NULL )
	{
	    xmmplay->rbuffersize = need;
	    if(( xmmplay->rbuffer = malloc( xmmplay->rbuffersize )) == NULL )
	    {
		xmmMutex_Lock( xmmplay->mutex_ctrl );
		xmmplay->playing &= ~PLAYING_AUDIO;
		xmmMutex_Unlock( xmmplay->mutex_ctrl );

		return xmm_SetError( pInput->sys.xmm, XMM_RET_ALLOC, __FUNCTION__"() audio buffer. (%i bytes)", xmmplay->rbuffersize );
	    }
	}

	read = pInput->AudioRead( pInput, 0, xmmplay->rbuffer, need, 0 );

#if DEBUG == 2
	xmm_logging( 1, __FUNCTION__ "() AudioRead() returned %i ( %i bytes needed )\n", read, need );
#endif

	if( read == XMM_RET_EOS )
	{
	    do
	    {
	    	xmmSYS_usleep( 1000 );

		xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_BUFFER_FREE, 0, &avail );
		xmmplay->pSound->Control( xmmplay->pSound, XMM_CTLGET_BUFFER_TOTAL, 0, &total );
	    } while( avail != total );

	    xmmMutex_Lock( xmmplay->mutex_ctrl );
	    xmmplay->playing &= ~PLAYING_AUDIO;
	    xmmMutex_Unlock( xmmplay->mutex_ctrl );
	}
	
	if( read >= XMM_RET_OK )
	{
	    decoded = xmmplay->fragsize;
	    ret = xmm_FilterAChain( xmmplay->xmm, xmmplay->rbuffer, read, &ptr, &decoded, NULL, &samples );
	    if( ret == XMM_RET_OK )
	    {
#if DEBUG == 2
		xmm_logging( 1, __FUNCTION__ "() FilterAChain() %i -> %i\n", read, decoded );
#endif
		pSound->Process( pSound, ptr, decoded, NULL, NULL, NULL );
		xmmplay->cSample += samples;
	    }
	}
  }

  xmmplay->player_thread_audio = NULL;
  return XMM_RET_OK;
}

/*
 * Player thread ( video, synchronize to system timer )
 */
static int player_thread_video( void *arg )
{
  XMM_Play		*xmmplay = (XMM_Play *) arg;
  XMM_PluginInput	*pInput	= xmmplay->pInput;
  XMM_PluginGraph	*pGraph = xmmplay->pGraph;
  uint64_t		baseTime, currTime;
  uint32_t		videoPTS = 0, freq, dsize, cf;
  int32_t		timeDiff;
  int			read;
#ifdef DEBUG_SYNC
  uint32_t		dropped = 0, frame;
#endif

  freq = xmmCPU_clockfreq() / 1000;	/* kHz */
  baseTime = xmmSYS_longcount() / freq;	/* ms */

  pInput->VideoPTS( pInput, 0, &videoPTS );
  baseTime -= videoPTS;

  /* Main loop */
  while( xmmplay->playing & PLAYING_VIDEO )
  {
	/*
	 * Video synchronization
	 */

	pInput->VideoPTS( pInput, 0, &videoPTS );
	currTime = xmmSYS_longcount() / freq; /* ms */
	timeDiff = videoPTS - (uint32_t)( currTime - baseTime ); /* ms */

#ifdef DEBUG_SYNC
	xmm_logging( 1, "\t\t\tvPTS: %i realPTS: %i async: %i   \r",
					videoPTS,
					(int32_t)( currTime - baseTime ),
					timeDiff );
#endif

	/* video to fast: sleep */
	if( timeDiff > 10 )	xmmSYS_usleep(( timeDiff * 1000 ) >> 1 );

	/* video to slow: drop frame */
	if( timeDiff < -150 )
	{
		read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 1 );

		cf = XMM_CODEC_DF_DROP;
		dsize = xmmplay->vbuffersize;
		xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

#ifdef DEBUG_SYNC
		dropped++;
#endif
	}

	/* video very slow: drop one more frame */
	if( timeDiff < -230 )
	{
		read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 1 );

		cf = XMM_CODEC_DF_DROP;
		dsize = xmmplay->vbuffersize;
		xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

#ifdef DEBUG_SYNC
		dropped++;
#endif
	}

	/* Read frame */
	read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 0 );
	if( read == XMM_RET_EOS )
	{
	    xmmMutex_Lock( xmmplay->mutex_ctrl );
	    xmmplay->playing &= ~PLAYING_VIDEO;
	    xmmMutex_Unlock( xmmplay->mutex_ctrl );
	}
	if( read >= XMM_RET_OK )
	{
	    /* Decode frame */
	    cf = 0;
	    dsize = xmmplay->vbuffersize;
	    xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

	    xmmMutex_Lock( xmmplay->mutex_graph );
	    pGraph->Draw( pGraph, xmmplay->image, NULL, 0, 0, xmmplay->fi.vi[0].fmt.width, xmmplay->fi.vi[0].fmt.height, 0 );
	    pGraph->Blit( pGraph );
	    xmmMutex_Unlock( xmmplay->mutex_graph );
	}
  }

#ifdef DEBUG_SYNC
  pInput->VideoInfo( pInput, 0, NULL, &frame );
  xmm_logging( 1, "INFO! Dropped %i/%i frames\n", dropped, frame );
#endif

  xmmplay->player_thread_video = NULL;
  return XMM_RET_OK;
}

/*
 * Player thread ( video, sync to audio, to system timer if no audio playing )
 */
static int player_thread_video_async( void *arg )
{
  XMM_Play		*xmmplay = (XMM_Play *) arg;
  XMM_PluginInput	*pInput	= xmmplay->pInput;
  XMM_PluginGraph	*pGraph = xmmplay->pGraph;
  XMM_PluginFilterAudio	*pSound = xmmplay->pSound;
  uint64_t		baseTime, currTime;
  uint32_t		videoPTS = 0, audioPTS = 0, dsize, cf, freq;
  int32_t		timeDiff;
  double		sndDelay = 0;
  int			read;
#ifdef DEBUG_SYNC
  uint32_t		dropped = 0, frame;
#endif


//  freq = xmmCPU_clockfreq() / 1000;	/* kHz */
//  baseTime = xmmSYS_longcount() / freq;	/* ms */

//  pInput->VideoPTS( pInput, 0, &videoPTS );
//  baseTime -= videoPTS;

  /* Main loop */
  while( xmmplay->playing & PLAYING_VIDEO )
  {
	/*
	 * Video synchronization
	 */

	pInput->VideoPTS( pInput, 0, &videoPTS );

#if 0
	if( xmmplay->playing & PLAYING_AUDIO )
	{
#endif
	    pInput->AudioPTS( pInput, 0, &audioPTS );

	    pSound->Control( pSound, XMM_CTLGET_DELAY, 0, &sndDelay );	/* sec */
	    timeDiff = videoPTS - ( audioPTS - (uint32_t)( sndDelay * 1000 ));

#ifdef DEBUG_SYNC
	    xmm_logging( 1, "\t\t\tvPTS: %i aPTS: %i async: %i sd: %.3fs   \r",
				videoPTS, audioPTS, timeDiff, sndDelay );
#endif

#if 0
	}
	else
	{
	    currTime = xmmSYS_longcount() / freq; /* ms */
	    timeDiff = videoPTS - (uint32_t)( currTime - baseTime ); /* ms */

#ifdef DEBUG_SYNC
	    xmm_logging( 1, "\t\t\tvPTS: %i realPTS: %i async: %i   \r",
					videoPTS,
					(int32_t)( currTime - baseTime ),
					timeDiff );
#endif
	}
#endif

	/* video to fast: sleep */
	if( timeDiff > 10 )	xmmSYS_usleep(( timeDiff * 1000 ) >> 1 );

	/* video to slow: drop frame */
	if( timeDiff < -150 )
	{
		read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 1 );

		cf = XMM_CODEC_DF_DROP;
		dsize = xmmplay->vbuffersize;
		xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

#ifdef DEBUG_SYNC
		dropped++;
#endif
	}

	/* video very slow: drop one more frame */
	if( timeDiff < -230 )
	{
		read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 1 );

		cf = XMM_CODEC_DF_DROP;
		dsize = xmmplay->vbuffersize;
		xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

#ifdef DEBUG_SYNC
		dropped++;
#endif
	}

	/* Read frame */
	read = pInput->VideoRead( pInput, 0, xmmplay->vrbuffer, 0 );
	if( read == XMM_RET_EOS )
	{
	    xmmMutex_Lock( xmmplay->mutex_ctrl );
	    xmmplay->playing &= ~PLAYING_VIDEO;
	    xmmMutex_Unlock( xmmplay->mutex_ctrl );
	}
	if( read >= XMM_RET_OK )
	{
	    /* Decode frame */
	    cf = 0;
	    dsize = xmmplay->vbuffersize;
	    xmmplay->pVCodec->Decode( xmmplay->pVCodec, xmmplay->vrbuffer[0], read, xmmplay->image, &dsize, &cf );

	    xmmMutex_Lock( xmmplay->mutex_graph );
	    pGraph->Draw( pGraph, xmmplay->image, NULL, 0, 0, xmmplay->fi.vi[0].fmt.width, xmmplay->fi.vi[0].fmt.height, 0 );
	    pGraph->Blit( pGraph );
	    xmmMutex_Unlock( xmmplay->mutex_graph );
	}
  }

#ifdef DEBUG_SYNC
  pInput->VideoInfo( pInput, 0, NULL, &frame );
  xmm_logging( 1, "INFO! Dropped %i/%i frames\n", dropped, frame );
#endif

  xmmplay->player_thread_video = NULL;
  return XMM_RET_OK;
}
