/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * divx.c
 * Plugin for DivX Codec 4.0
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

#include "config.h"
#include <decore.h>

/*
 * Definitions
 */

	/* decore handle */
#define MY_APP_ID		0x12345
#define	DIVX_POSTPROC		0
#define	DIVX_FASTEST		xmmFOURCC( 'Y','U','Y','2' )

/*
 * Types
 */

struct priv_t
{
    DEC_PARAM			dec_param;
    int				dec_opt_frame;
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	divx_cvi =
{
    XMM_CODEC_VCF_DECODE,
    "DivX Codec 4.0",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( 'D','I','V','X' ),	/* FOURCC */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Prototypes
 */

static uint32_t divx_format( uint32_t format );

/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *divx_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;
  DEC_SET		dec_set;
  int			codec_version = 0;
  char			*codec_info;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check codec */
  switch( vf->codec )
  {
		/* DivX Codec 4.0 */
	case	xmmFOURCC( 'D','I','V','X' ):		/* Tool_Sober.avi: noisy left and right margin */
			codec_info = "DivX 4.0";
			codec_version = 400;
			break;

#ifdef HAVE_DIVX4LINUX50
		/* DivX Codec 5.0 ( and ISO-MPEG ) */
	case	xmmFOURCC( 'D','X','5','0' ):
			codec_info = "DivX 5.0";
			codec_version = 500;
			break;
#endif

		/* OpenDivX */
	case	xmmFOURCC( 'd','v','x','1' ):		/* TODO: not correct */
			codec_info = "OpenDivX";
			codec_version = 500;
			break;

		/* MS MPEG-4 Low-Motion ( DivX ;-) 3.11 ) */
	case	xmmFOURCC( 'D','I','V','3' ):
			codec_info = "DivX MPEG-4 Low-Motion ( Microsoft MPEG-4 V3 )";
			codec_version = 311;
			break;

		/* MS MPEG-4 Fast-Motion ( DivX ;-) 3.11 ), TODO: TEST */
	case	xmmFOURCC( 'D','I','V','4' ):
			codec_info = "DivX MPEG-4 Fast-Motion ( Microsoft MPEG-4 V3 )";
			codec_version = 311;
			break;

		/* MS High-Speed MPEG-4 V3 */
	case	xmmFOURCC( 'M','P','4','3' ):
	case	xmmFOURCC( 'M','P','G','3' ):
			codec_info = "MS High-Speed MPEG-4 V3";
			codec_version = 311;
			break;

		/* MS High-Speed MPEG-4 V1/V2, TODO: TEST */
	case	xmmFOURCC( 'M','P','4','1' ):
	case	xmmFOURCC( 'M','P','4','2' ):
			codec_info = "MS High-Speed MPEG-4 V1/V2";
			codec_version = 311;
			break;

	default:	return (void *)XMM_RET_NOTSUPPORTED;
  }

  /* Only query codec  */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(DivX) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Set codec description */
  if( mode & XMM_CODEC_MODE_DECODE )
	strcpy( vf->desc, codec_info );

  /* Initialize decoding */
  priv->dec_param.x_dim = vf->width;
  priv->dec_param.y_dim = vf->height;
#ifdef HAVE_DIVX4LINUX50
  priv->dec_param.codec_version = codec_version;
  priv->dec_param.build_number = 0;
#endif


  xmm_logging( 2, "DIVX! decore version %i\n", decore( MY_APP_ID, DEC_OPT_VERSION, NULL, NULL ));
  xmm_logging( 2, "DIVX! using codec_version = %i\n", codec_version );

  /* DEC_OPT_INIT may only be called once, format set by XMM_CTLSET_GFORMAT */
#if 0
  priv->dec_param.output_format = divx_format( DIVX_FASTEST );
  decore( MY_APP_ID, DEC_OPT_INIT, &priv->dec_param, NULL );
#endif

  /* Postprocessing */
  dec_set.postproc_level = DIVX_POSTPROC;
  decore( MY_APP_ID, DEC_OPT_SETPP, &dec_set, NULL );

#ifndef HAVE_DIVX4LINUX50
  /* DivX ;-) 3.11 decoding */
  if( codec_version == 311 )	priv->dec_opt_frame = DEC_OPT_FRAME_311;
  else
#endif
  	priv->dec_opt_frame = DEC_OPT_FRAME;

  return pCodec;
}

/*
 * Free codec
 */
static int divx_Close( XMM_PluginCodecVideo *codec )
{
  /* release */
  decore( MY_APP_ID, DEC_OPT_RELEASE, NULL, NULL );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int divx_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  struct priv_t		*priv = codec->sys.priv;
  uint32_t		format;

  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if( divx_format((uint32_t)arg ) != 0 )	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = 0;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    *((uint32_t *)arg) = DIVX_FASTEST;
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_SIZE:
	    return XMM_RET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLGET_FORMAT_DATA:
	    return XMM_RET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLSET_GFORMAT:
	    if(( format = divx_format( (uint32_t)arg )) == 0 )
		return XMM_CTLRET_FALSE;

	    priv->dec_param.output_format = format;
	    decore( MY_APP_ID, DEC_OPT_INIT, &priv->dec_param, NULL );

	    return XMM_CTLRET_TRUE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )
	return xmm_SetError( codec->sys.xmm, XMM_RET_NOTSUPPORTED, "(DivX) cmd = 0x%x" );

  return xmm_SetError( codec->sys.xmm, XMM_RET_INVALID_ARG, "(DivX) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int divx_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t	*priv = codec->sys.priv;
  DEC_FRAME	dec_frame;

  dec_frame.length	= isize;
  dec_frame.bitstream 	= src;
  dec_frame.bmp		= (void *) dest[0];
  dec_frame.render_flag	= ( *flags & XMM_CODEC_DF_DROP ) ? 0 : 1;
  dec_frame.stride	= priv->dec_param.x_dim;

  decore( MY_APP_ID, priv->dec_opt_frame, &dec_frame, NULL );

  return isize;
}

/*
 * Encode data
 */
static int divx_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *divx_Info( void *xmm )
{
  return &divx_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"DivX",
				"Codec: DivX Codec 4.0/5.0 [Decoding]",
				"Copyright (c) 2001 Arthur Kleer",
				NULL, NULL },
				divx_Open, divx_Close, divx_Control,
				divx_Decode, divx_Encode, divx_Info };

/*
 * Private code
 */

static uint32_t divx_format( uint32_t format )
{
  uint32_t divx_format = 0;

  switch( format )
  {
	/** planar: Y V U */
	case XMM_GRAPH_FMT_YV12:	divx_format = DEC_YV12;
					break;
	/** planar: Y U V */
	case XMM_GRAPH_FMT_IYUV:	divx_format = DEC_420;
					break;
	/** packed: Y0 U0 Y1 V0 */
	case XMM_GRAPH_FMT_YUY2:	divx_format = DEC_YUY2;
					break;
	/** packed: U0 Y0 V0 Y1 */
	case XMM_GRAPH_FMT_UYVY:	divx_format = DEC_UYVY;
					break;

	/** RGB with depth of bpp */
	case XMM_GRAPH_FMT_RGB(32):	divx_format = DEC_RGB32;
					break;
	case XMM_GRAPH_FMT_RGB(24):	divx_format = DEC_RGB24;
					break;
	case XMM_GRAPH_FMT_RGB(16):	divx_format = DEC_RGB565;
					break;
	case XMM_GRAPH_FMT_RGB(15):	divx_format = DEC_RGB555;
					break;

	/** BGR with depth of bpp */
	case XMM_GRAPH_FMT_BGR(32):	divx_format = DEC_RGB32_INV;
					break;
	case XMM_GRAPH_FMT_BGR(24):	divx_format = DEC_RGB24_INV;
					break;
	case XMM_GRAPH_FMT_BGR(16):	divx_format = DEC_RGB555_INV;
					break;
	case XMM_GRAPH_FMT_BGR(15):	divx_format = DEC_RGB565_INV;
					break;
  }

  return divx_format;
}
