/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mjpeg.c
 * Video codec plugin for Motion-JPEG
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

#include <setjmp.h>
#include <jpeglib.h>
#include <jerror.h>

/*
 * Definitions
 */

#define	MJPEG_GFORMAT			xmmFOURCC( 'Y','V','1','2' )

#define	MJPEG_ILACE_ODD_FIRST		1
#define	MJPEG_ILACE_EVEN_FIRST		2

/*
 * Types
 */

struct priv_t
{
    uint32_t				depth;
    XMM_VideoFormat			vf;

    struct jpeg_decompress_struct	cinfo;
    struct jpeg_source_mgr		jpeg_src;

    struct mjpeg_error_s
    {
	    struct jpeg_error_mgr	pub;
	    jmp_buf			setjmp_buffer;
    } jpeg_err;

    int					ilace_type;
    uint8_t	**yuv_buffer[3];
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	mjpeg_cvi =
{
    XMM_CODEC_VCF_DECODE | XMM_CODEC_VCF_ENCODE,
    "Motion-JPEG",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( 'M','J','P','G' ),	/* FOURCC */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Prototypes
 */

static void	mjpeg_init_source( j_decompress_ptr cinfo );
static boolean	mjpeg_fill_input_buffer( j_decompress_ptr cinfo );
static void	mjpeg_skip_input_data( j_decompress_ptr cinfo, long num_bytes );
static void	mjpeg_term_source( j_decompress_ptr cinfo );

static void	mjpeg_ErrorHandler( j_common_ptr cinfo );

/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *mjpeg_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check codec */
  switch( vf->codec )
  {
	case	xmmFOURCC( 'M','J','P','G' ):
			break;

	default:	return (void *)XMM_RET_NOTSUPPORTED;
  }

  /* Only query codec  */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MJPEG) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Set codec description */
  if( mode & XMM_CODEC_MODE_DECODE )
	strcpy( vf->desc, "Motion-JPEG" );

  /* Get depth for decoding */
  priv->depth = vf->bpp;

  /* Save video format */
  memcpy( &priv->vf, vf, sizeof( XMM_VideoFormat ));
  priv->vf.extraSize = 0;

  /* initialize library */
  priv->cinfo.err = jpeg_std_error( &priv->jpeg_err.pub );
  priv->jpeg_err.pub.error_exit = mjpeg_ErrorHandler;

  if( setjmp( priv->jpeg_err.setjmp_buffer ))
  {
// * * * Here we get when an error occurs * * *
	jpeg_destroy_decompress( &priv->cinfo );
	xmm_SetError( xmm, XMM_RET_ERROR, "MJPEG! Error sent by JPEG lib.\n" );
	return NULL;
  }

  jpeg_create_decompress( &priv->cinfo );
  priv->jpeg_src.init_source = mjpeg_init_source;
  priv->jpeg_src.fill_input_buffer = mjpeg_fill_input_buffer;
  priv->jpeg_src.skip_input_data = mjpeg_skip_input_data;
  priv->jpeg_src.resync_to_restart = jpeg_resync_to_restart;
  priv->jpeg_src.term_source = mjpeg_term_source;
  priv->cinfo.src = &priv->jpeg_src;

  priv->ilace_type = MJPEG_ILACE_ODD_FIRST;

  return pCodec;
}

/*
 * Free codec
 */
static int mjpeg_Close( XMM_PluginCodecVideo *codec )
{
  struct priv_t *priv = codec->sys.priv;

  jpeg_destroy_decompress( &priv->cinfo );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int mjpeg_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if((uint32_t)arg == MJPEG_GFORMAT )	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = 0;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    *((uint32_t *)arg) = MJPEG_GFORMAT;
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_SIZE:
	    return XMM_RET_NOTSUPPORTED;

    case XMM_CTLGET_FORMAT_DATA:
	    return XMM_RET_NOTSUPPORTED;

    case XMM_CTLSET_GFORMAT:
	    return XMM_RET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )
	return xmm_SetError( codec->sys.xmm, XMM_RET_NOTSUPPORTED, "(MJPEG) cmd = 0x%x" );

  return xmm_SetError( codec->sys.xmm, XMM_RET_INVALID_ARG, "(MJPEG) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int mjpeg_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t		*priv = codec->sys.priv;
  int			hsf[3], vsf[3], i, field, cf, numfields, y, c, ywidth, cwidth;
  uint8_t		*yuv_base[3];

  /* start decompressing */
  priv->cinfo.src->bytes_in_buffer = isize;
  priv->cinfo.src->next_input_byte = (JOCTET *) src;
  jpeg_read_header( &priv->cinfo, TRUE );
  priv->cinfo.raw_data_out = TRUE;
  priv->cinfo.out_color_space = JCS_YCbCr;
  priv->cinfo.dct_method = JDCT_IFAST;
  jpeg_start_decompress( &priv->cinfo );

  if( priv->cinfo.output_components != 3 )
	return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "MJPEG! components: %i, expected: 3", priv->cinfo.output_components );

  /* Check sampling factors */
  for( i = 0; i < 3; i++ )
  {
	hsf[i] = priv->cinfo.comp_info[i].h_samp_factor;
	vsf[i] = priv->cinfo.comp_info[i].v_samp_factor;
   }

  if(( hsf[0] != 2 ) || ( hsf[1] != 1 ) || ( hsf[2] != 1 ) ||
    (( vsf[0] != 1 ) && ( vsf[0] != 2 )) || ( vsf[1] != 1 ) || ( vsf[2] != 1 ))
	return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "MJPEG! unsupported sampling factors: (%i:%i:%i) v (%i:%i:%i)", hsf[0], hsf[1], hsf[2], vsf[0], vsf[1], vsf[2] );

  /* Check fields */
  if( priv->cinfo.output_height == priv->vf.height )	numfields = 1;
  else	if(( 2 * priv->cinfo.output_height ) == priv->vf.height )	numfields = 2;
  else	return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "MJPEG! unsupported field height (%i) image height = %i", priv->cinfo.output_height, priv->vf.height );

  /* Interlace type */
  if( numfields == 2 )
  {
	priv->ilace_type = MJPEG_ILACE_ODD_FIRST;
	/* TODO: get (AVI MJPEG) interlace type from APP0 block */
  }

  /* Decode fields */
  for( field = 0; field < numfields; field++ )
  {
	if( field != 0 )
	{
	    jpeg_read_header( &priv->cinfo, TRUE );
	    priv->cinfo.raw_data_out = TRUE;
	    priv->cinfo.out_color_space = JCS_YCbCr;
	    priv->cinfo.dct_method = JDCT_IFAST;
	    jpeg_start_decompress( &priv->cinfo );
	}

	/* Alocate component buffer pointers - */
	if( priv->yuv_buffer[0] == NULL )
	{
	    priv->yuv_buffer[0] = malloc( priv->cinfo.output_height * sizeof( uint8_t * ));
	    priv->yuv_buffer[1] = malloc( priv->cinfo.output_height * sizeof( uint8_t * ));
	    priv->yuv_buffer[2] = malloc( priv->cinfo.output_height * sizeof( uint8_t * ));
	}

	if( numfields == 2 )
	{
	    switch( priv->ilace_type ) 
	    {
        	case MJPEG_ILACE_ODD_FIRST:
			cf = ( 1 - field );
			break;

		case MJPEG_ILACE_EVEN_FIRST:
			cf = field;
			break;

		default:
			return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "MJPEG! unsupported interlacing type (%i). internal error (?)", priv->ilace_type );
	    }
	}
	else	cf = 0;

	ywidth = AlignUP( priv->cinfo.output_width, DCTSIZE * 2 );
	cwidth = AlignUP( priv->cinfo.output_width >> 1, DCTSIZE );

	yuv_base[0] = dest[0] + ywidth * cf;
	yuv_base[1] = dest[2] + cwidth * cf;
	yuv_base[2] = dest[1] + cwidth * cf;

	while( priv->cinfo.output_scanline < priv->cinfo.output_height )
	{
	    for( y = 0, c = 0; y < ( vsf[0] * DCTSIZE ); y += vsf[0], c++ )
	    {
		priv->yuv_buffer[0][y] = yuv_base[0];
		yuv_base[0] += ywidth * numfields;

		if( vsf[0] == 2 )
		{
		    priv->yuv_buffer[0][y + 1] = yuv_base[0];
		    yuv_base[0] += ywidth * numfields;
		}

		priv->yuv_buffer[1][c] = yuv_base[1];
		priv->yuv_buffer[2][c] = yuv_base[2];

		if(( vsf[0] == 2 ) || ( c & 1 ))	/* vertical downsampling */
		{
		    yuv_base[1] += cwidth * numfields;
		    yuv_base[2] += cwidth * numfields;
		}
	    }

	    jpeg_read_raw_data( &priv->cinfo, priv->yuv_buffer, vsf[0] * DCTSIZE );
	}

	jpeg_finish_decompress( &priv->cinfo );

	if(( field == 0 ) && ( numfields > 1 ))
	{
	    while( priv->cinfo.src->bytes_in_buffer > 1 )
	    {
		if( priv->cinfo.src->next_input_byte[0] != 0xff )	break;
		if( priv->cinfo.src->next_input_byte[1] != 0xff )	break;

		priv->cinfo.src->bytes_in_buffer--;
		priv->cinfo.src->next_input_byte++;
	    }
	}
  }

  return isize;
}

/*
 * Encode data
 */
static int mjpeg_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *mjpeg_Info( void *xmm )
{
  return &mjpeg_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"MJPEG",
				"Codec: Motion-JPEG",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				mjpeg_Open, mjpeg_Close, mjpeg_Control,
				mjpeg_Decode, mjpeg_Encode, mjpeg_Info };

/*
 * Internal
 */

static void mjpeg_ErrorHandler( j_common_ptr cinfo )
{
  struct mjpeg_error_s *error = (struct mjpeg_error_s *) cinfo->err;

  longjmp( error->setjmp_buffer, 1);
}

/*
 * libjpeg  source manager
 */

/*
 * Initialize source
 */
static void mjpeg_init_source( j_decompress_ptr cinfo )
{
}

/*
 * Fill input buffer
 */
static boolean mjpeg_fill_input_buffer( j_decompress_ptr cinfo )
{
  static char EOI_data[2] = { 0xFF, 0xD9 };

  cinfo->src->next_input_byte = EOI_data;
  cinfo->src->bytes_in_buffer = 2;

  return TRUE;
}

/*
 * Skip data
 */
static void mjpeg_skip_input_data( j_decompress_ptr cinfo, long num_bytes )
{
  if( num_bytes > 0 )
  {
	if( num_bytes > cinfo->src->bytes_in_buffer )
		num_bytes = cinfo->src->bytes_in_buffer;

	cinfo->src->next_input_byte += num_bytes;
	cinfo->src->bytes_in_buffer -= num_bytes;
   }
}

/*
 * Terminate source
 */
static void mjpeg_term_source( j_decompress_ptr cinfo )
{
}
