/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * rgb.c
 * Plugin for RGB Uncompressed
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Types
 */

struct priv_t
{
    uint32_t			depth;
    XMM_VideoFormat		vf;
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	rgb_cvi =
{
    XMM_CODEC_VCF_DECODE | XMM_CODEC_VCF_ENCODE,
    "RGB ( uncompressed )",		/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( 'R','G','B',' ' ),	/* FOURCC */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *rgb_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;

  /* decoding / encoding supported */
  if(( mode & ( XMM_CODEC_MODE_DECODE | XMM_CODEC_MODE_ENCODE )) == 0 )
	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check codec */
  switch( vf->codec )
  {
	case	0x00000000:
	case	xmmFOURCC( 'R','G','B',' ' ):
	case	xmmFOURCC( 'D','I','B',' ' ):
			break;

	default:	return (void *)XMM_RET_NOTSUPPORTED;
  }

  /* Only query codec  */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(RGB) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Set codec description */
  if( mode & XMM_CODEC_MODE_DECODE )
	strcpy( vf->desc, "RGB (uncompressed)" );

  /* Get depth for decoding */
  priv->depth = vf->bpp;

  /* Set video format for encoding */
  priv->vf.codec = xmmFOURCC( 'R','G','B',' ' );
  priv->vf.width = vf->width;
  priv->vf.height = vf->height;
  priv->vf.planes = vf->planes;
  priv->vf.bpp = vf->bpp;
  priv->vf.imgSize = vf->imgSize;
  priv->vf.extraSize = 0;
  strcpy( priv->vf.desc, "" );

  return pCodec;
}

/*
 * Free codec
 */
static int rgb_Close( XMM_PluginCodecVideo *codec )
{
  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int rgb_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  struct priv_t *priv = codec->sys.priv;

  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if(((uint32_t)arg) == XMM_GRAPH_FMT_RGB( priv->depth ))	return XMM_CTLRET_TRUE;
	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = 1;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    *((uint32_t *)arg) = xmmFOURCC( 'R','G','B', priv->depth );
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_SIZE:
	    *((uint32_t *)arg) = sizeof( priv->vf );
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLGET_FORMAT_DATA:
	    *((char **)arg) = (char *)&priv->vf;
	    return XMM_CTLRET_ARG;			/* Result in arg */

    case XMM_CTLSET_GFORMAT:
	    return XMM_RET_NOTSUPPORTED;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )
	return xmm_SetError( codec->sys.xmm, XMM_RET_NOTSUPPORTED, "(RGB) cmd = 0x%x" );

  return xmm_SetError( codec->sys.xmm, XMM_RET_INVALID_ARG, "(RGB) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int rgb_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  memcpy( dest[0], src, isize );

  return isize;
}

/*
 * Encode data
 */
static int rgb_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  memcpy( dest, src[0], isize );

  return isize;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *rgb_Info( void *xmm )
{
  return &rgb_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"RGB",
				"Codec: Uncompressed RGB",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				rgb_Open, rgb_Close, rgb_Control,
				rgb_Decode, rgb_Encode,
				rgb_Info };
