/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wmv.c
 * Video codec plugin for VFW (DLL) and DirectShow video codecs
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/lpgraph.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include "wmv.h"

/*
 * Types
 */


struct priv_t
{
    struct decoder_s		decoder;
    int				yflip;
};

/*
 * Codec info
 */

static XMM_CodecVideoInfo	wmv_cvi =
{
    XMM_CODEC_VCF_DECODE,
    "VFW / DirectShow",			/* Name / Short description */
    "",					/* Filename. Will be initialized later */
    xmmFOURCC( '?','?','?','?' ),	/* FOURCC. */
};

/*
 * Global data
 */

extern XMM_PluginCodecVideo	plugin_info;

/*
 * Prototypes
 */


/*
 * Initialize Plugin
 */
static XMM_PluginCodecVideo *wmv_Open( void *xmm, int mode, XMM_VideoFormat *vf )
{
  XMM_PluginCodecVideo	*pCodec;
  struct priv_t		*priv;
  int			cna = XMM_RET_ERROR;
#ifdef WMV_USE_DS
  int			cds = 0;
#endif

  /* Only decoding supported */
  if(!( mode & XMM_CODEC_MODE_DECODE ))	return (void *)XMM_RET_NOTSUPPORTED;

  /* Check for VFW codec */
#ifdef WMV_USE_VFW
  cna = xmm_AcCodecFile( xmm, vf->codec, XMM_CFG_CODEC_VFW, NULL );
#endif
  /* Check for DS codec */
#ifdef WMV_USE_DS
  if( cna != XMM_RET_OK )
  {
	cds = 1;
	cna = xmm_AcCodecFile( xmm, vf->codec, XMM_CFG_CODEC_DSHOW_VIDEO, NULL );
  }
#endif

  /* Only query codec */
  if( mode & XMM_CODEC_MODE_QUERY )	return (void *)cna;

  /* No codec available */
  if( cna != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(WMV) No matching W32 codec found." );
	return NULL;
  }

  /* Allocate codec */
  if(( pCodec = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginCodecVideo ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pCodec[1];
  pCodec->sys.priv = (void *) priv;
  pCodec->sys.xmm = xmm;

  /* Set decoder struct */
#ifdef WMV_USE_DS
  if( cds == 1 )	wmv_decoder_init_ds( xmm, &priv->decoder );
  else
#endif
#ifdef WMV_USE_VFW
	wmv_decoder_init_vfw( xmm, &priv->decoder );
#endif

  /* Open codec */
  if( priv->decoder.Open( pCodec->sys.xmm, &priv->decoder, vf, &priv->yflip ) < 0 )
  {
	free( pCodec );
	return NULL;
  }

  return pCodec;
}

/*
 * Free codec
 */
static int wmv_Close( XMM_PluginCodecVideo *codec )
{
  struct priv_t		*priv = codec->sys.priv;

  /* Close WMV */
  priv->decoder.Close( codec->sys.xmm, &priv->decoder );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Codec control
 */
static int wmv_Control( XMM_PluginCodecVideo *codec, uint32_t cmd, void *arg )
{
  struct priv_t		*priv = codec->sys.priv;
  int			i;
  uint32_t		aFmt[] =
  {
	XMM_GRAPH_FMT_YUY2, XMM_GRAPH_FMT_YV12,
	XMM_GRAPH_FMT_RGB( 32 ), XMM_GRAPH_FMT_RGB( 24 ),
	XMM_GRAPH_FMT_RGB( 16 ), XMM_GRAPH_FMT_RGB( 15 ),
	0
  };

  switch( cmd )
  {
    case XMM_CTLQUERY_GFORMAT:
	    if( priv->decoder.QueryFormat( codec->sys.xmm, &priv->decoder, (uint32_t) arg ) >= 0 )
	    	return XMM_CTLRET_TRUE;

	    return XMM_CTLRET_FALSE;

    case XMM_CTLQUERY_YFLIP:
	    *((uint32_t *)arg) = priv->yflip;
	    return XMM_CTLRET_ARG;

    case XMM_CTLGET_GFORMAT:
	    for( i = 0; aFmt[i]; i++ )
	    {
		if( priv->decoder.QueryFormat( codec->sys.xmm, &priv->decoder, aFmt[i] ) == 0 )
		{
		    *((uint32_t *)arg) = aFmt[i];
		    return XMM_CTLRET_ARG;			/* Result in arg */
		}
	    }

	    return XMM_RET_NOTSUPPORTED;

    case XMM_CTLGET_FORMAT_SIZE:
	    return XMM_RET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLGET_FORMAT_DATA:
	    return XMM_RET_NOTSUPPORTED;	/* No encoding */

    case XMM_CTLSET_GFORMAT:
	    if( priv->decoder.SetFormat( codec->sys.xmm, &priv->decoder, ( uint32_t) arg ) >= 0 )
		return XMM_CTLRET_TRUE;

	    return XMM_CTLRET_FALSE;

    default:
	    break;
  }

  if( cmd & XMM_CTLMASK_CODEC )
	return xmm_SetError( codec->sys.xmm, XMM_RET_NOTSUPPORTED, "(WMV) cmd = 0x%x" );

  return xmm_SetError( codec->sys.xmm, XMM_RET_INVALID_ARG, "(WMV) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int wmv_Decode( XMM_PluginCodecVideo *codec, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t		*priv = codec->sys.priv;

  return priv->decoder.Decode( codec->sys.xmm, &priv->decoder, src, isize, dest, osize, flags );
}

/*
 * Encode data
 */
static int wmv_Encode( XMM_PluginCodecVideo *codec, uint8_t *src[], int isize, uint8_t *dest, int *osize, int *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Codec Info
 */
static XMM_CodecVideoInfo *wmv_Info( void *xmm )
{
  return &wmv_cvi;
}

/*
 * Plugin data
 */
XMM_PluginCodecVideo plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_CODEC,
				XMM_PLUGIN_TYPE_VCODEC,
				XMM_VERSION_NUM,
				"",
				"WMV",
				"Codec: WMV DLL support ( Decoding )",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				wmv_Open, wmv_Close, wmv_Control,
				wmv_Decode, wmv_Encode, wmv_Info };
