/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wmv_ds.c
 * DirectShow
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/error.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>

#include "wine/ldt_keeper.h"
#include "w32dll.h"

#define NOAVIFILE_HEADERS
#include "DirectShow/guids.h"
#include "DirectShow/DS_VideoDecoder.h"

#include "wmv.h"

#ifdef WMV_USE_DS

/*
 * Prototypes
 */

static int ds_Open( void *xmm, decoder_t *decoder, XMM_VideoFormat *vf, int *yflip );
static int ds_Close( void *xmm, decoder_t *decoder );
static int ds_QueryFormat( void *xmm, decoder_t *decoder, uint32_t format );
static int ds_SetFormat( void *xmm, decoder_t *decoder, uint32_t format );
static int ds_Decode( void *xmm, decoder_t *decoder, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags );

#ifdef VERBOSE
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text );
#endif

/*
 * Types
 */

struct priv_t
{
    void			*ds_handle;		/* DS handle */

    BITMAPINFOHEADER		*src_bm_head;		/* Input */
    BITMAPINFO			*dest_bm_head;		/* Output */

    LDT_FS			*ldt_fs;

    XMM_AcCodecProperties	cp;
};

/*
 * Public
 */

int wmv_decoder_init_ds( void *xmm, decoder_t *decoder )
{
  decoder->Open = ds_Open;
  decoder->Close = ds_Close;
  decoder->QueryFormat = ds_QueryFormat;
  decoder->SetFormat = ds_SetFormat;
  decoder->Decode = ds_Decode;

  return 0;
}

/*
 * Open
 */
static int ds_Open( void *xmm, decoder_t *decoder, XMM_VideoFormat *vf, int *yflip )
{
  struct priv_t		*priv = decoder->priv;
//  GUID			guid = { 0x4facbba1, 0xffd8, 0x4cd7, { 0x82, 0x28, 0x61, 0xe2, 0xf6, 0x5c, 0xb1, 0xae }};

  /* Allocate private data */
  if(( decoder->priv = malloc( sizeof( struct priv_t ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/DS) DS private data" );

  priv = decoder->priv;

  /* Get codec properties */
  if( xmm_AcCodecFile( xmm, vf->codec, XMM_CFG_CODEC_DSHOW_VIDEO, &priv->cp ) < XMM_RET_OK )
	return XMM_RET_NOTSUPPORTED;

  xmm_logging( 3, "WMV/DS! CodecConf: %s ( for codec 0x%x [%s] )\n", priv->cp.file, vf->codec, xmm_FOURCC_string( vf->codec ));
  xmm_logging( 3, "WMV/DS! CodecConf: GUID %x, %x, %x, %x %x %x %x %x %x %x %x\n", priv->cp.guid.v1, priv->cp.guid.v2, priv->cp.guid.v3, priv->cp.guid.v4[0], priv->cp.guid.v4[1], priv->cp.guid.v4[2], priv->cp.guid.v4[3], priv->cp.guid.v4[4], priv->cp.guid.v4[5], priv->cp.guid.v4[6], priv->cp.guid.v4[7] );

  /* Set input data */
  if(( priv->src_bm_head = malloc( sizeof( BITMAPINFOHEADER ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/DS) Unable to allocate src_bm_head" );

  priv->src_bm_head->biSize = 0x28;
  priv->src_bm_head->biWidth = vf->width;
  priv->src_bm_head->biHeight = vf->height;
  priv->src_bm_head->biPlanes = vf->planes;
  priv->src_bm_head->biBitCount = vf->bpp;
  priv->src_bm_head->biCompression = vf->codec;
  priv->src_bm_head->biSizeImage = vf->imgSize;
  priv->src_bm_head->biXPelsPerMeter = 0;
  priv->src_bm_head->biYPelsPerMeter = 0;
  priv->src_bm_head->biClrUsed = 0;
  priv->src_bm_head->biClrImportant = 0;

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "-DS_VideoDecoder_Open (src_bm_head)" );
#endif

  /* Open DirectShow video codec */
  SetCodecDLL( priv->cp.file );
  priv->ldt_fs = Setup_LDT_Keeper();

  priv->ds_handle = DS_VideoDecoder_Open( priv->cp.file, (GUID *)&priv->cp.guid, priv->src_bm_head, 0, 0 );
  if( priv->ds_handle == NULL )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/DS) Cannot open decoder ( Open() failed )" );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "+DS_VideoDecoder_Open (src_bm_head)" );
#endif

  /* Set codec description */
  strcpy( vf->desc, priv->cp.info );

  /* Y-Flipping */
  *yflip = ( priv->cp.flags & XMM_CFG_CODEC_PF_YFLIP ) ? 1 : 0;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int ds_Close( void *xmm, decoder_t *decoder )
{
  struct priv_t		*priv = decoder->priv;

  /* Close DirectShow decoder */
  if( priv->ds_handle )	DS_VideoDecoder_Destroy( priv->ds_handle );

  Restore_LDT_Keeper( priv->ldt_fs );

  free( priv->src_bm_head );
  free( priv->dest_bm_head );

  if( decoder->priv )	free( decoder->priv );
  decoder->priv = NULL;

  return XMM_RET_OK;
}

/*
 * Query format
 */
static int ds_QueryFormat( void *xmm, decoder_t *decoder, uint32_t format )
{
  struct priv_t		*priv = decoder->priv;
  char			*str;

  if( format == XMM_GRAPH_FMT_RGB( 32 ))	str = "RGB32";
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))	str = "RGB24";
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))	str = "RGB16";
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))	str = "RGB15";
  else if( format == XMM_GRAPH_FMT_YV12 )	str = "YV12";
  else if( format == XMM_GRAPH_FMT_YUY2 )	str = "YUY2";
  else return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/DS) Unknown format 0x%x [%s].", format, xmm_FOURCC_string( format ));

  if( strstr( priv->cp.ofmt, str ) != NULL )	return 0;

  return XMM_RET_NOTSUPPORTED;
}

/*
 * Set format
 */
static int ds_SetFormat( void *xmm, decoder_t *decoder, uint32_t format )
{
  struct priv_t	*priv = decoder->priv;

  /* Check if format supported by codec */
  if( ds_QueryFormat( xmm, decoder, format ) < 0 )
	return XMM_RET_NOTSUPPORTED;

  /* Set output data, we need BITMAPINFO with 2 additional bmiColors */
  if(( priv->dest_bm_head = malloc( sizeof( BITMAPINFO ) + 2 * sizeof( int ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/DS) Unable to allocated dest_bm_head" );

  priv->dest_bm_head->bmiHeader.biSize = sizeof( BITMAPINFO ) + 2 * sizeof( int );

  /* Set destination bitmap */
  if( format == XMM_GRAPH_FMT_RGB( 32 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 32;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 24;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0xF800;
	priv->dest_bm_head->bmiColors[1] = 0x07E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0x7C00;
	priv->dest_bm_head->bmiColors[1] = 0x03E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_YV12 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = format;
  }
  else if( format == XMM_GRAPH_FMT_YUY2 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = format;
  }
  else return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/DS) Unknown format '%s'.", xmm_FOURCC_string( format ));

  /* Set image size and flipping */
  priv->dest_bm_head->bmiHeader.biWidth = priv->src_bm_head->biWidth;
  priv->dest_bm_head->bmiHeader.biHeight = priv->src_bm_head->biHeight;
  priv->dest_bm_head->bmiHeader.biSizeImage = priv->dest_bm_head->bmiHeader.biWidth * priv->dest_bm_head->bmiHeader.biHeight * priv->dest_bm_head->bmiHeader.biBitCount / 8;

  priv->dest_bm_head->bmiHeader.biPlanes = priv->src_bm_head->biPlanes;
  priv->dest_bm_head->bmiHeader.biXPelsPerMeter = 0;
  priv->dest_bm_head->bmiHeader.biYPelsPerMeter = 0;
  priv->dest_bm_head->bmiHeader.biClrUsed = 0;
  priv->dest_bm_head->bmiHeader.biClrImportant = 0;

  /* YUV Hack */
  if((( format == XMM_GRAPH_FMT_YV12 ) || ( format == XMM_GRAPH_FMT_YUY2 )) &&
	( priv->cp.flags & XMM_CFG_CODEC_PF_YUVHACK ))
  {
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }

#ifdef VERBOSE
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-DS_VideoDecoder_Open (dest_bm_head)" );
#endif

  /* Set destination format */
  DS_VideoDecoder_SetDestFmt(	priv->ds_handle,
				priv->dest_bm_head->bmiHeader.biBitCount,
				priv->dest_bm_head->bmiHeader.biCompression );

  /* YUV Hack */
  if((( format == XMM_GRAPH_FMT_YV12 ) || ( format == XMM_GRAPH_FMT_YUY2 )) &&
	( priv->cp.flags & XMM_CFG_CODEC_PF_YUVHACK ))
  {
	priv->dest_bm_head->bmiHeader.biCompression = format;
  }

  /* Start DirectShow video codec */
  DS_VideoDecoder_StartInternal( priv->ds_handle );

  xmm_logging( 2, "WMV/DS! Desired format %x [%s]\n", format, xmm_FOURCC_string( format ));
  xmm_logging( 2, "WMV/DS! Decoding to %x [%s] biBitCount = %i\n", priv->dest_bm_head->bmiHeader.biCompression, xmm_FOURCC_string( priv->dest_bm_head->bmiHeader.biCompression ), priv->dest_bm_head->bmiHeader.biBitCount );

  return XMM_RET_OK;
}

/*
 * Decode data
 */
static int ds_Decode( void *xmm, decoder_t *decoder, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t	*priv = decoder->priv;

  if( DS_VideoDecoder_DecodeInternal( priv->ds_handle, src, isize, 0, dest[0] ) != 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/DS) Cannot decode frame ( DecodeInternal() error )" );

  return isize;
}

/*
 * Print BITMAPINFOHEADER
 */

#ifdef VERBOSE
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text )
{

  xmm_logging( 1, "WMV/DS! --> %s\n", text );
  xmm_logging( 1, "\tbi_head: Size = %li\n", bih->biSize );
  xmm_logging( 1, "\tbi_head: Width = %li\n", bih->biWidth );
  xmm_logging( 1, "\tbi_head: Height = %li\n", bih->biHeight );
  xmm_logging( 1, "\tbi_head: Planes = %i\n", bih->biPlanes );
  xmm_logging( 1, "\tbi_head: BitCount = %i\n", bih->biBitCount );
  xmm_logging( 1, "\tbi_head: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( bih->biCompression ), bih->biCompression );
  xmm_logging( 1, "\tbi_head: SizeImage = %li\n", bih->biSizeImage );
  xmm_logging( 1, "\tbi_head: XPels/m = %li\n", bih->biXPelsPerMeter );
  xmm_logging( 1, "\tbi_head: YPels/m = %li\n", bih->biYPelsPerMeter );
  xmm_logging( 1, "\tbi_head: ClrUsed = %li\n", bih->biClrUsed );
  xmm_logging( 1, "\tbi_head: ClrImportant = %li\n", bih->biClrImportant );
}
#endif

#endif
