/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wmv_vfw.c
 * VFW
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/error.h>
#include <libxmm/lpcodecv.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>

#include "wine/windef.h"
#include "wine/driver.h"
#include "wine/vfw.h"
#include "wine/ldt_keeper.h"
#include "w32dll.h"
#include "wmv.h"

#ifdef WMV_USE_VFW

/*
 * Prototypes
 */

static int vfw_Open( void *xmm, decoder_t *decoder, XMM_VideoFormat *vf, int *yflip );
static int vfw_Close( void *xmm, decoder_t *decoder );
static int vfw_QueryFormat( void *xmm, decoder_t *decoder, uint32_t format );
static int vfw_SetFormat( void *xmm, decoder_t *decoder, uint32_t format );
static int vfw_Decode( void *xmm, decoder_t *decoder, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags );

#ifdef VERBOSE
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text );
#endif

/*
 * Types
 */

struct priv_t
{
    HIC				hic;			/* Decompressor handle */

    BITMAPINFOHEADER		*src_bm_head;		/* Input */
    BITMAPINFO			*dest_bm_head;		/* Output */

    LDT_FS			*ldt_fs;

    XMM_AcCodecProperties	cp;
};

/*
 * Public
 */

int wmv_decoder_init_vfw( void *xmm, decoder_t *decoder )
{
  decoder->Open = vfw_Open;
  decoder->Close = vfw_Close;
  decoder->QueryFormat = vfw_QueryFormat;
  decoder->SetFormat = vfw_SetFormat;
  decoder->Decode = vfw_Decode;

  return 0;
}

/*
 * Open
 */
static int vfw_Open( void *xmm, decoder_t *decoder, XMM_VideoFormat *vf, int *yflip )
{
  struct priv_t		*priv = decoder->priv;
  HRESULT		h;

  /* Allocate private data */
  if(( decoder->priv = malloc( sizeof( struct priv_t ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/VFW) VFW private data" );

  priv = decoder->priv;

  /* Get codec properties */
  if( xmm_AcCodecFile( xmm, vf->codec, XMM_CFG_CODEC_VFW, &priv->cp ) < XMM_RET_OK )
	return XMM_RET_NOTSUPPORTED;

  xmm_logging( 3, "WMV/VFW! Using DLL: %s ( for codec 0x%x [%s] )\n", priv->cp.file, vf->codec, xmm_FOURCC_string( vf->codec ));

  /* Set input data */
  if(( priv->src_bm_head = malloc( sizeof( BITMAPINFOHEADER ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/VFW) Unable to allocate src_bm_head" );

  priv->src_bm_head->biSize = 0x28;
  priv->src_bm_head->biWidth = vf->width;
  priv->src_bm_head->biHeight = vf->height;
  priv->src_bm_head->biPlanes = vf->planes;
  priv->src_bm_head->biBitCount = vf->bpp;
  priv->src_bm_head->biCompression = vf->codec;
  priv->src_bm_head->biSizeImage = vf->imgSize;
  priv->src_bm_head->biXPelsPerMeter = 0;
  priv->src_bm_head->biYPelsPerMeter = 0;
  priv->src_bm_head->biClrUsed = 0;
  priv->src_bm_head->biClrImportant = 0;

  /* Set output data, we need BITMAPINFO with 2 additional bmiColors */
  if(( priv->dest_bm_head = malloc( sizeof( BITMAPINFO ) + 2 * sizeof( int ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMV/VFW) Unable to allocated dest_bm_head" );

  priv->dest_bm_head->bmiHeader.biSize = sizeof( BITMAPINFO ) + 2 * sizeof( int );

  /* Open VFW */
  SetCodecDLL( priv->cp.file );
  priv->ldt_fs = Setup_LDT_Keeper();

  priv->hic = ICOpen( xmmFOURCC( 'v','i','d','c' ), priv->src_bm_head->biCompression, ICMODE_FASTDECOMPRESS );
  if( priv->hic == 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Cannot open decoder ( ICOpen failed )" );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "-ICDecompressGetFormat" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-ICDecompressGetFormat" );
#endif

  h = ICDecompressGetFormat( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Cannot handle format ( ICDecompressGetFormat failed, error = %li )", h );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "+ICDecompressGetFormat" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+ICDecompressGetFormat" );
#endif

  /* Set codec description */
  strcpy( vf->desc, priv->cp.info );

  /* Y-Flipping */
  *yflip = ( priv->cp.flags & XMM_CFG_CODEC_PF_YFLIP ) ? 1 : 0;

  return XMM_RET_OK;
}

/*
 * Close
 */
static int vfw_Close( void *xmm, decoder_t *decoder )
{
  struct priv_t		*priv = decoder->priv;
#if 0
  HRESULT	h;

  /* Stop decoder */
  h = ICDecompressEnd( priv->hic );
  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) IC error ( ICDecompressEnd, error = %li )", h );

  h = ICClose( priv->hic );
  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) IC error ( ICClose, error = %li )", h );
#endif

  Restore_LDT_Keeper( priv->ldt_fs );

  free( priv->src_bm_head );
  free( priv->dest_bm_head );

  if( decoder->priv )	free( decoder->priv );
  decoder->priv = NULL;

  return XMM_RET_OK;
}

/*
 * Query format
 */
static int vfw_QueryFormat( void *xmm, decoder_t *decoder, uint32_t format )
{
  struct priv_t		*priv = decoder->priv;
  char			*str;

  if( format == XMM_GRAPH_FMT_RGB( 32 ))	str = "RGB32";
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))	str = "RGB24";
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))	str = "RGB16";
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))	str = "RGB15";
  else if( format == XMM_GRAPH_FMT_YV12 )	str = "YV12";
  else if( format == XMM_GRAPH_FMT_YUY2 )	str = "YUY2";
  else return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Unknown format 0x%x [%s].", format, xmm_FOURCC_string( format ));

  if( strstr( priv->cp.ofmt, str ) != NULL )	return 0;

  return XMM_RET_NOTSUPPORTED;
}

/*
 * Set format
 */
static int vfw_SetFormat( void *xmm, decoder_t *decoder, uint32_t format )
{
  struct priv_t	*priv = decoder->priv;
  HRESULT	h;

  /* Check if format supported by codec */
  if( vfw_QueryFormat( xmm, decoder, format ) < 0 )
	return XMM_RET_NOTSUPPORTED;

  /* Set destination bitmap */
  if( format == XMM_GRAPH_FMT_RGB( 32 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 32;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 24 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 24;
	priv->dest_bm_head->bmiHeader.biCompression = 0;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 16 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0xF800;
	priv->dest_bm_head->bmiColors[1] = 0x07E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_RGB( 15 ))
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
	priv->dest_bm_head->bmiHeader.biCompression = 3; 	/* 3 == BI_BITFIELDS */
	priv->dest_bm_head->bmiColors[0] = 0x7C00;
	priv->dest_bm_head->bmiColors[1] = 0x03E0;
	priv->dest_bm_head->bmiColors[2] = 0x001F;
  }
  else if( format == XMM_GRAPH_FMT_YV12 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
  }
  else if( format == XMM_GRAPH_FMT_YUY2 )
  {
	priv->dest_bm_head->bmiHeader.biBitCount = 16;
  }
  else return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Unknown format '%s'.", xmm_FOURCC_string( format ));

  /* Set image size and flipping */
  priv->dest_bm_head->bmiHeader.biSizeImage = priv->src_bm_head->biWidth * priv->src_bm_head->biHeight * priv->src_bm_head->biBitCount / 8;
  priv->dest_bm_head->bmiHeader.biHeight = priv->src_bm_head->biHeight;

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "-After ICDecompressQuery" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-After ICDecompressQuery" );
#endif

  /* Query decoder */
  h = ICDecompressQuery( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Cannot Query decompress ( ICDecompressQuery, error = %li )", h );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "+After ICDecompressQuery" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+After ICDecompressQuery" );
#endif

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "-After ICDecompressBegin" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-After ICDecompressBegin" );
#endif

  /* Start decoder */
  h = ICDecompressBegin( priv->hic, priv->src_bm_head, priv->dest_bm_head );
  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Cannot handle format ( ICDecompressBegin, error = %li )", h );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "+After ICDecompressBegin" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+After ICDecompressBegin" );
#endif

  /* YUV Hack */
  if((( format == XMM_GRAPH_FMT_YV12 ) || ( format == XMM_GRAPH_FMT_YUY2 )) &&
	( priv->cp.flags & XMM_CFG_CODEC_PF_YUVHACK ))
  {
	priv->dest_bm_head->bmiHeader.biCompression = format;
  }

  xmm_logging( 2, "WMV/VFW! Desired format %x [%s]\n", format, xmm_FOURCC_string( format ));
  xmm_logging( 2, "WMV/VFW! Decoding to %x [%s] biBitCount = %i\n", priv->dest_bm_head->bmiHeader.biCompression, xmm_FOURCC_string( priv->dest_bm_head->bmiHeader.biCompression ), priv->dest_bm_head->bmiHeader.biBitCount );

  return XMM_RET_OK;
}

/*
 * Decode data
 */
static int vfw_Decode( void *xmm, decoder_t *decoder, uint8_t *src, int isize, uint8_t *dest[], int *osize, int *flags )
{
  struct priv_t	*priv = decoder->priv;
  HRESULT	h;

  priv->src_bm_head->biSizeImage = isize;

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "-Before ICDecompress" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "-Before ICDecompress" );
#endif

  h = ICDecompress( priv->hic, ICDECOMPRESS_NOTKEYFRAME, priv->src_bm_head, src,
			    (BITMAPINFOHEADER *)priv->dest_bm_head, dest[0] );

#ifdef VERBOSE
  print_bitmapinfoheader( priv->src_bm_head, "+Before ICDecompress" );
  print_bitmapinfoheader( &priv->dest_bm_head->bmiHeader, "+Before ICDecompress" );
#endif

  if( h != ICERR_OK )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMV/VFW) Cannot decode frame ( ICDecompress, error = %li )", h );

  return isize;
}

/*
 * Print BITMAPINFOHEADER
 */

#ifdef VERBOSE
static void print_bitmapinfoheader( BITMAPINFOHEADER *bih, char *text )
{

  xmm_logging( 1, "WMV/VFW! --> %s\n", text );
  xmm_logging( 1, "\tbi_head: Size = %li\n", bih->biSize );
  xmm_logging( 1, "\tbi_head: Width = %li\n", bih->biWidth );
  xmm_logging( 1, "\tbi_head: Height = %li\n", bih->biHeight );
  xmm_logging( 1, "\tbi_head: Planes = %i\n", bih->biPlanes );
  xmm_logging( 1, "\tbi_head: BitCount = %i\n", bih->biBitCount );
  xmm_logging( 1, "\tbi_head: Compression = '%s' ( %lx )\n", xmm_FOURCC_string( bih->biCompression ), bih->biCompression );
  xmm_logging( 1, "\tbi_head: SizeImage = %li\n", bih->biSizeImage );
  xmm_logging( 1, "\tbi_head: XPels/m = %li\n", bih->biXPelsPerMeter );
  xmm_logging( 1, "\tbi_head: YPels/m = %li\n", bih->biYPelsPerMeter );
  xmm_logging( 1, "\tbi_head: ClrUsed = %li\n", bih->biClrUsed );
  xmm_logging( 1, "\tbi_head: ClrImportant = %li\n", bih->biClrImportant );
}
#endif

#endif
