/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * ac3.c
 * Audio codec plugin: Dolby Digital AC3 [decoding]
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include "libac3/ac3.h"

/*
 * Definitions
 */

/*
 * Types
 */

struct priv_t
{
    XMM_BufferQueue		bq;
    XMM_AudioFormat		daf;

    int				framesize;
    int				framesize_pcm;

    uint8_t			*pcm_buffer;
    uint32_t			pcm_buffer_pos;
};

/* for type casting */
typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

/*
 * Filter info
 */

static XMM_FilterAudioInfo	ac3_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "AC3 / Dolby Digital",		/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

static int			ac3_codec_initialized = 0;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *ac3_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_AC3 ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* AC3 library may only be used once */
  if( ac3_codec_initialized )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(AC3) AC3 codec already in use." );
	return NULL;
  }

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(AC3) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set filter description */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_AC3 ))
	strcpy( saf->desc, "AC3 / Dolby Digital" );

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = 2;	// saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  /* Initialize data */
  priv->pcm_buffer_pos = 0;
  priv->framesize = 0;
  priv->framesize_pcm = 0;

  /* Init AC3 and buffer queue */
  ac3dec_init();
  xmmBQ_Init( &priv->bq );
  ac3_codec_initialized = 1;

  return pFilter;
}

/*
 * Free filter
 */
static int ac3_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;

  /* free buffer queue */
  xmmBQ_Free( &priv->bq );
  ac3_codec_initialized = 0;

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int ac3_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		if( priv->framesize == 0 )	*((uint32_t *)data ) = 4096;
		else *((uint32_t *)data ) = DivideUP( param, priv->framesize_pcm ) * priv->framesize;
	        return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &ac3_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		priv->pcm_buffer_pos = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(AC3) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(AC3) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int ac3_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		ret;

  /* Queue the source data */
  xmmBQ_Add( &priv->bq, src, isize );

  /* */
  ret = filter->ProcessBQ( filter, NULL, dest, osize, flags );
  if( ret < XMM_RET_OK )	return ret;

  return isize;
}

/*
 * Process data (BQ)
 */
static int ac3_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		decoded, os = 0, needed;

  /*
   * First, we queue the source data
   */
  if( bq )	xmmBQ_Cat( &priv->bq, bq );

  /*
   * Get unused PCM data
   */
  if( priv->pcm_buffer_pos > 0 )
  {
	/* Calculate needed amount of data */
	os = priv->framesize_pcm - priv->pcm_buffer_pos;
	if( os > *osize )	os = *osize;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer + priv->pcm_buffer_pos, os );
	dest += os;

	/* Updata buffer position */
	priv->pcm_buffer_pos += os;

	if( priv->pcm_buffer_pos == priv->framesize_pcm )
		priv->pcm_buffer_pos = 0;
  }

  /*
   * Decode data
   */
  while( os < *osize )
  {
	/* Decode frame */
	decoded = ac3dec_decode_data( (void *)&priv->pcm_buffer, &priv->framesize, (bq_raw_read_t)xmmBQ_Read, (bq_raw_size_t)xmmBQ_Size, &priv->bq );
	if( decoded == 0 )	break;

	/* Set new frame size */
	priv->framesize_pcm = decoded;

	/* Calculate needed amount of data */
	needed = *osize - os;
	if( needed > priv->framesize_pcm )	needed = priv->framesize_pcm;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer, needed );

	dest += needed;
	os += needed;

	/* Update buffer pos ( this is the last loop ) */
	if( needed < priv->framesize_pcm )	priv->pcm_buffer_pos = needed;
  }

  if( os == 0 )	return XMM_RET_ERROR;
  *osize = os;

  return XMM_RET_OK;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"AC3",
				"Codec: AC3 (decoding)",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				ac3_Open, ac3_Close, ac3_Control,
				ac3_Process, ac3_ProcessBQ };
