/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * ccitt.c
 * Audio codec plugin: CCITT G.711/G.721/G.723/G.726 [decoding]
 * ( CCITT - International Telegraph and Telephone Consultative Committee )
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

#include "ccitt/g72x.h"

/*
 * Definitions
 */

/*
 * Types
 */

struct format_conv
{
    char		desc[256];
    uint32_t		xmm;
    uint32_t		mask;

    int			(*g711_decoder)( unsigned char val );
    unsigned char	(*g711_encoder)( int pcm_val );
    int			(*g72x_decoder)( int code, int out_coding, struct g72x_state *state_ptr );
    int			bits;

    int			blockSize;
};

/*
 * Private data
 */

struct priv_t
{
    /* Decoder */
    struct g72x_state	g72x;
    int			(*g711_decoder)( unsigned char val );
    int			(*g72x_decoder)( int code, int out_coding, struct g72x_state *state_ptr );
    unsigned char	(*g711_encoder)( int pcm_val );
    int			ccitt_bits;

    XMM_AudioFormat	*daf;		/* Decoding output audio format */
    int			g72x_be;

    /* Bit decoding */
    uint32_t		bitbuf;
    int			bitidx;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	ccitt_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "CCITT G.711/G.721/G.723/G.726",	/* Name */
    "",					/* Description */
    "Copyright (c) 2001 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int getconv( uint32_t format, struct format_conv **conv );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *ccitt_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			(*g711_decoder)( unsigned char val );
  unsigned char		(*g711_encoder)( int pcm_val );
  int			(*g72x_decoder)( int code, int out_coding, struct g72x_state *state_ptr );
  int			g72x_be, encode = 0;
  int			ret = 0;
  struct format_conv	*conv;

  g711_decoder = NULL;
  g711_encoder = NULL;
  g72x_decoder = NULL;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /*
   * Check format ID
   */
  if(( daf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_PCM )
  {
	getconv( saf->format, &conv );
	if( conv == NULL )
	{
		xmm_SetError( xmm, XMM_RET_ERROR, "(CCITT) Unknown/Unsupported XMM format 0x%x [decoding]", saf->format );
		return (void *)ret;
	}
	encode = 0;

	g72x_be = saf->format & XMM_AUDIO_MASK_BE;
	g711_decoder = conv->g711_decoder;
	g72x_decoder = conv->g72x_decoder;

	if(( g711_decoder == NULL ) && ( g72x_decoder == NULL ))
	{
		if(( flags & XMM_FILTER_AOF_QUERY ) == 0 )
			xmm_SetError( xmm, XMM_RET_ERROR, "(CCITT) No decoder available for format 0x%x", saf->format );
		return (void *)ret;
	}
  }
  else	if(( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_PCM )
  {
	getconv( daf->format, &conv );
	if( conv == NULL )
	{
		if(( flags & XMM_FILTER_AOF_QUERY ) == 0 )
			xmm_SetError( xmm, XMM_RET_ERROR, "(CCITT) Unknown/Unsupported XMM format 0x%x [encode]", daf->format );
		return (void *)ret;
	}
	encode = 1;

	g72x_be = daf->format & XMM_AUDIO_MASK_BE;
	g711_encoder = conv->g711_encoder;

	if( g711_encoder == NULL )
	{
		xmm_SetError( xmm, XMM_RET_ERROR, "(CCITT) No encoder available for format 0x%x", daf->format );
		return (void *)ret;
	}
  }
  else
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(CCITT) Unsupported conversion ( 0x%x -> 0x%x )", saf->format, daf->format );
	return (void *)ret;
  }

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(CCITT) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Allocate memory */
  if(( priv->daf = malloc( sizeof( XMM_AudioFormat ) + 0 )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(NULL) XMM_AudioFormat" );
	return NULL;
  }

  if((( daf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_PCM ))
  {
	/* Set filter description */
	strcpy( saf->desc, conv->desc );

	/* destination XMM audio format */
	priv->daf->format = XMM_AUDIO_FMT_S16LE;
	priv->daf->samprate = saf->samprate;
	priv->daf->channels = saf->channels;
	priv->daf->bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf->format & XMM_AUDIO_MASK_SIZE ), 8 );
	priv->daf->blockSize = saf->channels * DivideUP(( priv->daf->format & XMM_AUDIO_MASK_SIZE ), 8 );
	priv->daf->extraSize = 0;
	strcpy( priv->daf->desc, "PCM (uncompressed)" );
  }
  else	/* Encoding */
  {
	/* Set filter description */
	strcpy( saf->desc, "PCM (uncompressed)" );

	/* destination XMM audio format */
	priv->daf->format = daf->format;
	priv->daf->samprate = saf->samprate;
	priv->daf->channels = saf->channels;
	priv->daf->bitrate = saf->samprate * conv->bits / 8;
	priv->daf->blockSize = conv->blockSize;
	priv->daf->extraSize = 0;
	strcpy( priv->daf->desc, conv->desc );
  }

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, priv->daf, sizeof(XMM_AudioFormat));

  /* Initialize system */
  priv->g72x_be		= g72x_be;
  priv->g711_decoder	= g711_decoder;
  priv->g711_encoder	= g711_encoder;
  priv->g72x_decoder	= g72x_decoder;
  priv->ccitt_bits	= conv->bits;

  if( priv->g72x_decoder )	g72x_init_state( &priv->g72x );

  return pFilter;
}

/*
 * Free filter
 */
static int ccitt_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int ccitt_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data ) = ( param >> 1 ) * priv->ccitt_bits / 8;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &ccitt_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		priv->bitidx = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(CCITT) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(CCITT) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int ccitt_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		i;
  short		*pcm_ptr;
  int		(*g711_decoder)( unsigned char val );
  int		(*g72x_decoder)( int code, int out_coding, struct g72x_state *state_ptr );
  unsigned char	(*g711_encoder)( int pcm_val );
  uint32_t	val, left;

  if( priv->g711_decoder != NULL )
  {
	g711_decoder = priv->g711_decoder;
	pcm_ptr = (short *) dest;
  
	for( i = 0; i < ( *osize >> 1 ); i++ )
	{
	    if( i >= isize )	break;

	    *pcm_ptr++ = g711_decoder( *src++ );
	}

	*osize = i << 1;
	return i / priv->daf->channels;
  }
  if( priv->g711_encoder != NULL )
  {
	g711_encoder = priv->g711_encoder;
	pcm_ptr = (short *) src;

	for( i = 0; i < *osize; i++ )
	{
	    if( i >= ( isize >> 1 ))	break;

	    *dest++ = g711_encoder( *pcm_ptr++ );
	}

	*osize = i;
	return i << 1;
  }
  else if( priv->g72x_decoder != NULL )
  {
	pcm_ptr = (short *) dest;

#define	DECODE_LOOP( REFILL, GET ) \
	for( i = 0; i < ( *osize >> 1 ); i++ ) \
	{ \
	    /* Read some bits */ \
	    for( val = 0; priv->bitidx < priv->ccitt_bits; left-- ) \
	    { \
		if( left <= 0 ) \
		{ \
		    val = 0xFFFF; \
		    break; \
		} \
 \
		REFILL \
	    } \
 \
	    if( val == 0xFFFF )	continue; \
 \
	    GET \
 \
	    /* Decode value */ \
	    *pcm_ptr++ = g72x_decoder( val, AUDIO_ENCODING_LINEAR, &priv->g72x ); \
	}


	g72x_decoder = priv->g72x_decoder;
	left = isize;

	if( priv->g72x_be == 0 )	/* Little endian */
	{
	    DECODE_LOOP(
		priv->bitbuf <<= 8;				/* Refill */
		priv->bitbuf |= *src++;
		priv->bitidx += 8; ,
		priv->bitidx -= priv->ccitt_bits;		/* Get bits */
		val = priv->bitbuf >> priv->bitidx;
		val &= ( 0xffffffff >> ( 32 - priv->ccitt_bits )); )
	}
	else				/* Big endian */
	{
	    DECODE_LOOP(
		priv->bitbuf |= ( *src++ << priv->bitidx );	/* Refill */
		priv->bitidx += 8; ,
		priv->bitidx -= priv->ccitt_bits;		/* Get bits */
		val = priv->bitbuf & (( 1 << priv->ccitt_bits ) - 1 );
		priv->bitbuf >>= priv->ccitt_bits; )
	}

	*osize = i << 1;
	return ( isize - left );
  }

  return xmm_SetError( filter->sys.xmm, XMM_RET_ERROR, "(CCITT) No decoder initialized. Codec internal error." );
}

/*
 * Decode data (BQ)
 */
static int ccitt_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"CCITT",
				"Codec: CCITT G.711/G.721/G.723/G.726",
				"Copyright (c) 2001 Arthur Kleer",
				NULL, NULL },
				ccitt_Open, ccitt_Close, ccitt_Control,
				ccitt_Process, ccitt_ProcessBQ };

/*
 * Internal code
 */

static struct format_conv format_table[] = 
{
    { "CCITT G.711: u-Law", 		XMM_AUDIO_CODEC_ULAW, XMM_AUDIO_MASK_CODEC, ulaw2linear, linear2ulaw, NULL, 8, 1 },
    { "CCITT G.711: A-Law", 		XMM_AUDIO_CODEC_ALAW, XMM_AUDIO_MASK_CODEC, alaw2linear, linear2alaw, NULL, 8, 1 },
    { "CCITT G.721: 4-bit ADPCM",	XMM_AUDIO_CODEC_G721 | 4, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g721_decoder, 4, 64 },

    { "CCITT G.723: 3-bit ADPCM",	XMM_AUDIO_CODEC_G723 | 3, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g723_24_decoder, 3, 48 },
    { "CCITT G.723: 4-bit ADPCM",	XMM_AUDIO_CODEC_G723 | 4, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g721_decoder, 4, 64 },
    { "CCITT G.723: 5-bit ADPCM",	XMM_AUDIO_CODEC_G723 | 5, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g723_40_decoder, 5, 80 },

    { "CCITT G.726: 2-bit ADPCM",	XMM_AUDIO_CODEC_G726 | 2, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g723_16_decoder, 2, 32 },
    { "CCITT G.726: 3-bit ADPCM",	XMM_AUDIO_CODEC_G726 | 3, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g723_24_decoder, 3, 48 },
    { "CCITT G.726: 4-bit ADPCM",	XMM_AUDIO_CODEC_G726 | 4, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g721_decoder, 4, 64 },
    { "CCITT G.726: 5-bit ADPCM",	XMM_AUDIO_CODEC_G726 | 5, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE, NULL, NULL, g723_40_decoder, 5, 80 },
};

static int getconv( uint32_t format, struct format_conv **conv )
{
  int xmm_fmt_idx;

  /* Find XMM format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if(( format & format_table[xmm_fmt_idx].mask ) == format_table[xmm_fmt_idx].xmm )
		break;

  if( format_table[xmm_fmt_idx].xmm == 0 )	*conv = NULL;
  else	*conv = &format_table[xmm_fmt_idx];

  return 0;
}
