/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * ieee.c
 * Audio codec plugin: IEEE floating point (32bit, 64bit) [decoding]
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/endian.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>

/*
 * Types
 */

struct priv_t
{
    XMM_AudioFormat		daf;			/* Decoding format */
    int				ieee_bits;
    int				ieee_be;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	ieee_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "IEEE (uncompressed)",		/* Name */
    "",					/* Description */
    "Copyright (c) 2001 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *ieee_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_IEEE ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(IEEE) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set filter description */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_IEEE ))
	strcpy( saf->desc, "IEEE (uncompressed)" );

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  priv->ieee_bits = saf->format & XMM_AUDIO_MASK_SIZE;
  priv->ieee_be = saf->format & XMM_AUDIO_MASK_BE;

  return pFilter;
}

/*
 * Free filter
 */
static int ieee_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int ieee_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		if( priv->ieee_bits == 32 )
			*((uint32_t *)data) = ( param >> 1 ) * sizeof( float );
		if( priv->ieee_bits == 64 )
			*((uint32_t *)data) = ( param >> 1 ) * sizeof( double );
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &ieee_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(IEEE) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(IEEE) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int ieee_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  float		*float_ptr;
  int32_t	*ptr32, val32;
  double	*double_ptr;
  int64_t	*ptr64, val64;
  int		i;
  int16_t	*pcm_ptr;
  int32_t	value;

  if( priv->ieee_be == 0 )
  {
	if( priv->ieee_bits == 32 )
	{
	    ptr32 = (uint32_t *) src;
	    pcm_ptr = (int16_t *) dest;
	    float_ptr = (float *) &val32;
	    isize /= sizeof( float );

	    for( i = 0; i < ( *osize >> 1 ); i++ )
	    {
		if( i >= isize )	break;

		val32 = *ptr32++;
		val32 = XMM_INT32_LE( val32 );

		value = (int16_t)( *float_ptr * 32767.0 );
		if( value > 32767 )		value = 32767;
		if( value < -32768 )	value = -32768;

		*pcm_ptr++ = (int16_t) value;
	    }

	    return i * sizeof( float );
	}

	if( priv->ieee_bits == 64 )
	{
	    ptr64 = (uint64_t *) src;
	    pcm_ptr = (int16_t *) dest;
	    double_ptr = (double *) &val64;
	    isize /= sizeof( double );

	    for( i = 0; i < ( *osize >> 1 ); i++ )
	    {
		if( i >= isize )	break;

		val64 = *ptr64++;
		val64 = XMM_INT64_LE( val64 );

		value = (int16_t)( *double_ptr * 32767.0 );
		if( value > 32767 )		value = 32767;
		if( value < -32768 )	value = -32768;

		*pcm_ptr++ = (int16_t) value;
	    }

	    return i * sizeof( double );
	}
  }

  if( priv->ieee_be )
  {
	if( priv->ieee_bits == 32 )
	{
	    ptr32 = (uint32_t *) src;
	    pcm_ptr = (int16_t *) dest;
	    float_ptr = (float *) &val32;
	    isize /= sizeof( float );

	    for( i = 0; i < ( *osize >> 1 ); i++ )
	    {
		if( i >= isize )	break;

		val32 = *ptr32++;
		val32 = XMM_INT32_BE( val32 );

		value = (int16_t)( *float_ptr * 32767.0 );
		if( value > 32767 )		value = 32767;
		if( value < -32768 )	value = -32768;

		*pcm_ptr++ = (int16_t) value;
	    }

	    return i * sizeof( float );
	}

	if( priv->ieee_bits == 64 )
	{
	    ptr64 = (uint64_t *) src;
	    pcm_ptr = (int16_t *) dest;
	    double_ptr = (double *) &val64;
	    isize /= sizeof( double );

	    for( i = 0; i < ( *osize >> 1 ); i++ )
	    {
		if( i >= isize )	break;

		val64 = *ptr64++;
		val64 = XMM_INT64_BE( val64 );

		value = (int16_t)( *double_ptr * 32767.0 );
		if( value > 32767 )		value = 32767;
		if( value < -32768 )	value = -32768;

		*pcm_ptr++ = (int16_t) value;
	    }

	    return i * sizeof( double );
	}
  }

  return XMM_RET_ERROR;
}

/*
 * Decode data (BQ)
 */
static int ieee_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"IEEE",
				"Codec: IEEE (uncompressed)",
				"Copyright (c) 2001 Arthur Kleer",
				NULL, NULL },
				ieee_Open, ieee_Close, ieee_Control,
				ieee_Process, ieee_ProcessBQ };
