/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lame.c
 * Audio codec plugin: MPEG Layer 3 [encoding]
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include <libxmm/error.h>
#include "lame/lame.h"
#include "avimp3.h"

/*
 * Definitions
 */

#define BITRATE			128

/*
 * Types
 */

struct priv_t
{
    XMM_AudioFormat		saf;
    XMM_AudioFormat		*daf;

    lame_global_flags		gf;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	lame_fai =
{
    XMM_FILTER_ACF_ENCODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "MPEG (audio) layer 3",		/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *lame_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t 	*priv;
  wave_mp3format_t	*mp3extra;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_MPEG ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Check format */
  if(( saf->format & XMM_AUDIO_MASK_SIZE ) != 16 )
		return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(LAME) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set filter description */
  strcpy( saf->desc, "PCM (uncompressed)" );

  /* Allocate memory */
  if(( priv->daf = malloc( sizeof( XMM_AudioFormat ) + WAVE_MP3_EXTRA_SIZE )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(NULL) XMM_AudioFormat" );
	return NULL;
  }

  /* Output format */
  priv->daf->format = XMM_AUDIO_CODEC_MPEG | XMM_AUDIO_MASK_SIGNED | 0x10;
  priv->daf->samprate = saf->samprate;
  priv->daf->channels = saf->channels;
  priv->daf->bitrate = BITRATE * 1000;
  priv->daf->blockSize = 1;
  priv->daf->extraSize = WAVE_MP3_EXTRA_SIZE;
  priv->daf->extraType = XMM_AUDIO_EXT_WAVE;
  strcpy( priv->daf->desc, "MPEG Audio Layer 3" );

  mp3extra = (wave_mp3format_t *) &priv->daf[1];
  mp3extra->wID = WAVE_MP3_ID_MPEG;
  mp3extra->dwFlags = WAVE_MP3_FLAG_PADDING_OFF;
  mp3extra->nBlockSize = (uint16_t)((double) BITRATE * 144000 / saf->samprate );
  mp3extra->nFramesPerBlock = 1;
  mp3extra->nCodecDelay = 0x0571;

  /* Set destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
  {
	memcpy( daf, priv->daf, sizeof(XMM_AudioFormat));
	daf->extraSize = 0;
  }

  /* Initialize lame */
  lame_init( &priv->gf );

  priv->gf.num_channels = saf->channels;
  priv->gf.in_samplerate = saf->samprate;
  priv->gf.brate = BITRATE;
  priv->gf.mode = 1;
  priv->gf.quality = 2;
  priv->gf.silent = 1;

  lame_init_params( &priv->gf );

  return pFilter;
}

/*
 * Free filter
 */
static int lame_Close( XMM_PluginFilterAudio *filter )
{
  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int lame_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &lame_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH
	:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(LAME) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(LAME) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int lame_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		done;
  uint32_t	samples;

  samples = isize / ( priv->daf->channels * 2 );

  if( priv->gf.num_channels == 1 )
	done = lame_encode_buffer( &priv->gf, (short *)src, (short *)src, samples, (char *)dest, *osize );
  else
  	done = lame_encode_buffer_interleaved( &priv->gf, (short *)src, samples, (char *)dest, *osize );

  if( done < 0 )	return XMM_RET_ERROR;

  *osize = done;
  return isize;
}

/*
 * Process data (BQ)
 */
static int lame_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"LAME",
				"Codec: MPEG Layer 3 (encoding)",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				lame_Open, lame_Close, lame_Control,
				lame_Process, lame_ProcessBQ };
