/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mad.c
 * Audio codec plugin: MPEG 1.0/2.0/2.5 (audio) Layer 1/2/3 [decoding]
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include "libmad/mad.h"
#include "avimp3.h"

#include <libxmm/endian.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' for verbose mode */
#ifndef VERBOSE
#undef	VERBOSE
#endif

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

#define	IBUFFER_SIZE		8192
#define	MPEG_FRAME_MAXSIZE	6912


/*
 * Types
 */

struct priv_t
{
    struct mad_stream		stream;
    struct mad_frame		frame;
    struct mad_synth		synth;

    XMM_BufferQueue		bq;
    XMM_AudioFormat		daf;			/* Output format */

    int				vbr;
    int				framesize;
    int				framesize_pcm;

    uint32_t			pcm_offset;
    uint8_t			ibuffer[IBUFFER_SIZE];
};

/* for type casting */
typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

/*
 * Filter info
 */

static XMM_FilterAudioInfo	mad_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "MPEG (audio) layer 1/2/3",		/* Name */
    "based on libmad",			/* Description */
    "Copyright (c) 2002 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *mad_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  wave_mp3format_t	*mp3extra;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_MPEG ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MPG123) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set filter description */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_MPEG ))
	strcpy( saf->desc, "MPEG Audio Layer 1/2/3" );

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  /*
   * Get frame size
   */

/* Always add 1 byte for padding in RAW framesize calculation */
#define	MPEG_PADDING			1

/* Samples / Frame MPEG Layer multiplicator */
#define	MPEG_PCM_LMUL_I			384
#define	MPEG_PCM_LMUL_II_III		1152
#define	MPEG_PCM_LMUL			MPEG_PCM_LMUL_II_III
/* Samples / Frame MPEG Type divider */
#define	MPEG_PCM_TDIV_1			1
#define	MPEG_PCM_TDIV_2			2
#define	MPEG_PCM_TDIV_25		4
#define	MPEG_PCM_TDIV			MPEG_PCM_TDIV_1
/* Samples / Frame */
#define	MPEG_PCM			( MPEG_PCM_LMUL / MPEG_PCM_TDIV )

/* MPEG Layer multiplicator for framesize */
#define	MPEG_LAYER_FSMUL_I		48000
#define	MPEG_LAYER_FSMUL_II_III		144000
#define	MPEG_LAYER_FSMUL		MPEG_LAYER_FSMUL_II_III

  /* Guess framesize, assumes MPEG 1.0 */
  priv->framesize = (int)(((double)ABS(saf->bitrate / 8)) / 125 * MPEG_LAYER_FSMUL / saf->samprate ) + MPEG_PADDING;
  priv->framesize_pcm = saf->channels * 2 * MPEG_PCM;

  if( saf->extraType == XMM_AUDIO_EXT_WAVE )
  {
	if( saf->extraSize == WAVE_MP3_EXTRA_SIZE )
	{
	    mp3extra = (wave_mp3format_t *) &saf[1];
	    priv->framesize = mp3extra->nBlockSize / mp3extra->nFramesPerBlock + MPEG_PADDING;
	    priv->framesize_pcm = saf->channels * 2 * MPEG_PCM;
	}
	else	xmm_logging( 1, "MPG123! %i bytes extra data found, %i bytes expected for format 0x%.4x\n", saf->extraSize, WAVE_MP3_EXTRA_SIZE, saf->format );
  }

#ifdef DEBUG
  xmm_logging( 1, "MPG123! Framesize: RAW = %i PCM = %i\n", priv->framesize, priv->framesize_pcm );
#endif

  /* Init libmad */
  mad_stream_init( &priv->stream );
  mad_frame_init( &priv->frame );
  mad_synth_init( &priv->synth );

  /* Init MP3 and buffer queue and some other data */
  xmmBQ_Init( &priv->bq );
  priv->vbr = ( saf->bitrate < 0 ) ? 1 : 0;

  return pFilter;
}

/*
 * Free filter
 */
static int mad_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;

  /* Exit mp3 and free buffer queue */
  mad_synth_finish( &priv->synth );
  mad_frame_finish( &priv->frame );
  mad_stream_finish( &priv->stream );

  xmmBQ_Free( &priv->bq );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int mad_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  int32_t		temp;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		temp = (int32_t) param;

		/* If data in PCM buffer, calculate less source data */
		temp -= priv->pcm_offset;
		if( temp < 0 )	temp = 0;

		if( priv->vbr )	temp = DivideUP( temp, priv->framesize_pcm ) * MPEG_FRAME_MAXSIZE;
		else	temp = DivideUP( temp, priv->framesize_pcm ) * ( priv->framesize + 4 );

		temp -= xmmBQ_Size( &priv->bq );
		if( temp < 0 )	temp = 0;

#ifdef DEBUG
		xmm_logging( 1, "MPG123! SrcSize: %i (dest) bytes from %i (source) bytes\n", param, temp );
#endif
		*((uint32_t *)data ) = (uint32_t) temp;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &mad_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		priv->pcm_offset = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(MPG123) cmd ( 0x%x )" );
}

static inline signed int scale( mad_fixed_t sample )
{
  /* round */
  sample += ( 1L << ( MAD_F_FRACBITS - 16 ));

  /* clip */
  if( sample >= MAD_F_ONE )		sample = MAD_F_ONE - 1;
  else if( sample < -MAD_F_ONE )	sample = -MAD_F_ONE;

  /* quantize */
  return sample >> ( MAD_F_FRACBITS + 1 - 16 );
}

/*
 * Process data
 */
static int mad_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		ret, rem, oldrem = 0, read, j;
  int16_t	sample, *s16ptr;
  uint32_t	done = 0, osamples;

  osamples = ( *osize >> 1 ) / priv->daf.channels;

#ifdef DEBUG
  xmm_logging( 1, "MAD! isize: %i, osize: %i ( %i samples )\n", isize, *osize, osamples );
#endif

  /*
   * Conversion: Fixed -> S16
   */

  if( priv->pcm_offset )
  {
#ifdef DEBUG
    xmm_logging( 1, "MAD! converting remaining synth data to unsigned short\n" );
#endif

    s16ptr = (int16_t *) dest;

    for( j = priv->pcm_offset; j < priv->synth.pcm.length; j++ )
    {
	if(( done + j - priv->pcm_offset ) >= osamples )	break;

	sample = (int16_t) ( priv->synth.pcm.samples[0][j] >> ( MAD_F_FRACBITS - 15 ));
	sample = scale( priv->synth.pcm.samples[0][j] );
	*s16ptr++ = sample;

	if( priv->synth.pcm.channels == 2 )
	{
	    sample = (int16_t) ( priv->synth.pcm.samples[1][j] >> ( MAD_F_FRACBITS - 15 ));
	    sample = scale( priv->synth.pcm.samples[1][j] );
	    *s16ptr++ = sample;
	}
    }

    done += ( j - priv->pcm_offset );
    dest += ( j - priv->pcm_offset ) * 2 * priv->synth.pcm.channels;

#ifdef DEBUG
    xmm_logging( 1, "MAD! %i samples converted, %i samples synth data left\n", j - priv->pcm_offset, priv->synth.pcm.length - j );
#endif

    if( j == priv->synth.pcm.length )	j = 0;
    priv->pcm_offset = j;
  }

  /*
   * Decode loop
   */
  while( done < osamples )
  {
	/*
	 * Fill input buffer
	 */
	if(( priv->stream.buffer == NULL ) || ( priv->stream.error == MAD_ERROR_BUFLEN ))
	{
#ifdef DEBUG
	    xmm_logging( 1, "MAD! (re)filling input buffer\n" );
#endif
	    if( priv->stream.next_frame != NULL )
	    {
		rem = priv->stream.bufend - priv->stream.next_frame;
		memmove( priv->ibuffer, priv->stream.next_frame, rem );

#ifdef DEBUG
		xmm_logging( 1, "MAD! %i bytes in source buffer (left)\n", rem );
#endif
	    }
	    else	rem = 0;

	    if( rem < IBUFFER_SIZE )
	    {
		/* Get data from buffer queue */
		rem += xmmBQ_Read( &priv->bq, priv->ibuffer + rem, IBUFFER_SIZE - rem );

#ifdef DEBUG
		xmm_logging( 1, "MAD! %i bytes in source buffer (left, bq)\n", rem );
#endif
	    }

	    if( rem < IBUFFER_SIZE )
	    {
		read = isize;
		if(( rem + read ) > IBUFFER_SIZE )	read = IBUFFER_SIZE - rem;

#ifdef DEBUG
		xmm_logging( 1, "MAD! getting %i bytes from source\n", read );
#endif

		memcpy( priv->ibuffer + rem, src, read );
		src += read;
		isize -= read;
		rem += read;
	    }

	    if( rem == oldrem )	break;
	    oldrem = rem;

#ifdef DEBUG
	    xmm_logging( 1, "MAD! mad_stream_buffer() gets %i bytes, remaining: %i (queue) %i (source)\n", rem, xmmBQ_Size( &priv->bq ), isize );
#endif
	    mad_stream_buffer( &priv->stream, priv->ibuffer, rem );
	    priv->stream.error = 0;
	}

#ifdef DEBUG
	xmm_logging( 1, "MAD! decoding frame\n" );
#endif

	/*
	 * Decode frame
	 */
	ret = mad_frame_decode( &priv->frame, &priv->stream );
	if( ret == -1 )	/* Error */
	{
	    if( MAD_RECOVERABLE( priv->stream.error ))
	    {
		xmm_logging( 2, "MAD! WARNING: frame level error: %s [recoverable]\n",
					mad_stream_errorstr( &priv->stream ));
		continue;
	    }
	    else
	    {
		if( priv->stream.error == MAD_ERROR_BUFLEN )	continue;

		xmm_logging( 1, "MAD! WARNING: frame level error: %s [unrecoverable]\n",
					mad_stream_errorstr( &priv->stream ));
		break;
	    }
	}

#ifdef VERBOSE
	xmm_logging( 1, "MAD! MPEG Audio Layer %i, %i Hz at %i kBit/s\n", priv->frame.header.layer, priv->frame.header.samplerate, priv->frame.header.bitrate / 1000 );
#endif

	/*
	 * Synthesize PCM data
	 */

#ifdef DEBUG
	xmm_logging( 1, "MAD! synthesizing PCM data\n" );
#endif

	mad_synth_frame( &priv->synth, &priv->frame );

	/*
	 * Conversion: Fixed -> S16
	 */

#ifdef DEBUG
	xmm_logging( 1, "MAD! converting synth data to unsigned short ( %i channels, %i samples )\n", priv->synth.pcm.channels, priv->synth.pcm.length );
#endif

	s16ptr = (int16_t *) dest;

	for( j = 0; j < priv->synth.pcm.length; j++ )
	{
		if(( done + j ) >= osamples )
		{
			priv->pcm_offset = j;
			break;
		}

		sample = (int16_t) ( priv->synth.pcm.samples[0][j] >> ( MAD_F_FRACBITS - 15 ));
		sample = scale( priv->synth.pcm.samples[0][j] );
		*s16ptr++ = sample;

		if( priv->synth.pcm.channels == 2 )
		{
		    sample = (int16_t) ( priv->synth.pcm.samples[1][j] >> ( MAD_F_FRACBITS - 15 ));
		    sample = scale( priv->synth.pcm.samples[1][j] );
		    *s16ptr++ = sample;

		}
	}

	done += j;
	dest += j * 2 * priv->synth.pcm.channels;
#ifdef DEBUG
	xmm_logging( 1, "MAD! %i samples converted, %i samples synth data left, %i bytes source left\n", j, priv->synth.pcm.length - j, isize );
#endif
  }

  if( done == 0 )	return XMM_RET_MOREDATA;

  /* Queue the remaining source data */
  if( isize )	xmmBQ_Add( &priv->bq, src, isize );

  *osize = done * 2 * priv->daf.channels;
  return isize;
}



/*
 * Process data (BQ)
 */
static int mad_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
#if 0
  /* */
  ret = filter->ProcessBQ( filter, NULL, dest, osize, flags );
  if( ret < XMM_RET_OK )	return ret;
#endif
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"MAD",
				"Codec: MPEG Layer 1/2/3 (decoding)",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				mad_Open, mad_Close, mad_Control,
				mad_Process, mad_ProcessBQ };
