/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * mpg123.c
 * Audio codec plugin: MPEG 1.0/2.0/2.5 (audio) Layer 1/2/3 [decoding]
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include <libxmm/util/buffer.h>
#include "libmpg123/libmpg123.h"
#include "avimp3.h"

/*
 * Definitions
 */

/* Change 'undef' in 'define' for verbose mode */
#ifndef VERBOSE
#define	VERBOSE
#endif

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#define	DEBUG
#endif

#define	PCM_BUFFER_SIZE		16384
#define	MPEG_FRAME_MAXSIZE	6912


/*
 * Types
 */

struct priv_t
{
    mpg123_t			mpg123;
    mpg123_reader_t		reader;

    XMM_BufferQueue		bq;
    XMM_AudioFormat		daf;			/* Output format */

    int				vbr;
    int				framesize;
    int				framesize_pcm;

    uint8_t			*pcm_buffer;
    uint32_t			pcm_buffer_pos;
    uint32_t			pcm_buffer_size;
};

/* for type casting */
typedef int (*bq_raw_read_t)( void *bq, void *data, int size );
typedef int (*bq_raw_size_t)( void *bq );

/*
 * Filter info
 */

static XMM_FilterAudioInfo	mpg123_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "MPEG (audio) layer 1/2/3",		/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *mpg123_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  wave_mp3format_t	*mp3extra;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_MPEG ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MPG123) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set filter description */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_MPEG ))
	strcpy( saf->desc, "MPEG Audio Layer 1/2/3" );

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  /*
   * Get frame size
   */

/* Always add 1 byte for padding in RAW framesize calculation */
#define	MPEG_PADDING			1

/* Samples / Frame MPEG Layer multiplicator */
#define	MPEG_PCM_LMUL_I			384
#define	MPEG_PCM_LMUL_II_III		1152
#define	MPEG_PCM_LMUL			MPEG_PCM_LMUL_II_III
/* Samples / Frame MPEG Type divider */
#define	MPEG_PCM_TDIV_1			1
#define	MPEG_PCM_TDIV_2			2
#define	MPEG_PCM_TDIV_25		4
#define	MPEG_PCM_TDIV			MPEG_PCM_TDIV_1
/* Samples / Frame */
#define	MPEG_PCM			( MPEG_PCM_LMUL / MPEG_PCM_TDIV )

/* MPEG Layer multiplicator for framesize */
#define	MPEG_LAYER_FSMUL_I		48000
#define	MPEG_LAYER_FSMUL_II_III		144000
#define	MPEG_LAYER_FSMUL		MPEG_LAYER_FSMUL_II_III

  /* Guess framesize, assumes MPEG 1.0 */
  priv->framesize = (int)(((double)ABS(saf->bitrate / 8)) / 125 * MPEG_LAYER_FSMUL / saf->samprate ) + MPEG_PADDING;
  priv->framesize_pcm = saf->channels * 2 * MPEG_PCM;

  if( saf->extraType == XMM_AUDIO_EXT_WAVE )
  {
	if( saf->extraSize == WAVE_MP3_EXTRA_SIZE )
	{
	    mp3extra = (wave_mp3format_t *) &saf[1];
	    priv->framesize = mp3extra->nBlockSize / mp3extra->nFramesPerBlock + MPEG_PADDING;
	    priv->framesize_pcm = saf->channels * 2 * MPEG_PCM;
	}
	else	xmm_logging( 1, "MPG123! %i bytes extra data found, %i bytes expected for format 0x%.4x\n", saf->extraSize, WAVE_MP3_EXTRA_SIZE, saf->format );
  }

#ifdef DEBUG
  xmm_logging( 1, "MPG123! Framesize: RAW = %i PCM = %i\n", priv->framesize, priv->framesize_pcm );
#endif

  /* Allocate PCM buffer */
  priv->pcm_buffer_size = PCM_BUFFER_SIZE;
  priv->pcm_buffer_pos = 0;

  if(( priv->pcm_buffer = malloc( priv->pcm_buffer_size )) == NULL )
  {
	free( pFilter );
	xmm_SetError( xmm, XMM_RET_ALLOC, "(MPG123) PCM buffer" );
	return NULL;
  }

  /* Init reader */
  priv->reader.read = (bq_raw_read_t) xmmBQ_Read;
  priv->reader.size = (bq_raw_size_t) xmmBQ_Size;

  /* Init MP3 and buffer queue and some other data */
  MPG123_Init( &priv->mpg123, /* MMACCEL_X86_MMX | */ MMACCEL_CPU_586 | MMACCEL_CPU_586 );
  xmmBQ_Init( &priv->bq );
  priv->vbr = ( saf->bitrate < 0 ) ? 1 : 0;

  return pFilter;
}

/*
 * Free filter
 */
static int mpg123_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;

  free( priv->pcm_buffer );

  /* Exit mp3 and free buffer queue */
  MPG123_Exit( &priv->mpg123 );
  xmmBQ_Free( &priv->bq );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int mpg123_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  int32_t		temp;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		temp = (int32_t) param;

		/* If data in PCM buffer, calculate less source data */
		if( priv->pcm_buffer_pos > 0 )	 temp = temp - ( priv->framesize_pcm - priv->pcm_buffer_pos );
		if( temp < 0 )	temp = 0;

		if( priv->vbr )	temp = DivideUP( temp, priv->framesize_pcm ) * MPEG_FRAME_MAXSIZE;
		else	temp = DivideUP( temp, priv->framesize_pcm ) * priv->framesize;

		temp -= xmmBQ_Size( &priv->bq );
		if( temp < 0 )	temp = 0;

#ifdef DEBUG
		xmm_logging( 1, "MPG123! SrcSize: %i (dest) bytes from %i (source) bytes\n", param, temp );
#endif
		*((uint32_t *)data ) = (uint32_t) temp + 10;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &mpg123_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		priv->pcm_buffer_pos = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(MPG123) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(MPG123) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int mpg123_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		ret;

xmm_logging( 1, ">Process() got %i bytes, in bq %i bytes\n", isize, xmmBQ_Size( &priv->bq ));

  /* Queue the source data */
  xmmBQ_Add( &priv->bq, src, isize );

  /* */
  ret = filter->ProcessBQ( filter, NULL, dest, osize, flags );
  if( ret < XMM_RET_OK )	return ret;

  return isize;
}

/*
 * Process data (BQ)
 */
static int mpg123_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = filter->sys.priv;
  int		decoded, ret, os = 0, needed;

  /*
   * First, we queue the source data
   */
  if( bq )	xmmBQ_Cat( &priv->bq, bq );

#ifdef DEBUG
  xmm_logging( 1, "mpg123! PCM bufferpos = %i, PCM framesize = %i\n", priv->pcm_buffer_pos, priv->framesize_pcm );
#endif

  /*
   * Get unused PCM data
   */
  if( priv->pcm_buffer_pos > 0 )
  {
	/* Calculate amount of data int PCM buffer */
	os = priv->framesize_pcm - priv->pcm_buffer_pos;
	if( os > *osize )	os = *osize;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer + priv->pcm_buffer_pos, os );
	dest += os;

	/* Updata buffer position */
	priv->pcm_buffer_pos += os;

	if( priv->pcm_buffer_pos == priv->framesize_pcm )
		priv->pcm_buffer_pos = 0;
  }

#ifdef DEBUG
  xmm_logging( 1, "mpg123! Copied: %i bytes, still needed %i\n", os, *osize - os );
#endif

  /*
   * Decode data
   */
  while( os < *osize )
  {
	/* Decode frame */
	decoded = priv->pcm_buffer_size;
	ret = MPG123_DecodeFrame( &priv->mpg123, priv->pcm_buffer, &decoded, &priv->reader, &priv->bq );
	if( ret != MP3_OK )
	{
		xmm_logging( 1, "MPG123! DecodeFrame not OK (%i)\n", ret );
		break;
	}

	/* Set new frame size */
	priv->framesize_pcm = decoded;

	/* Calculate needed amount of data */
	needed = *osize - os;
	if( needed > priv->framesize_pcm )	needed = priv->framesize_pcm;

	/* Copy data */
	memcpy( dest, priv->pcm_buffer, needed );

	dest += needed;
	os += needed;

	/* Update buffer pos ( this is the last loop ) */
	if( needed < priv->framesize_pcm )	priv->pcm_buffer_pos = needed;

#ifdef DEBUG
	xmm_logging( 1, "mpg123! decoded %i, total %i/%i\n", decoded, os, *osize );
#endif
  }

#ifdef DEBUG
  xmm_logging( 1, "mpg123! PCM bufferpos = %i, PCM framesize = %i, left source = %i\n", priv->pcm_buffer_pos, priv->framesize_pcm, priv->reader.size( &priv->bq ));
#endif

  *osize = os;
  if( os == 0 )	return XMM_RET_ERROR;

  return XMM_RET_OK;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"MPG123",
				"Codec: MPEG Layer 1/2/3 (decoding)",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				mpg123_Open, mpg123_Close, mpg123_Control,
				mpg123_Process, mpg123_ProcessBQ };
