/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * w49gsm.c
 * Audio codec plugin: WAVE GSM 6.10 ( ID: 0x0031 ) [decoding]
 */

#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/error.h>
#include <libxmm/util/utils.h>
#include "gsm.h"

/*
 * Definitions
 */

/*
 * Types
 */

struct priv_t
{
    XMM_AudioFormat		daf;

    gsm				gsm_handle;
    gsm_signal			gsm_dest[320];
    int				gsm_left;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	gsm_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "GSM 6.10 (MS)",			/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *gsm_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			value = 1;
  int			ret = 0;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_GSM610MS ) ||
	(( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
		return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Check extra data */
  if( saf->extraSize != 2 )
  {
	xmm_logging( 1, "WAV49 GSM! WARNING: %i bytes extradata found. %i Expected.\n", saf->extraSize, 2 );
  }
  else if(((uint16_t *)&saf[1])[0] != 320 )	/* WORD	wSamplesPerBlock */
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(WAV49 GSM) 320 SamplesPerBlock expected. %i found.\n", ((uint16_t *)&saf[1])[0] );
	return (void *)ret;
  }

  /* Only query codec */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate codec */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(WAV49 GSM) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set codec description */
  if((( saf->format & XMM_AUDIO_MASK_CODEC ) == XMM_AUDIO_CODEC_GSM610MS ))
	strcpy( saf->desc, "GSM 6.10 (MS)" );

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  /* Init GSM stuff */
  if(( priv->gsm_handle = gsm_create()) == NULL )
  {
	free( pFilter );
	xmm_SetError( xmm, XMM_RET_ERROR, "(WAV49 GSM) Cannot initialize gsm pass." );
	return NULL;
  }

  if( gsm_option( priv->gsm_handle, GSM_OPT_WAV49, &value ) == -1 )
  {
	free( pFilter );
	xmm_SetError( xmm, XMM_RET_ERROR, "(WAV49 GSM) Cannot set WAV-style byte ordering option. Not enabled in gsm library ?" );
	return NULL;
  }

  priv->gsm_left = 0;

  return pFilter;
}

/*
 * Free codec
 */
static int gsm_Close( XMM_PluginFilterAudio *codec )
{
  struct priv_t *priv = codec->sys.priv;

  /* destroy gsm object */
  gsm_destroy( priv->gsm_handle );

  free( codec );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int gsm_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data ) = DivideUP( param, 640 ) * 65;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &gsm_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		priv->gsm_left = 0;
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(WAV49 GSM) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(WAV49 GSM) cmd ( 0x%x )" );
}

/*
 * Decode data
 */
static int gsm_Process( XMM_PluginFilterAudio *codec, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t *priv = codec->sys.priv;
  int		ipos = 0, bytes = 0, done;

  memcpy( dest, &priv->gsm_dest[ 320 - priv->gsm_left ], priv->gsm_left * 2 );
  dest += priv->gsm_left * 2;
  done = priv->gsm_left;
  priv->gsm_left = 0;

  while( done < *osize )
  {
	if( ipos == isize )	break;		/* No data left */

	if(( ipos + 65 ) > isize )
		return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "(WAV49 GSM) Only %i bytes left in source. Frame size = %i bytes\n", isize - ipos, 65 );

	if( gsm_decode( priv->gsm_handle, src, priv->gsm_dest ) < 0 )
		return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "(WAV49 GSM) Cannot decode" );

	if( gsm_decode( priv->gsm_handle, src + 33, priv->gsm_dest + 160 ) < 0 )
		return xmm_SetError( codec->sys.xmm, XMM_RET_ERROR, "(WAV49 GSM) Cannot decode" );

	bytes = ( *osize - done ) >> 1;
	if( bytes < 320 )	priv->gsm_left = 320 - bytes;
	else	bytes = 320;

	memcpy( dest, priv->gsm_dest, bytes * 2 );
	dest += bytes * 2;
	done += bytes * 2;
	src += 65;
	ipos += 65;
  }

  *osize = done;
  return ipos;
}

/*
 * Decode data (BQ)
 */
static int gsm_ProcessBQ( XMM_PluginFilterAudio *codec, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"GSM610",
				"Codec: GSM 6.10 (MS) (decoding)",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				gsm_Open, gsm_Close, gsm_Control,
				gsm_Process, gsm_ProcessBQ };
