/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wma.c
 * Audio codec plugin: ACM (DLL) and DirectShow audio codecs [decoding]
 */

#include <stdio.h>
#include <stdlib.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>
#include <libxmm/config.h>

#include "wma.h"

/*
 * Types
 */

struct priv_t
{
    XMM_AudioFormat		daf;
    decoder_t			decoder;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	wma_fai =
{
    XMM_FILTER_ACF_DECODE | XMM_FILTER_ACF_CODEC,
    "",					/* Filename. Will be initialized later */
    "WMA ( W32 DLL )",			/* Name */
    "",					/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *wma_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			cna = XMM_RET_ERROR;
#ifdef WMA_USE_DS
  int			cds = 0;
#endif
  int			ret = 0;
  decoder_t		decoder;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  /* Check conversion - only decoding */
  if((( daf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM ))
	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPCHAN )
	if(( saf->channels != daf->channels ) &&
	    ( saf->channels != XMM_AUDIO_ANY_CHANN ))	return (void *)ret;

  if( flags & XMM_FILTER_AOF_CMPSRATE )
	if(( saf->samprate != daf->samprate ) &&
	    ( saf->samprate != XMM_AUDIO_ANY_SRATE ))	return (void *)ret;

#if 0
  if( flags & XMM_FILTER_AOF_CMPBRATE )
	if(( saf->bitrate != daf->bitrate ) &&
	    ( saf->bitrate != XMM_AUDIO_ANY_BRATE ))	return (void *)ret;
#endif

  /* Check for ACM codec */
#ifdef WMA_USE_ACM
  cna = xmm_AcCodecFile( xmm, ( saf->format & XMM_AUDIO_MASK_CODEC ) >> 16, XMM_CFG_CODEC_ACM, NULL );
#endif
  /* Check for DS codec */
#ifdef WMA_USE_DS
  if( cna != XMM_RET_OK )
  {
	cds = 1;
	cna = xmm_AcCodecFile( xmm, ( saf->format & XMM_AUDIO_MASK_CODEC ) >> 16, XMM_CFG_CODEC_DSHOW_AUDIO, NULL );
  }
#endif

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )
  {
  	if( cna != 0 )	return (void *)cna;

	/* Set decoder struct */
#ifdef WMA_USE_DS
	if( cds == 1 )	wma_decoder_init_ds( xmm, &decoder );
	else
#endif
#ifdef WMA_USE_ACM
	    wma_decoder_init_acm( xmm, &decoder );
#endif

	/* Open codec */
	if( decoder.Open( xmm, NULL, saf ) < 0 )	return (void *)ret;

	return (void *)NULL;
  }

  /* No codec available */
  if( cna != XMM_RET_OK )
  {
	xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(WMA) No matching W32 codec found." );
	return NULL;
  }

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Set decoder struct */
#ifdef WMA_USE_DS
  if( cds == 1 )	wma_decoder_init_ds( xmm, &priv->decoder );
  else
#endif
#ifdef WMA_USE_ACM
	wma_decoder_init_acm( xmm, &priv->decoder );
#endif

  /* only interest in WAVE audio extra data */
  if( saf->extraType != XMM_AUDIO_EXT_WAVE )	saf->extraSize = 0;

  /* Open codec */
  if( priv->decoder.Open( pFilter->sys.xmm, &priv->decoder, saf ) < 0 )
  {
	free( pFilter );
	return NULL;
  }

  /* destination XMM audio format */
  priv->daf.format = XMM_AUDIO_FMT_S16LE;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.blockSize = saf->channels * DivideUP(( priv->daf.format & XMM_AUDIO_MASK_SIZE ), 8 );
  priv->daf.extraSize = 0;
  strcpy( priv->daf.desc, "PCM (uncompressed)" );

  /* Save destination format */
  if( !( flags & XMM_FILTER_AOF_DESTRO ))
	memcpy( daf, &priv->daf, sizeof(XMM_AudioFormat));

  return pFilter;
}

/*
 * Free filter
 */
static int wma_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t		*priv = filter->sys.priv;

  /* Close WMA */
  priv->decoder.Close( filter->sys.xmm, &priv->decoder );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int wma_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;
  uint32_t		srcsize;

  switch( cmd )
  {
	case XMM_CTLGET_DATA_SSIZE:
		priv->decoder.SourceSize( filter->sys.xmm, &priv->decoder, param, &srcsize );
		*((uint32_t *)data ) = (uint32_t)srcsize;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &wma_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(WMA) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(WMA) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int wma_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t		*priv = filter->sys.priv;

  return priv->decoder.Convert( filter->sys.xmm, &priv->decoder, src, isize, dest, osize, flags );
}

/*
 * Process data (BQ)
 */
static int wma_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin data
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_ACODEC,
				XMM_VERSION_NUM,
				"",
				"WMA",
				"Codec: WMA DLL support (decoding)",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				wma_Open, wma_Close, wma_Control,
				wma_Process, wma_ProcessBQ };
