/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wma_acm.c
 * ACM
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/afilter.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>

#include "wine/windef.h"
#include "wine/driver.h"
#include "wine/msacm.h"
#include "wineacm.h"

#include "wine/ldt_keeper.h"
#include "w32dll.h"
#include "wma.h"

#ifdef WMA_USE_ACM

/*
 * Prototypes
 */

static int acm_Open( void *xmm, decoder_t *decoder, XMM_AudioFormat *af );
static int acm_Close( void *xmm, decoder_t *decoder );
static int acm_SourceSize( void *xmm, decoder_t *decoder, uint32_t ssize, uint32_t *dsize );
static int acm_Convert( void *xmm, decoder_t *decoder, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, int *flags );

#ifdef VERBOSE
static void print_waveformatex( WAVEFORMATEX *wave_fmt_head );
#endif

/*
 * Types
 */

struct priv_t
{
    HACMSTREAM			acm_stream_handle;	/* acm stream handle */
    WAVEFORMATEX		*srcfmt;		/* Input */
    WAVEFORMATEX		*destfmt;		/* Output */

    LDT_FS			*ldt_fs;

    uint8_t			*dbuffer;
    uint32_t			dbuffersize;
    uint32_t			dbufferpos;
};

/*
 * Public
 */

int wma_decoder_init_acm( void *xmm, decoder_t *decoder )
{
  decoder->Open = acm_Open;
  decoder->Close = acm_Close;
  decoder->SourceSize = acm_SourceSize;
  decoder->Convert = acm_Convert;

  return 0;
}

/*
 * Open
 */
static int acm_Open( void *xmm, decoder_t *decoder, XMM_AudioFormat *af )
{
  struct priv_t		*priv;
  MMRESULT		h;
  XMM_AcCodecProperties	cp;

  /* Allocate private data */
  if(( priv = malloc( sizeof( struct priv_t ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA) ACM private data" );

  /* Get codec properties */
  if( xmm_AcCodecFile( xmm, ( af->format & XMM_AUDIO_MASK_CODEC ) >> 16, XMM_CFG_CODEC_ACM, &cp ) < XMM_RET_OK )
  {
	free( priv );
	return XMM_RET_NOTSUPPORTED;
  }

  xmm_logging( 3, "WMA/ACM! Using DLL: %s ( for codec 0x%x ) formatID = 0x%x\n", cp.file, af->format & XMM_AUDIO_MASK_CODEC, cp.msid );

  /* Set input data */
  if(( priv->srcfmt = malloc( sizeof( WAVEFORMATEX ) + af->extraSize )) == NULL )
  {
	free( priv );
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA/ACM) Unable to allocate srcfmt" );
  }

  priv->srcfmt->wFormatTag = cp.msid;
  priv->srcfmt->nChannels = af->channels;
  priv->srcfmt->nSamplesPerSec = af->samprate;
  priv->srcfmt->nAvgBytesPerSec = af->bitrate / 8;
  priv->srcfmt->nBlockAlign = af->blockSize;
  priv->srcfmt->wBitsPerSample = ( af->format & XMM_AUDIO_MASK_SIZE );
  priv->srcfmt->cbSize = 0;

  if( af->extraType == XMM_AUDIO_EXT_WAVE )
  {
	priv->srcfmt->cbSize = af->extraSize;
	memcpy( (char *)priv->srcfmt + 18, (void *)&af[1], af->extraSize );
  }

  /* Set output data */
  if(( priv->destfmt = malloc( sizeof( WAVEFORMATEX ))) == NULL )
  {
	free( priv->srcfmt );
	free( priv );
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA/ACM) Unable to allocate destfmt" );
  }
 
  priv->destfmt->wFormatTag = WAVE_FORMAT_PCM;
  priv->destfmt->nChannels = priv->srcfmt->nChannels;
  priv->destfmt->nSamplesPerSec = priv->srcfmt->nSamplesPerSec;
  priv->destfmt->nAvgBytesPerSec = 2 * priv->destfmt->nSamplesPerSec * priv->destfmt->nChannels;
  priv->destfmt->nBlockAlign = 2 * priv->srcfmt->nChannels;
  priv->destfmt->wBitsPerSample = 16;
  priv->destfmt->cbSize = 0;

#ifdef VERBOSE
  xmm_logging( 1, "WMA! ===== ===== >>> WAVE source <<< ===== =====\n" );
  print_waveformatex( priv->srcfmt );
  xmm_logging( 1, "WMA! ===== =====  >>> WAVE dest <<<  ===== =====\n" );
  print_waveformatex( priv->destfmt );
#endif

  /* Open ACM */
  SetCodecDLL( cp.file );
  priv->ldt_fs = Setup_LDT_Keeper();

  /* Open ACM conversion stream */
  h = acmStreamOpen(	&priv->acm_stream_handle, NULL,
			priv->srcfmt,
			priv->destfmt,
			NULL, 0, 0, ( decoder == NULL ) ? ACM_STREAMOPENF_QUERY : 0 );
  if( h )
  {
	free( priv );
  	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) Cannot open decoder ( acmStreamOpen failed, error %lx )", h );
  }

  /* Set codec description */
  if( decoder )
  {
	strcpy( af->desc, cp.info );
	decoder->priv = priv;

	/* initialize decode buffer */
	acmStreamSize( priv->acm_stream_handle, priv->srcfmt->nBlockAlign,
			(DWORD *) &priv->dbuffersize, ACM_STREAMSIZEF_SOURCE );

#ifdef VERBOSE
	xmm_logging( 1, "WMA/ACM! nBlockAlign = %i dbuffersize = %i\n", priv->srcfmt->nBlockAlign, priv->dbuffersize );
#endif

	if(( priv->dbuffer = malloc( priv->dbuffersize )) == NULL )
		return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA/ACM) Unable to allocate decodebuffer ( %i bytes )", priv->dbuffersize );

	priv->dbufferpos = priv->dbuffersize;
  }
  else
  {
	Restore_LDT_Keeper( priv->ldt_fs );
	free( priv->destfmt );
	free( priv->srcfmt );
	free( priv );
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int acm_Close( void *xmm, decoder_t *decoder )
{
  struct priv_t		*priv = decoder->priv;

  /* Close ACM */
  acmStreamClose( priv->acm_stream_handle, 0 );

  /* Free resources */
  free( priv->srcfmt );
  free( priv->destfmt );

  Restore_LDT_Keeper( priv->ldt_fs );

  if( priv->dbuffer )	free( priv->dbuffer );

  if( decoder->priv )	free( decoder->priv );
  decoder->priv = NULL;

  return XMM_RET_OK;
}

/*
 * Query source size
 */
static int acm_SourceSize( void *xmm, decoder_t *decoder, uint32_t dsize, uint32_t *ssize )
{
  struct priv_t		*priv = decoder->priv;

  if( dsize <= ( priv->dbuffersize - priv->dbufferpos ))
  {
	*ssize = 0;

#ifdef VERBOSE
	xmm_logging( 1, "WMA/ACM! SourceSize() %i -> %i ( %i buffered )\n", *ssize, dsize, priv->dbuffersize - priv->dbufferpos );
#endif
	return XMM_RET_OK;
  }

  dsize = dsize - ( priv->dbuffersize - priv->dbufferpos );
  if( dsize < priv->dbuffersize )	dsize = priv->dbuffersize;

  /* ACM Query */
  acmStreamSize( priv->acm_stream_handle, dsize, (DWORD *) ssize, ACM_STREAMSIZEF_DESTINATION );

#ifdef VERBOSE
  xmm_logging( 1, "WMA/ACM! SourceSize() %i -> %i\n", *ssize, dsize );
#endif

  return XMM_RET_OK;
}

/*
 * Convert audio data
 */
static int acm_Convert( void *xmm, decoder_t *decoder, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, int *flags )
{
  struct priv_t		*priv = decoder->priv;
  HRESULT		h;
  ACMSTREAMHEADER	ash;
  DWORD			srcsize;
  uint32_t		used = 0, dsize = *osize, os;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/ACM! Convert() isize = %i osize = %i\n", isize, *osize );
#endif

  if(( priv->dbuffersize - priv->dbufferpos ) > 0 )
  {
	/* Calculate amount of data int PCM buffer */
	os = priv->dbuffersize - priv->dbufferpos;
	if( os > dsize )	os = dsize;

	/* Copy data */
	memcpy( dest, priv->dbuffer + priv->dbufferpos, os );

	/* Updata buffer position */
	priv->dbufferpos += os;
	dsize -= os;
	dest += os;

	if( dsize == 0 )	return 0;

#ifdef VERBOSE
	xmm_logging( 1, "WMA/ACM! Convert() buffered = %i todo = %i\n", os, dsize );
#endif
  }

  /* Get needed input data size */
  acmStreamSize( priv->acm_stream_handle,
		    ( dsize < priv->dbuffersize ) ? priv->dbuffersize : dsize,
		    &srcsize, ACM_STREAMSIZEF_DESTINATION );
  if( srcsize > isize )
	    return xmm_SetError( xmm, XMM_RET_MOREDATA, "(WMA/ACM) input data: got %i bytes, %i bytes needed", isize, srcsize );

  /*
   * Decode data
   */
  memset( &ash, 0, sizeof( ACMSTREAMHEADER ));
  ash.cbStruct = sizeof( ACMSTREAMHEADER );
  ash.fdwStatus = 0;
  ash.dwUser = 0; 
  ash.pbSrc = (BYTE *)src;
  ash.cbSrcLength = srcsize;
  ash.pbDst = (BYTE *)dest;
  ash.cbDstLength = ( dsize / priv->dbuffersize ) * priv->dbuffersize;

  if( ash.cbSrcLength && ash.cbDstLength )
  {
#ifdef VERBOSE
	xmm_logging( 1, "WMA/ACM! Converting(1) %i -> %i\n", ash.cbSrcLength, ash.cbDstLength );
#endif

	/* prepare */
	h = acmStreamPrepareHeader( priv->acm_stream_handle, &ash, 0 );
	if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamPrepareHeader() error %lx", h );

	/* convert */
	h = acmStreamConvert( priv->acm_stream_handle, &ash, 0 );
	if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamConvert() error %lx", h );

	used += ash.cbSrcLengthUsed;
	src += ash.cbSrcLengthUsed;

	dsize -= ash.cbDstLengthUsed;
	dest += ash.cbDstLengthUsed;

	/* Unprepare */
	h = acmStreamUnprepareHeader( priv->acm_stream_handle, &ash, 0 );
	if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamUnprepareHeader() error %lx", h );

	/* Done ? */
	if( dsize == 0 )	return used;
  }

  /*
   * Decode data ( into destination buffer )
   */
  memset( &ash, 0, sizeof( ACMSTREAMHEADER ));
  ash.cbStruct = sizeof( ACMSTREAMHEADER );
  ash.fdwStatus = 0;
  ash.dwUser = 0; 
  ash.pbSrc = (BYTE *)src;
  ash.cbSrcLength = srcsize - used;
  ash.pbDst = (BYTE *)priv->dbuffer;
  ash.cbDstLength = priv->dbuffersize;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/ACM! Converting(2) %i -> %i\n", ash.cbSrcLength, ash.cbDstLength );
#endif

  /* prepare */
  h = acmStreamPrepareHeader( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamPrepareHeader() error %lx", h );

  /* convert */
  h = acmStreamConvert( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamConvert() error %lx", h );

  used += ash.cbSrcLengthUsed;

  /* Copy data */
  memcpy( dest, priv->dbuffer, dsize );
  priv->dbufferpos = dsize;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/ACM! Convert() buffered = %i\n", priv->dbuffersize - priv->dbufferpos );
#endif

  /* Unprepare */
  h = acmStreamUnprepareHeader( priv->acm_stream_handle, &ash, 0 );
  if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/ACM) acmStreamUnprepareHeader() error %lx", h );

  return used;
}

/*
 * Private code
 */

#ifdef VERBOSE

static void print_waveformatex( WAVEFORMATEX *wave_fmt_head )
{
  xmm_logging( 1, "\twave fmt: FormatTag = 0x%x\n", wave_fmt_head->wFormatTag );
  xmm_logging( 1, "\twave fmt: Channels = %i\n", wave_fmt_head->nChannels );
  xmm_logging( 1, "\twave fmt: Samples/Sec = %li\n",  wave_fmt_head->nSamplesPerSec );
  xmm_logging( 1, "\twave fmt: Avg Bytes/Sec = %li\n", wave_fmt_head->nAvgBytesPerSec );
  xmm_logging( 1, "\twave fmt: BlockAlign = %i\n", wave_fmt_head->nBlockAlign );
  xmm_logging( 1, "\twave fmt: Bits / Sample = %i\n", wave_fmt_head->wBitsPerSample );
  xmm_logging( 1, "\twave fmt: Extra data: %i bytes\n", wave_fmt_head->cbSize );
}

#endif
#endif
