/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * wma_ds.c
 * DirectShow audio
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/afilter.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>

#include "wine/windef.h"
#include "wine/driver.h"
#include "wine/msacm.h"
#include "wineacm.h"

#include "wine/ldt_keeper.h"
#include "w32dll.h"

#define NOAVIFILE_HEADERS
#include "DirectShow/guids.h"
#include "DirectShow/DS_AudioDecoder.h"

#include "wma.h"

#ifdef WMA_USE_DS

/*
 * Prototypes
 */

static int ds_Open( void *xmm, decoder_t *decoder, XMM_AudioFormat *af );
static int ds_Close( void *xmm, decoder_t *decoder );
static int ds_SourceSize( void *xmm, decoder_t *decoder, uint32_t ssize, uint32_t *dsize );
static int ds_Convert( void *xmm, decoder_t *decoder, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, int *flags );

#ifdef VERBOSE
static void print_waveformatex( WAVEFORMATEX *wave_fmt_head );
#endif

/*
 * Types
 */

struct priv_t
{
    void			*ds_handle;		/* DS handle */
    WAVEFORMATEX		*srcfmt;		/* Input */

    LDT_FS			*ldt_fs;
    int				samplesize;

    uint8_t			*dbuffer;
    uint32_t			dbuffersize;
    uint32_t			dbufferpos;
};

/*
 * Public
 */

int wma_decoder_init_ds( void *xmm, decoder_t *decoder )
{
  decoder->Open = ds_Open;
  decoder->Close = ds_Close;
  decoder->SourceSize = ds_SourceSize;
  decoder->Convert = ds_Convert;

  return 0;
}

/*
 * Open
 */
static int ds_Open( void *xmm, decoder_t *decoder, XMM_AudioFormat *af )
{
  struct priv_t		*priv;
  XMM_AcCodecProperties	cp;

  /* Allocate private data */
  if(( priv = malloc( sizeof( struct priv_t ))) == NULL )
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA) DS private data" );

  /* Get codec properties */
  if( xmm_AcCodecFile( xmm, ( af->format & XMM_AUDIO_MASK_CODEC ) >> 16, XMM_CFG_CODEC_DSHOW_AUDIO, &cp ) < XMM_RET_OK )
  {
	free( priv );
	return XMM_RET_NOTSUPPORTED;
  }

  xmm_logging( 3, "WMA/DS! CodecConf: %s ( for codec 0x%x ) formatID = 0x%x\n", cp.file, af->format & XMM_AUDIO_MASK_CODEC, cp.msid );
  xmm_logging( 3, "WMA/DS! CodecConf: GUID %x, %x, %x, %x %x %x %x %x %x %x %x\n", cp.guid.v1, cp.guid.v2, cp.guid.v3, cp.guid.v4[0], cp.guid.v4[1], cp.guid.v4[2], cp.guid.v4[3], cp.guid.v4[4], cp.guid.v4[5], cp.guid.v4[6], cp.guid.v4[7] );

  /* Set input data */
  if(( priv->srcfmt = malloc( sizeof( WAVEFORMATEX ) + af->extraSize )) == NULL )
  {
	free( priv );
	return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA/DS) Unable to allocate srcfmt" );
  }

  priv->srcfmt->wFormatTag = cp.msid;
  priv->srcfmt->nChannels = af->channels;
  priv->srcfmt->nSamplesPerSec = af->samprate;
  priv->srcfmt->nAvgBytesPerSec = af->bitrate / 8;
  priv->srcfmt->nBlockAlign = af->blockSize;
  priv->srcfmt->wBitsPerSample = ( af->format & XMM_AUDIO_MASK_SIZE );
  priv->srcfmt->cbSize = 0;

  if( af->extraType == XMM_AUDIO_EXT_WAVE )
  {
	priv->srcfmt->cbSize = af->extraSize;
	memcpy( (char *)priv->srcfmt + 18, (void *)&af[1], af->extraSize );
  }

  /* Open DirectShow audio codec */
  SetCodecDLL( cp.file );
  priv->ldt_fs = Setup_LDT_Keeper();

  priv->ds_handle = DS_AudioDecoder_Open( cp.file, (GUID *)&cp.guid, priv->srcfmt );
  if( priv->ds_handle == NULL )
  {
	free( priv );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/DS) Cannot open decoder ( Open() failed )" );
  }

#ifdef VERBOSE
  xmm_logging( 1, "WMA! ===== ===== >>> WAVE source <<< ===== =====\n" );
  print_waveformatex( priv->srcfmt );
#endif

  /* Set codec description */
  if( decoder )
  {
	strcpy( af->desc, cp.info );
	decoder->priv = priv;

	/* Calculate min PCM buffer */
	priv->dbuffersize = DivideUP( priv->srcfmt->nSamplesPerSec *
		    priv->srcfmt->nBlockAlign, priv->srcfmt->nAvgBytesPerSec );

#ifdef VERBOSE
	xmm_logging( 1, "WMA/DS! nBlockAlign = %i dbuffersize = %i\n", priv->srcfmt->nBlockAlign, priv->dbuffersize );
#endif

	if(( priv->dbuffer = malloc( priv->dbuffersize )) == NULL )
		return xmm_SetError( xmm, XMM_RET_ALLOC, "(WMA/DS) Unable to allocate decodebuffer ( %i bytes )", priv->dbuffersize );

	priv->dbufferpos = priv->dbuffersize;
	priv->samplesize = af->channels * 2;
  }
  else
  {
	DS_AudioDecoder_Destroy( priv->ds_handle );
	Restore_LDT_Keeper( priv->ldt_fs );
	free( priv->srcfmt );
	free( priv );
  }

  return XMM_RET_OK;
}

/*
 * Close
 */
static int ds_Close( void *xmm, decoder_t *decoder )
{
  struct priv_t		*priv = decoder->priv;

  /* Close DirectShow decoder */
  if( priv->ds_handle )	DS_AudioDecoder_Destroy( priv->ds_handle );

  Restore_LDT_Keeper( priv->ldt_fs );

  if( priv->srcfmt )	free( priv->srcfmt );
  if( priv->dbuffer )	free( priv->dbuffer );

  if( decoder->priv )	free( decoder->priv );
  decoder->priv = NULL;

  return XMM_RET_OK;
}

/*
 * Query source size
 */
static int ds_SourceSize( void *xmm, decoder_t *decoder, uint32_t dsize, uint32_t *ssize )
{
  struct priv_t		*priv = decoder->priv;

  if( dsize <= ( priv->dbuffersize - priv->dbufferpos ))
  {
	*ssize = 0;

#ifdef VERBOSE
	xmm_logging( 1, "WMA/DS! SourceSize() %i -> %i ( %i buffered )\n", *ssize, dsize, priv->dbuffersize - priv->dbufferpos );
#endif
	return XMM_RET_OK;
  }

  dsize = dsize - ( priv->dbuffersize - priv->dbufferpos );
  if( dsize < priv->dbuffersize )	dsize = priv->dbuffersize;

  /* DS SrcSize Query */
  *ssize = DS_AudioDecoder_GetSrcSize( priv->ds_handle, dsize ) / priv->samplesize;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/DS! SourceSize() %i -> %i\n", *ssize, dsize );
#endif

  return XMM_RET_OK;
}

/*
 * Convert audio data
 */
static int ds_Convert( void *xmm, decoder_t *decoder, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, int *flags )
{
  struct priv_t		*priv = decoder->priv;
  HRESULT		h;
  ACMSTREAMHEADER	ash;
  DWORD			srcsize;
  uint32_t		used = 0, dsize = *osize, os;
  uint32_t		SrcLengthUsed, DstLengthUsed;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/DS! Convert() isize = %i osize = %i\n", isize, *osize );
#endif

  if(( priv->dbuffersize - priv->dbufferpos ) > 0 )
  {
	/* Calculate amount of data int PCM buffer */
	os = priv->dbuffersize - priv->dbufferpos;
	if( os > dsize )	os = dsize;

	/* Copy data */
	memcpy( dest, priv->dbuffer + priv->dbufferpos, os );

	/* Updata buffer position */
	priv->dbufferpos += os;
	dsize -= os;
	dest += os;

	if( dsize == 0 )	return 0;

#ifdef VERBOSE
	xmm_logging( 1, "WMA/DS! Convert() buffered = %i todo = %i\n", os, dsize );
#endif
  }

  /* Get needed input data size */
  srcsize = DS_AudioDecoder_GetSrcSize( priv->ds_handle, ( dsize < priv->dbuffersize ) ? priv->dbuffersize : dsize ) / priv->samplesize;
  if( srcsize > isize )
	    return xmm_SetError( xmm, XMM_RET_MOREDATA, "(WMA/DS) input data: got %i bytes, %i bytes needed", isize, srcsize );

  /*
   * Decode data
   */
  memset( &ash, 0, sizeof( ACMSTREAMHEADER ));
  ash.cbStruct = sizeof( ACMSTREAMHEADER );
  ash.fdwStatus = 0;
  ash.dwUser = 0; 
  ash.pbSrc = (BYTE *)src;
  ash.cbSrcLength = srcsize;
  ash.pbDst = (BYTE *)dest;
  ash.cbDstLength = ( dsize / priv->dbuffersize ) * priv->dbuffersize;

  /* DS_AudioDecoder_Convert() only converts one frame */
  while( ash.cbSrcLength && ash.cbDstLength )
  {
#ifdef VERBOSE
	xmm_logging( 1, "WMA/DS! Converting(1) %i -> %i\n", ash.cbSrcLength, ash.cbDstLength );
#endif

	/* convert */
	h = DS_AudioDecoder_Convert(	priv->ds_handle,
					ash.pbSrc, ash.cbSrcLength,
					ash.pbDst, ash.cbDstLength,
		    			&SrcLengthUsed, &DstLengthUsed );
	if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/DS) Convert() error %lx", h );

#ifdef VERBOSE
	xmm_logging( 1, "WMA/DS! Converting(1) %i -> %i [done]\n", SrcLengthUsed, DstLengthUsed );
#endif

	used += SrcLengthUsed;
	src += SrcLengthUsed;

	dsize -= DstLengthUsed;
	dest += DstLengthUsed;

	/* update */
	ash.pbSrc = (BYTE *)src;
	ash.cbSrcLength = srcsize;
	ash.pbDst = (BYTE *)dest;
	ash.cbDstLength = ( dsize / priv->dbuffersize ) * priv->dbuffersize;

	/* Done ? */
	if( dsize == 0 )	return used;
  }

  /*
   * Decode data ( into destination buffer )
   */
  memset( &ash, 0, sizeof( ACMSTREAMHEADER ));
  ash.cbStruct = sizeof( ACMSTREAMHEADER );
  ash.fdwStatus = 0;
  ash.dwUser = 0; 
  ash.pbSrc = (BYTE *)src;
  ash.cbSrcLength = srcsize - used;
  ash.pbDst = (BYTE *)priv->dbuffer;
  ash.cbDstLength = priv->dbuffersize;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/DS! Converting(2) %i -> %i\n", ash.cbSrcLength, ash.cbDstLength );
#endif

  /* convert */
  h = DS_AudioDecoder_Convert(	priv->ds_handle,
				ash.pbSrc, ash.cbSrcLength,
				ash.pbDst, ash.cbDstLength,
		    		&SrcLengthUsed, &DstLengthUsed );
  if( h )	xmm_SetError( xmm, XMM_RET_ERROR, "(WMA/DS) Convert() error %lx", h );

  used += SrcLengthUsed;

  /* Copy data */
  memcpy( dest, priv->dbuffer, dsize );
  priv->dbufferpos = dsize;

#ifdef VERBOSE
  xmm_logging( 1, "WMA/DS! Convert() buffered = %i\n", priv->dbuffersize - priv->dbufferpos );
#endif

  return used;
}

/*
 * Private code
 */

#ifdef VERBOSE

static void print_waveformatex( WAVEFORMATEX *wave_fmt_head )
{
  xmm_logging( 1, "\twave fmt: FormatTag = 0x%x\n", wave_fmt_head->wFormatTag );
  xmm_logging( 1, "\twave fmt: Channels = %i\n", wave_fmt_head->nChannels );
  xmm_logging( 1, "\twave fmt: Samples/Sec = %li\n",  wave_fmt_head->nSamplesPerSec );
  xmm_logging( 1, "\twave fmt: Avg Bytes/Sec = %li\n", wave_fmt_head->nAvgBytesPerSec );
  xmm_logging( 1, "\twave fmt: BlockAlign = %i\n", wave_fmt_head->nBlockAlign );
  xmm_logging( 1, "\twave fmt: Bits / Sample = %i\n", wave_fmt_head->wBitsPerSample );
  xmm_logging( 1, "\twave fmt: Extra data: %i bytes\n", wave_fmt_head->cbSize );
}

#endif
#endif
