/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * alsa.c
 * ALSA sound output
 *
 * TODO:
 *  - Flush() is not implemented, as it did not work. Should be fixed soon.
 */

#include <stdlib.h>
#include <sys/asoundlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

/* Change 'undef' in 'define' to get debug info */
#ifndef DEBUG
#undef	DEBUG
#endif

/* Change 'undef' in 'define' to get verbose info */
#ifndef VERBOSE
#define	VERBOSE
#endif

/* Used to set buffer num and size */
#define	FRAG_SIZE	16384
#define	FRAG_COUNT	16

/* Mixer group */
#define	GROUP_NAME	"PCM"
#define	GROUP_INDEX	0

/*
 * Global data
 */

			/* Sound device */
static char		SoundDev[] = "0:0";
static int		card = 0, device = 0;

			/* global sound driver data */
static snd_pcm_t	*sound_handle = NULL;

			/* Default fragment size and number */
static int		fsize = 16384;
static int		fcount = -1;

/*
 * Types
 */

struct priv_t
{
	long			bufferBPS;
	double			bufferDelay;
	XMM_AudioFormat		daf;
};

/*
 * Filter info
 */

#define	ALSA_AUDIO_FORMATS	5

static XMM_AudioFormat alsa_af[ALSA_AUDIO_FORMATS] =
{
    { XMM_AUDIO_FMT_ULAW, 22050, 1, 1, 0, 0, 0, "ulaw" },
    { XMM_AUDIO_FMT_S16LE, 22050, 1, 1, 0, 0, 0, "s16le" },
    { XMM_AUDIO_FMT_S16LE, 24000, 1, 1, 0, 0, 0, "s16le" },
    { XMM_AUDIO_FMT_S16LE, 44100, 1, 1, 0, 0, 0, "s16le" },
    { XMM_AUDIO_FMT_S16LE, 48000, 1, 1, 0, 0, 0, "s16le" },
};

static XMM_FilterAudioInfo	alsa_fai =
{
    0,
    "",					/* Filename. Will be initialized later */
    "ALSA",				/* Name */
    "",					/* Description */
    "Copyright (c) 1999 Arthur Kleer",	/* Copyright */
    ALSA_AUDIO_FORMATS,			/* Number of supported formats */
    alsa_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int alsa_GetVolume( void *xmm, uint32_t *volume );
static int alsa_SetVolume( void *xmm, uint32_t volume );
static int alsa_init( void *xmm, XMM_AudioFormat *saf, int query, int *afmt );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *alsa_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio		*pFilter;
  struct priv_t			*priv;
  snd_pcm_channel_params_t	params;
  snd_pcm_channel_setup_t	setup;
  int				ret = XMM_RET_OK, afmt, err;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( alsa_init( xmm, saf, flags & XMM_FILTER_AOF_QUERY, &afmt ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(ALSA) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Initialize ALSA */
  memset( &params, 0, sizeof(params));
  params.channel = SND_PCM_CHANNEL_PLAYBACK;
  params.mode = SND_PCM_MODE_BLOCK;
  memset( &params.format, 0, sizeof(params.format));
  params.format.interleave = 1;
  params.format.format = afmt;
  params.format.rate = saf->samprate;
  params.format.voices = saf->channels;
  params.start_mode = SND_PCM_START_FULL;
  params.stop_mode = SND_PCM_STOP_STOP;
  params.buf.block.frag_size = FRAG_SIZE;
  if( fsize > 0 )	params.buf.block.frag_size = fsize;
  params.buf.block.frags_max = FRAG_COUNT;
  if( fcount > 0 )	params.buf.block.frags_max = fcount;
  params.buf.block.frags_min = 1;

  if(( err = snd_pcm_channel_params( sound_handle, &params )) != 0 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Error setting parameters: %s", snd_strerror(err));
	return NULL;
  }

  snd_pcm_channel_flush( sound_handle, SND_PCM_CHANNEL_PLAYBACK );

  if(( err = snd_pcm_channel_prepare( sound_handle, SND_PCM_CHANNEL_PLAYBACK )) != 0 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Error preparing playback: %s", snd_strerror(err));
	return NULL;
  }

  memset( &setup, 0, sizeof(setup));
  setup.mode = SND_PCM_MODE_BLOCK;
  setup.channel = SND_PCM_CHANNEL_PLAYBACK;

  if(( err = snd_pcm_channel_setup( sound_handle, &setup )) != 0 )
  {
	xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Setup error: %s", snd_strerror(err));
	return NULL;
  }

  xmm_logging( 2, "ALSA! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
	saf->samprate, saf->channels, saf->format, setup.buf.block.frag_size );

  /* Initialize data */
  priv->bufferBPS = (long)( saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferDelay = (double) setup.buf.block.frag_size / priv->bufferBPS;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';

  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int alsa_Close( XMM_PluginFilterAudio *filter )
{
  filter->Control( filter, XMM_CTLSET_FLUSH, 0, NULL );

  snd_pcm_close( sound_handle );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Control function
 */
static int alsa_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t			*priv = filter->sys.priv;
  snd_pcm_channel_status_t	status;

  memset( &status, 0, sizeof( status ));
  if( snd_pcm_channel_status( sound_handle, &status ) < 0 )
	return xmm_SetError( filter->sys.xmm, XMM_RET_ERROR, "(ALSA) Could not get channel status" );

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( alsa_init( filter->sys.xmm, (XMM_AudioFormat *)data, 1, NULL ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = (double) status.count / priv->bufferBPS;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = (uint32_t) status.count + status.free;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = (uint32_t) status.free;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		alsa_GetVolume( filter->sys.xmm, (uint32_t *) data );
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &alsa_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		alsa_SetVolume( filter->sys.xmm, param );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_FLUSH:
#if 0
		/*
		 * This causes a 'bad file descriptor' error in snd_pcm_write()
		 */
		snd_pcm_playback_drain( sound_handle );
#endif
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(ALSA) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(ALSA) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int alsa_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  snd_pcm_channel_status_t	status;
  int				ret, i;

  for( i = 0; i < 2; i++ )
  {
	ret = snd_pcm_write( sound_handle, src, isize );
	if( ret < 0 )	xmm_logging( 1, "ALSA! Error '%s': src = %p isize = %i\n", snd_strerror(ret), src, isize );

	memset( &status, 0, sizeof( status ));
	if( snd_pcm_channel_status( sound_handle, &status ) < 0 )
		xmm_logging( 1, "ALSA! Could not get channel status\n" );

	if( status.underrun == 0 )	break;

	xmm_logging( 1, "ALSA! Underrun. Ressetting channel...\n" );
	snd_pcm_channel_flush( sound_handle, SND_PCM_CHANNEL_PLAYBACK );
	snd_pcm_playback_prepare( sound_handle );
  }

  if( ret <= 0 )	return xmm_SetError( filter->sys.xmm, XMM_RET_ERROR, "(ALSA) Error on write ( Underrun ??? )\n" );

  if( osize )	*osize = 0;	/* Nothing in dest */
  return ret;
}



/*
 * Process data (BQ)
 */
static int alsa_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"ALSA",
				"Sound: ALSA",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				alsa_Open, alsa_Close, alsa_Control,
				alsa_Process, alsa_ProcessBQ };

/*
 * Internal code
 */

/* XMMP sound format --> ALSA sound format */
struct format_conv
{
	int	xmm;
	int	alsa;
};

static struct format_conv format_table[] =
{
	/* PCM */
	{ XMM_AUDIO_FMT_S8		, SND_PCM_SFMT_S8		},
	{ XMM_AUDIO_FMT_U8		, SND_PCM_SFMT_U8		},
	{ XMM_AUDIO_FMT_S16LE		, SND_PCM_SFMT_S16_LE		},
	{ XMM_AUDIO_FMT_S16BE		, SND_PCM_SFMT_S16_BE		},
	{ XMM_AUDIO_FMT_U16LE		, SND_PCM_SFMT_U16_LE		},
	{ XMM_AUDIO_FMT_U16BE		, SND_PCM_SFMT_U16_BE		},
	{ XMM_AUDIO_FMT_S24LE		, SND_PCM_SFMT_S24_LE		},
	{ XMM_AUDIO_FMT_S24BE		, SND_PCM_SFMT_S24_BE		},
	{ XMM_AUDIO_FMT_U24LE		, SND_PCM_SFMT_U24_LE		},
	{ XMM_AUDIO_FMT_U24BE		, SND_PCM_SFMT_U24_BE		},
	{ XMM_AUDIO_FMT_S32LE		, SND_PCM_SFMT_S32_LE		},
	{ XMM_AUDIO_FMT_S32BE		, SND_PCM_SFMT_S32_BE		},
	{ XMM_AUDIO_FMT_U32LE		, SND_PCM_SFMT_U32_LE		},
	{ XMM_AUDIO_FMT_U32BE		, SND_PCM_SFMT_U32_BE		},
	/* IEEE floating point */
	{ XMM_AUDIO_FMT_IEEE32LE	, SND_PCM_SFMT_FLOAT_LE		},
	{ XMM_AUDIO_FMT_IEEE32BE	, SND_PCM_SFMT_FLOAT_BE		},
	{ XMM_AUDIO_FMT_IEEE64LE	, SND_PCM_SFMT_FLOAT64_LE	},
	{ XMM_AUDIO_FMT_IEEE64BE	, SND_PCM_SFMT_FLOAT64_BE	},
	/* */
	{ XMM_AUDIO_FMT_ULAW		, SND_PCM_SFMT_MU_LAW		},
	{ XMM_AUDIO_FMT_ALAW		, SND_PCM_SFMT_A_LAW		},
	{ XMM_AUDIO_CODEC_ADPCM_IMA	, SND_PCM_SFMT_IMA_ADPCM	},
	{ XMM_AUDIO_FMT_MPEG		, SND_PCM_SFMT_MPEG		},
	{ XMM_AUDIO_CODEC_GSM610MS	, SND_PCM_SFMT_GSM		},
	{ 0, 0 }
};

/*
 * Check output parameters
 */
static int alsa_init( void *xmm, XMM_AudioFormat *saf, int query, int *afmt )
{
  snd_pcm_channel_info_t	ci;
  int				err, xmm_fmt_idx;
  char				*cardname;

  /* Check Sound Device string */
  if(( SoundDev[1] != ':' ) && SoundDev[3] )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Wrong sound device format. Check config file ( 'card:device' )" );

  card = SoundDev[0] - 48;
  device = SoundDev[2] - 48;

  if(( err = snd_card_get_longname( card, &cardname )) != 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Error optainig card name %i,%i: %s", card, device, snd_strerror(err));

  xmm_logging( 2, "ALSA! INFO: soundcard '%s'\n", cardname );
  free( cardname );

  /* Open PCM output */
  if((err = snd_pcm_open( &sound_handle, card, device, SND_PCM_OPEN_PLAYBACK )))
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Error opening audio device %i,%i: %s", card, device, snd_strerror(err));

  /* Get channel information */
  ci.channel = SND_PCM_CHANNEL_PLAYBACK;
  snd_pcm_channel_info( sound_handle, &ci );

#ifdef VERBOSE
  xmm_logging( 1, "ALSA! INFO: formats (0x%x):", ci.formats );
  if( ci.formats & SND_PCM_FMT_S8 )		xmm_logging( 1, " S8" );
  if( ci.formats & SND_PCM_FMT_U8 )		xmm_logging( 1, " U8" );
  if( ci.formats & SND_PCM_FMT_S16_LE )		xmm_logging( 1, " S16LE" );
  if( ci.formats & SND_PCM_FMT_S16_BE )		xmm_logging( 1, " S16BE" );
  if( ci.formats & SND_PCM_FMT_U16_LE )		xmm_logging( 1, " U16LE" );
  if( ci.formats & SND_PCM_FMT_U16_BE )		xmm_logging( 1, " U16BE" );
  if( ci.formats & SND_PCM_FMT_S24_LE )		xmm_logging( 1, " S24LE" );
  if( ci.formats & SND_PCM_FMT_S24_BE )		xmm_logging( 1, " S24BE" );
  if( ci.formats & SND_PCM_FMT_U24_LE )		xmm_logging( 1, " U24LE" );
  if( ci.formats & SND_PCM_FMT_U24_BE )		xmm_logging( 1, " U24BE" );
  if( ci.formats & SND_PCM_FMT_S32_LE )		xmm_logging( 1, " S32LE" );
  if( ci.formats & SND_PCM_FMT_S32_BE )		xmm_logging( 1, " S32BE" );
  if( ci.formats & SND_PCM_FMT_U32_LE )		xmm_logging( 1, " U32LE" );
  if( ci.formats & SND_PCM_FMT_U32_BE )		xmm_logging( 1, " U32BE" );
  if( ci.formats & SND_PCM_FMT_FLOAT_LE )	xmm_logging( 1, " IEEE(32,LE)" );
  if( ci.formats & SND_PCM_FMT_FLOAT_BE )	xmm_logging( 1, " IEEE(32,BE)" );
  if( ci.formats & SND_PCM_FMT_FLOAT64_LE )	xmm_logging( 1, " IEEE(64,LE)" );
  if( ci.formats & SND_PCM_FMT_FLOAT64_BE )	xmm_logging( 1, " IEEE(64,BE)" );
  if( ci.formats & SND_PCM_FMT_MU_LAW	)	xmm_logging( 1, " u-law" );
  if( ci.formats & SND_PCM_FMT_A_LAW )		xmm_logging( 1, " a-law" );
  if( ci.formats & SND_PCM_FMT_IMA_ADPCM )	xmm_logging( 1, " IMA ADPCM" );
  if( ci.formats & SND_PCM_FMT_MPEG )		xmm_logging( 1, " MPEG" );
  if( ci.formats & SND_PCM_FMT_GSM )		xmm_logging( 1, " GSM" );
  xmm_logging( 1, "\n" );
  xmm_logging( 1, "ALSA! INFO: sample rate range [%i,%i])\n", ci.min_rate, ci.max_rate );
  xmm_logging( 1, "ALSA! INFO: channel range [%i,%i])\n", ci.min_voices, ci.max_voices );
#endif

  /* Find XMM format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if( saf->format == format_table[xmm_fmt_idx].xmm )
		break;

  if( format_table[xmm_fmt_idx].xmm == 0 )
  {
	snd_pcm_close( sound_handle );
	sound_handle = NULL;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Unknown/Unsupported XMM format 0x%x", saf->format );
  }

  /* Check if format supported by ALSA */
  if(( ci.formats & ( 1 << format_table[xmm_fmt_idx].alsa )) == 0 )
  {
	snd_pcm_close( sound_handle );
	sound_handle = NULL;
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) ALSA format ( 0x%x ) not supported [ needed for XMM format 0x%x ]", format_table[xmm_fmt_idx].alsa, format_table[xmm_fmt_idx].xmm );
  }

  /* Check sample rate */
  if(( saf->samprate > ci.max_rate ) || ( saf->samprate < ci.min_rate ))
  {
	snd_pcm_close( sound_handle );
	sound_handle = NULL;
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(ALSA) Sample rate (%i) not supported. Range: [%i,%i]", saf->samprate, ci.min_rate, ci.max_rate );
  }

  /* Check channels */
  if(( saf->channels > ci.max_voices ) || ( saf->channels < ci.min_voices ))
  {
	snd_pcm_close( sound_handle );
	sound_handle = NULL;
	return xmm_SetError( xmm, XMM_RET_NOTSUPPORTED, "(ALSA) Channel number (%i) not supported. Range: [%i,%i]", saf->channels, ci.min_voices, ci.max_voices );
  }


  /* close device - only in query mode */
  if( query )
  {
	snd_pcm_close( sound_handle );
	sound_handle = NULL;
  }

  if( afmt )	*afmt = format_table[xmm_fmt_idx].alsa;

  return XMM_RET_OK;
}

/*
 * Mixer
 */

static int get_percent( int val, int min, int max )
{
  if( max == min )		return 0;
  return (( val - min ) * 100 ) / ( max - min );
}

static int get_range( int val, int min, int max )
{
  if( max == min )	return 0;
  return ( val * ( max - min ) / 100 ) + min;
}

/*
 * Get volume
 */
static int alsa_GetVolume( void *xmm, uint32_t *volume )
{
  int			err, left, right;
  snd_mixer_t		*handle;
  snd_mixer_gid_t	gid;
  snd_mixer_group_t	group;

#ifdef VERBOSE
  xmm_logging( 1, "ALSA! Mixer: Group '%s' (%i)\n", GROUP_NAME, GROUP_INDEX );
#endif

  /* Open Mixer */
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Mixer %i:%i open error: %s", card, device, snd_strerror(err));

  /* Fill group id struct */
  memset( &gid, 0, sizeof( snd_mixer_gid_t ));
  strncpy( gid.name, GROUP_NAME, sizeof( gid.name ) - 1 );
  gid.index = GROUP_INDEX;

  /* Fill group struct */
  memset( &group, 0, sizeof( snd_mixer_group_t ));
  memcpy( &group.gid, &gid, sizeof( snd_mixer_gid_t ));	/* Set Group ID */

  /* Read Group info */
  if(( err = snd_mixer_group_read( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Mixer %i:%i group read error: %s", card, device, snd_strerror(err));
  }

  /* Output informations */
  if( group.channels == 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Group '%s(%i)' has no channels", GROUP_NAME, GROUP_INDEX );
  }

  if( group.channels == SND_MIXER_CHN_MASK_MONO )
  {
	left = get_percent( group.volume.names.front_left, group.min, group.max );
	right = left;
  }
  else
  {
	left = get_percent( group.volume.names.front_left, group.min, group.max );
	right = get_percent( group.volume.names.front_right, group.min, group.max );
  }

  *volume = left | ( right << 16 );

  snd_mixer_close( handle );
  return 0;
}

/*
 * Set Volume
 */
static int alsa_SetVolume( void *xmm, uint32_t volume )
{
  int			err, ch;
  snd_mixer_t		*handle;
  snd_mixer_gid_t	gid;
  snd_mixer_group_t	group;

#ifdef VERBOSE
  xmm_logging( 1, "ALSA! Mixer: Group '%s' (%i)\n", GROUP_NAME, GROUP_INDEX );
#endif

  /* Open mixer */
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Mixer %i:%i open error: %s", card, device, snd_strerror( err ));

  /* Fill group id struct */
  memset( &gid, 0, sizeof( snd_mixer_gid_t ));
  strncpy( gid.name, GROUP_NAME, sizeof( gid.name ) - 1 );
  gid.index = GROUP_INDEX;

  /* Fill group struct */
  memset( &group, 0, sizeof( snd_mixer_group_t ));
  memcpy( &group.gid, &gid, sizeof( snd_mixer_gid_t ));	/* Set Group ID */

  /* Read Group info */
  if(( err = snd_mixer_group_read( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Mixer %i:%i group read error: %s", card, device, snd_strerror( err ));
  }

  /* Fill group struct with new settings */
  if( group.caps & SND_MIXER_GRPCAP_JOINTLY_VOLUME )
  {
	for( ch = 0; ch < SND_MIXER_CHN_LAST; ch++ )
	    if( group.channels & ( 1 << ch ))	group.volume.values[ch] = get_range( volume & 0x0000FFFF, group.min, group.max );
  }
  else
  {
	group.volume.values[SND_MIXER_CHN_FRONT_LEFT] = get_range( volume & 0x0000FFFF, group.min, group.max );
	group.volume.values[SND_MIXER_CHN_FRONT_RIGHT] = get_range(( volume & 0xFFFF0000 ) >> 16, group.min, group.max );
  }

  /* Write group */
  if(( err = snd_mixer_group_write( handle, &group )) < 0 )
  {
	snd_mixer_close( handle );
	return xmm_SetError( xmm, XMM_RET_ERROR, "(ALSA) Mixer %i:%i group write error: %s", card, device, snd_strerror( err ));
  }

  /* Close mixer */
  snd_mixer_close( handle );

  return 0;
}

#ifdef VERBOSE

static int mixer_info( void )
{
  int			err;
  snd_mixer_t		*handle;
  snd_mixer_info_t	info;
	
  if(( err = snd_mixer_open( &handle, card, device )) < 0 )
  {
	xmm_logging( 3, "ALSA! Mixer %i:%i open error: %s\n", card, device, snd_strerror( err ));
	return -1;
  }

  if(( err = snd_mixer_info( handle, &info )) < 0 )
  {
	xmm_logging( 3, "ALSA! Mixer %i:%i info error: %s\n", card, device, snd_strerror( err ));
	return -1;
  }

  xmm_logging( 3, "ALSA! ALSA Mixer ( ID: %s ) %s\n", info.id, info.name );
  xmm_logging( 3, "ALSA! \tType (Soundcard) : 0x%x\n", info.type );
  xmm_logging( 3, "ALSA! \tAttribute        : 0x%x\n", info.attrib );
  xmm_logging( 3, "ALSA! \tElements         : %i\n", info.elements );
  xmm_logging( 3, "ALSA! \tGroups           : %i\n", info.groups );

  snd_mixer_close(handle);

  return 1;
}

static void mixer_group_info( snd_mixer_group_t *group )
{
  int ch;

  xmm_logging( 3, "ALSA! Mixer Group Capabilities:" );

  if( group->caps & SND_MIXER_GRPCAP_VOLUME )		xmm_logging( 3, " volume" );
  if( group->caps & SND_MIXER_GRPCAP_MUTE )		xmm_logging( 3, " mute" );
  if( group->caps & SND_MIXER_GRPCAP_JOINTLY_MUTE )	xmm_logging( 3, " jointly-mute" );
  if( group->caps & SND_MIXER_GRPCAP_CAPTURE )		xmm_logging( 3, " capture" );
  if( group->caps & SND_MIXER_GRPCAP_JOINTLY_CAPTURE )	xmm_logging( 3, " jointly-capture" );
  if( group->caps & SND_MIXER_GRPCAP_EXCL_CAPTURE )	xmm_logging( 3, " exclusive-capture" );
  xmm_logging( 3, "\n" );

  if(( group->caps & SND_MIXER_GRPCAP_CAPTURE ) && ( group->caps & SND_MIXER_GRPCAP_EXCL_CAPTURE ))
	    xmm_logging( 3, "ALSA! Mixer Group: Capture exclusive group: %i\n", group->capture_group );

  xmm_logging( 3, "ALSA! Mixer Group: Limits: min = %i, max = %i\n", group->min, group->max );

  xmm_logging( 3, "ALSA! Mixer Group: Channels:\n" );
  if( group->channels == SND_MIXER_CHN_MASK_MONO )
  {
	xmm_logging( 3, "ALSA! Mixer Group: Mono: %i [%i%%] [%s]\n",  group->volume.names.front_left,
					    get_percent( group->volume.names.front_left, group->min, group->max ),
					    group->mute & SND_MIXER_CHN_MASK_MONO ? "mute" : "on" );
  }
  else
  {
	for( ch = 0; ch <= SND_MIXER_CHN_LAST; ch++ )
	{
	    if( group->channels & ( 1 << ch ))
		xmm_logging( 3, "ALSA! Mixer Group: %s: %i [%i%%] [%s] [%s]\n",	snd_mixer_channel_name( ch ),
						group->volume.values[ch],
						get_percent( group->volume.values[ch], group->min, group->max ),
						group->mute & ( 1 << ch ) ? "mute" : "on",
						group->capture & ( 1 << ch ) ? "capture" : "---" );
	}
  }
}

#endif
