/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * au.c
 * Sun / NeXT audio output ( writer )
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/endian.h>
#include <libxmm/util/utils.h>
#include "au.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384

/*
 * Global data
 */

			/* Default fragment size and number */
static int		fsize = 16384;

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioFormat		daf;

    int				data_size;

    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Filter info
 */

#define	AU_AUDIO_FORMATS	11

static XMM_AudioFormat au_af[AU_AUDIO_FORMATS] =
{
    { XMM_AUDIO_FMT_U8, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "PCM (8bit)" },
    { XMM_AUDIO_FMT_S16LE, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "PCM (16bit,signed,LE)" },
    { XMM_AUDIO_FMT_S24LE, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "PCM (24bit,signed,LE)" },
    { XMM_AUDIO_FMT_S32LE, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "PCM (32bit,signed,LE)" },
    { XMM_AUDIO_FMT_IEEE32LE, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "IEEE FP(32bit,signed,LE)" },
    { XMM_AUDIO_FMT_IEEE64LE, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "IEEE FP(64bit,signed,LE)" },
    { XMM_AUDIO_CODEC_ULAW, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "u-Law" },
    { XMM_AUDIO_CODEC_ALAW, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "A-Law" },
    { XMM_AUDIO_CODEC_G721, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "G.721 (4-bit)" },
    { XMM_AUDIO_CODEC_G723 | 3, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "G.723 (3-bit)" },
    { XMM_AUDIO_CODEC_G723 | 5, 0xFFFFFFFF, 0xFFFF, 0, 0, 0, 0, "G.723 (5-bit)" },
};

static XMM_FilterAudioInfo	au_fai =
{
    0,
    "",					/* Filename. Will be initialized later */
    "AU",				/* Name */
    "Sun/Next audio",			/* Description */
    "Copyright (c) 2002 Arthur Kleer",	/* Copyright */
    AU_AUDIO_FORMATS,			/* Number of supported formats */
    au_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int WRITE32( XMM_PluginIO *pIO, uint32_t val );
static int au_check( void *xmm, XMM_AudioFormat *saf, int query, uint32_t *fmt );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *au_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = XMM_RET_OK, len;
  const char		*ofn;
  char			software[256];
  uint32_t		fmt;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( au_check( xmm, saf, flags & XMM_FILTER_AOF_QUERY, &fmt ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(AU) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Open output */
  if(( ofn = xmm_AcOutputFilename( xmm, NULL, ".au" )) == NULL )
	return NULL;

  if(( priv->pIO = xmm_IOOpen( xmm, ofn, XMM_IO_WRITE )) == NULL )
	return NULL;

  /**/
  memset( software, 0, 256 );
  strcpy( software, "Created by "XMM_PROJECT_NAME" AU Plugin v"XMM_VERSION_STRING );
  len = strlen( software ) + 1;
  len = AlignUP( len, 4 );

  /* Write header */
  WRITE32( priv->pIO, XMM_INT32_BE( AU_FORMAT_ID ));	/* ID */
  WRITE32( priv->pIO, XMM_INT32_BE( 24 + len ));	/* Offset */
  WRITE32( priv->pIO, XMM_INT32_BE( 0xFFFFFFFF ));	/* Size */
  WRITE32( priv->pIO, XMM_INT32_BE( fmt ));		/* Encoding/Type */
  WRITE32( priv->pIO, XMM_INT32_BE( saf->samprate ));	/* Sample rate */
  WRITE32( priv->pIO, XMM_INT32_BE( saf->channels ));	/* Channels */
  priv->pIO->Write( priv->pIO, software, len, 1 );

  priv->data_size = 0;

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  xmm_logging( 2, "AU! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
			saf->samprate, saf->channels, saf->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)saf->bitrate;
  priv->bufferSSize = ( saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferSize = fsize;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';

  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int au_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;

  /* Fix / Rewrite Header */
  priv->pIO->Seek( priv->pIO, 8, XMM_SEEK_SET );
  WRITE32( priv->pIO, XMM_INT32_BE( priv->data_size ));

  priv->pIO->Close( priv->pIO );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int au_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( au_check( filter->sys.xmm, (XMM_AudioFormat *)data, 1, NULL ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &au_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(AU) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(AU) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int au_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t	*priv = filter->sys.priv;
  int		done;

  done = priv->pIO->Write( priv->pIO, src, 1, isize );
  priv->data_size += done;

  if( done == 0 )	return XMM_RET_ERROR;

  if( osize )	*osize = 0;	/* Nothing in dest */
  return done;
}

/*
 * Process data (BQ)
 */
static int au_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"AU",
				"Sound: Sun/NeXt ",
				"Copyright (c) 2002 Arthur Kleer",
				NULL, NULL },
				au_Open, au_Close, au_Control,
				au_Process, au_ProcessBQ };

/*
 * Internal code
 */

/*
 * Data output
 */

static int WRITE32( XMM_PluginIO *pIO, uint32_t val )
{
  return pIO->Write( pIO, &val, 4, 1 );
}

/* XMMP sound format --> AU Encoding/Type */
struct format_conv
{
	int	au;
	int	xmm;
	int	mask;
};

static struct format_conv format_table[] =
{
    { AU_DATA_ULAW	, XMM_AUDIO_CODEC_ULAW | 8	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { AU_DATA_ALAW	, XMM_AUDIO_CODEC_ALAW | 8	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { AU_DATA_PCM8	, XMM_AUDIO_FMT_S8		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { AU_DATA_PCM16	, XMM_AUDIO_FMT_S16BE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { AU_DATA_PCM24	, XMM_AUDIO_FMT_S24BE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { AU_DATA_PCM32	, XMM_AUDIO_FMT_S32BE		, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE | XMM_AUDIO_MASK_SIGNED },
    { AU_DATA_FLOAT	, XMM_AUDIO_FMT_IEEE32BE	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE },
    { AU_DATA_DOUBLE	, XMM_AUDIO_FMT_IEEE64BE	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE | XMM_AUDIO_MASK_BE },
    { AU_DATA_G721	, XMM_AUDIO_CODEC_G721 | 4	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { AU_DATA_G723_3	, XMM_AUDIO_CODEC_G723 | 3	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { AU_DATA_G723_5	, XMM_AUDIO_CODEC_G723 | 5	, XMM_AUDIO_MASK_CODEC | XMM_AUDIO_MASK_SIZE },
    { 0, 0, 0 }
};

/*
 * Initialize AU
 */
static int au_check( void *xmm, XMM_AudioFormat *saf, int query, uint32_t *fmt )
{
  int		xmm_fmt_idx;

  /* Find XMM format */
  for( xmm_fmt_idx = 0; format_table[xmm_fmt_idx].xmm; xmm_fmt_idx++ )
	if(( saf->format & format_table[xmm_fmt_idx].mask ) == format_table[xmm_fmt_idx].xmm )
		break;

  if( format_table[xmm_fmt_idx].xmm == 0 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(AU) Unknown/Unsupported XMM format 0x%x", saf->format );

  /* Query only */
  if( query )	return XMM_RET_OK;

  *fmt = format_table[xmm_fmt_idx].au;

  /* return */
  return XMM_RET_OK;
}
