/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * lame.c
 * MPEG Layer 3 audio output ( writer )
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/lpio.h>
#include <libxmm/error.h>
#include <libxmm/config.h>
#include <libxmm/util/utils.h>
#include "lame/lame.h"

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		16384

/*
 * Global data
 */

			/* Default fragment size and number */
static int		fsize = 16384;

/*
 * Types
 */

struct priv_t
{
    XMM_PluginIO		*pIO;
    XMM_AudioFormat		daf;

    lame_global_flags		gf;
    char			*mp3data;
    int				mp3data_size;

    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Filter info
 */

#define	LAME_AUDIO_FORMATS	51

static XMM_AudioFormat lame_af[LAME_AUDIO_FORMATS] =
{
    /*
     * Stereo
     */

    /* MPEG 1.0 */
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 320 * 1000, 0, 0, "320 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 256 * 1000, 0, 0, "256 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 48000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 2, 1,  96 * 1000, 0, 0, " 96 kBit/s, 48000Hz, Stereo" },

    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 320 * 1000, 0, 0, "320 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 256 * 1000, 0, 0, "256 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 44100Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 2, 1,  96 * 1000, 0, 0, " 96 kBit/s, 44100Hz, Stereo" },

    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 320 * 1000, 0, 0, "320 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 256 * 1000, 0, 0, "256 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 224 * 1000, 0, 0, "224 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 192 * 1000, 0, 0, "192 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 160 * 1000, 0, 0, "160 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 128 * 1000, 0, 0, "128 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1, 112 * 1000, 0, 0, "112 kBit/s, 32000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 2, 1,  96 * 1000, 0, 0, " 96 kBit/s, 32000Hz, Stereo" },

    /* MPEG 2.0 */
    { XMM_AUDIO_CODEC_MPEG, 24000, 2, 1, 64 * 1000, 0, 0, "64 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 24000, 2, 1, 56 * 1000, 0, 0, "56 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 24000, 2, 1, 48 * 1000, 0, 0, "48 kBit/s, 24000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 24000, 2, 1, 40 * 1000, 0, 0, "40 kBit/s, 24000Hz, Stereo" },

    { XMM_AUDIO_CODEC_MPEG, 22050, 2, 1, 64 * 1000, 0, 0, "64 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 22050, 2, 1, 56 * 1000, 0, 0, "56 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 22050, 2, 1, 48 * 1000, 0, 0, "48 kBit/s, 22050Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 22050, 2, 1, 40 * 1000, 0, 0, "40 kBit/s, 22050Hz, Stereo" },

    { XMM_AUDIO_CODEC_MPEG, 16000, 2, 1, 56 * 1000, 0, 0, "56 kBit/s, 16000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 16000, 2, 1, 48 * 1000, 0, 0, "48 kBit/s, 16000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 16000, 2, 1, 40 * 1000, 0, 0, "40 kBit/s, 16000Hz, Stereo" },
    { XMM_AUDIO_CODEC_MPEG, 16000, 2, 1, 32 * 1000, 0, 0, "32 kBit/s, 16000Hz, Stereo" },

    /*
     * Mono
     */

    /* MPEG 1.0 */
    { XMM_AUDIO_CODEC_MPEG, 48000, 1, 1, 64 * 1000, 0, 0, "64 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 1, 1, 56 * 1000, 0, 0, "56 kBit/s, 48000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 48000, 1, 1, 48 * 1000, 0, 0, "48 kBit/s, 48000Hz, Mono" },

    { XMM_AUDIO_CODEC_MPEG, 44100, 1, 1, 64 * 1000, 0, 0, "64 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 1, 1, 56 * 1000, 0, 0, "56 kBit/s, 44100Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 44100, 1, 1, 48 * 1000, 0, 0, "48 kBit/s, 44100Hz, Mono" },

    { XMM_AUDIO_CODEC_MPEG, 32000, 1, 1, 64 * 1000, 0, 0, "64 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 1, 1, 56 * 1000, 0, 0, "56 kBit/s, 32000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 32000, 1, 1, 48 * 1000, 0, 0, "48 kBit/s, 48000Hz, Mono" },

    /* MPEG 2.0 */
    { XMM_AUDIO_CODEC_MPEG, 24000, 1, 1, 32 * 1000, 0, 0, "32 kBit/s, 24000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 24000, 1, 1, 24 * 1000, 0, 0, "24 kBit/s, 24000Hz, Mono" },

    { XMM_AUDIO_CODEC_MPEG, 22050, 1, 1, 32 * 1000, 0, 0, "32 kBit/s, 22050Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 22050, 1, 1, 24 * 1000, 0, 0, "24 kBit/s, 22050Hz, Mono" },

    { XMM_AUDIO_CODEC_MPEG, 16000, 1, 1, 32 * 1000, 0, 0, "32 kBit/s, 16000Hz, Mono" },
    { XMM_AUDIO_CODEC_MPEG, 16000, 1, 1, 24 * 1000, 0, 0, "24 kBit/s, 16000Hz, Mono" },

};

static XMM_FilterAudioInfo	lame_fai =
{
    0,
    "",					/* Filename. Will be initialized later */
    "LAME",				/* Name */
    "MPEG (Audio) Layer 3",		/* Description */
    "Copyright (c) 2000 Arthur Kleer",	/* Copyright */
    LAME_AUDIO_FORMATS,			/* Number of supported formats */
    lame_af				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int lame_check( void *xmm, XMM_AudioFormat *aformat, int query );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *lame_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = XMM_RET_OK;
  const char		*ofn;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( lame_check( xmm, saf, flags & XMM_FILTER_AOF_QUERY ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(LAME) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* Open output */
  if(( ofn = xmm_AcOutputFilename( xmm, NULL, ".mp3" )) == NULL )
	return NULL;

  if(( priv->pIO = xmm_IOOpen( xmm, ofn, XMM_IO_WRITE )) == NULL )
	return NULL;

  /* Init LAME stuff */
  lame_init( &priv->gf );

  priv->gf.num_channels = saf->channels;
  priv->gf.in_samplerate = saf->samprate;
  priv->gf.brate = saf->bitrate / 1000;
  priv->gf.mode = 1;
  priv->gf.quality = 2;
  priv->gf.silent = 1;

  lame_init_params( &priv->gf );

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  /* Calculate and allocate mp3 buffer */
  priv->mp3data_size = fsize * 5 / 4 + 7200;
  if(( priv->mp3data = malloc( priv->mp3data_size )) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(LAME) mp3 buffer." );
	return NULL;
  }

  xmm_logging( 2, "LAME! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
			saf->samprate, saf->channels, saf->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ));
  priv->bufferSSize = ( saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferSize = fsize;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';
  
  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int lame_Close( XMM_PluginFilterAudio *filter )
{
  struct priv_t *priv = filter->sys.priv;
  int		done;

  done = lame_encode_finish( &priv->gf, priv->mp3data, priv->mp3data_size );
  if( done > 0 )	priv->pIO->Write( priv->pIO, priv->mp3data, done, 1 );

  priv->pIO->Close( priv->pIO );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int lame_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( lame_check( filter->sys.xmm, (XMM_AudioFormat *)data, 1 ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &lame_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */
  
	case XMM_CTLSET_VOLUME:
		return XMM_RET_NOTSUPPORTED;

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(LAME) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(LAME) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int lame_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  struct priv_t	*priv = filter->sys.priv;
  int		done;

  if( priv->gf.num_channels == 1 )
	done = lame_encode_buffer( &priv->gf, (short *)src, (short *)src, isize / priv->bufferSSize, priv->mp3data, priv->mp3data_size );
  else
  	done = lame_encode_buffer_interleaved( &priv->gf, (short *)src, isize / priv->bufferSSize, priv->mp3data, priv->mp3data_size );

  if( done > 0 )	priv->pIO->Write( priv->pIO, priv->mp3data, done, 1 );
  if( done < 0 )	done = 0;
  if( done == 0 )	return XMM_RET_ERROR;

  if( osize )	*osize = 0;	/* Nothing in dest */
  return done;
}

/*
 * Process data (BQ)
 */
static int lame_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"LAME",
				"Sound: MPEG Layer 3",
				"Copyright (c) 2000 Arthur Kleer",
				NULL, NULL },
				lame_Open, lame_Close, lame_Control,
				lame_Process, lame_ProcessBQ };

/*
 * Internal code
 */

/*
 * Initialize LAME
 */
static int lame_check( void *xmm, XMM_AudioFormat *saf, int query )
{

  if(( saf->format & XMM_AUDIO_MASK_CODEC ) != XMM_AUDIO_CODEC_PCM )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(LAME) Format 0x%x not supported ( 16bit PCM data needed )", saf->format );

  if(( saf->format & XMM_AUDIO_MASK_SIZE ) != 16 )
	return xmm_SetError( xmm, XMM_RET_ERROR, "(LAME) MPEG Audio supports only 16 bit PCM data" );

  /* return */
  return XMM_RET_OK;
}
