/*
 *  XMMP - LinuX MultiMedia Project ( www.frozenproductions.com )
 *  Copyright (c) 1999 - 2002 Arthur Kleer <kleer@frozenproductions.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * null.c
 * Dummy audio output
 */

#include <stdlib.h>
#include <string.h>

#include <libxmm/xmmp.h>
#include <libxmm/version.h>
#include <libxmm/xmmctl.h>
#include <libxmm/afilter.h>
#include <libxmm/util/utils.h>
#include <libxmm/error.h>

/*
 * Definitions
 */

#define	PCM_BUFFER_SIZE		8192

/*
 * Global data
 */

			/* Default fragment size and number */
static int		fsize = 16384;

/*
 * Types
 */

struct priv_t
{
    XMM_AudioFormat		daf;
    double			bufferDelay;
    long			bufferSize;
    long			bufferSSize;
};

/*
 * Filter info
 */

static XMM_FilterAudioInfo	null_fai =
{
    XMM_FILTER_ACF_AUTOSELECT,
    "",					/* Filename. Will be initialized later */
    "NULL",				/* Name */
    "",					/* Description */
    "Copyright (c) 1999 Arthur Kleer",	/* Copyright */
    0,					/* Number of supported formats */
    NULL				/* Pointer to format data */
};

/*
 * Global data
 */

extern XMM_PluginFilterAudio	plugin_info;

/*
 * Prototypes
 */

static int null_init( void *xmm, XMM_AudioFormat *saf, int query );
static int null_GetVolume( void *xmm, uint32_t *volume );
static int null_SetVolume( void *xmm, uint32_t volume );

/*
 * Initialize Plugin
 */
static XMM_PluginFilterAudio *null_Open( void *xmm, XMM_AudioFormat *saf, XMM_AudioFormat *daf, uint32_t flags )
{
  XMM_PluginFilterAudio	*pFilter;
  struct priv_t		*priv;
  int			ret = XMM_RET_OK;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	ret = XMM_RET_NOTSUPPORTED;

  if( null_init( xmm, saf, flags & XMM_FILTER_AOF_QUERY ) != XMM_RET_OK )
	return (void *)ret;

  /* Only query filter */
  if( flags & XMM_FILTER_AOF_QUERY )	return (void *)NULL;

  /* Allocate filter */
  if(( pFilter = xmm_memdup_x( &plugin_info, sizeof( XMM_PluginFilterAudio ), sizeof( struct priv_t ))) == NULL )
  {
	xmm_SetError( xmm, XMM_RET_ALLOC, "(NULL) Unable to duplicate plugin_info" );
	return NULL;
  }

  priv = (struct priv_t *) &pFilter[1];
  pFilter->sys.priv = (void *) priv;
  pFilter->sys.xmm = xmm;

  /* correct fsize */
  if( fsize <= 0 )	fsize = PCM_BUFFER_SIZE;

  xmm_logging( 2, "NULL! Started ( %i Hz, %i channel(s), format %x, bs = %i )\n",
			saf->samprate, saf->channels, saf->format, fsize );

  /* Initialize data */
  priv->bufferDelay = fsize / (double)( saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ));
  priv->bufferSSize = ( saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8 );
  priv->bufferSize = fsize;

  /* Set initialized format */
  priv->daf.format = saf->format;
  priv->daf.samprate = saf->samprate;
  priv->daf.channels = saf->channels;
  priv->daf.bitrate = saf->samprate * saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE );
  priv->daf.blockSize = saf->channels * ( saf->format & XMM_AUDIO_MASK_SIZE ) / 8;
  priv->daf.extraSize = 0;
  priv->daf.extraType = 0;
  priv->daf.desc[0] = '\0';

  /* return filter address */
  return pFilter;
}

/*
 * Free filter
 */
static int null_Close( XMM_PluginFilterAudio *filter )
{
  filter->Control( filter, XMM_CTLSET_FLUSH, 0, NULL );

  free( filter );
  return XMM_RET_OK;
}

/*
 * Filter control
 */
static int null_Control( XMM_PluginFilterAudio *filter, uint32_t cmd, uint32_t param, void *data )
{
  struct priv_t		*priv = filter->sys.priv;

  switch( cmd )
  {
	case XMM_CTLQUERY_AFORMAT:
		if( null_init( filter->sys.xmm, (XMM_AudioFormat *)data, 1 ) == XMM_RET_OK )
			return XMM_CTLRET_TRUE;
		else	return XMM_CTLRET_FALSE;

	case XMM_CTLQUERY_CONFIG:
		return XMM_CTLRET_FALSE;

	case XMM_CTLGET_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_MAX_DELAY:
		*((double *)data) = priv->bufferDelay;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_TOTAL:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_BUFFER_FREE:
		*((uint32_t *)data) = priv->bufferSize;
		return XMM_CTLRET_ARG;		/* Result in arg */

	case XMM_CTLGET_VOLUME:
		null_GetVolume( filter->sys.xmm, (uint32_t *) data );
  		return XMM_CTLRET_ARG;		/* Result in arg */

  	case XMM_CTLGET_DATA_SSIZE:
		*((uint32_t *)data) = param;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_AFORMAT_PTR:			/* Get sound format */
		*((XMM_AudioFormat **) data) = &priv->daf;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLGET_FILTER_INFO:
		*((XMM_FilterAudioInfo **)data) = &null_fai;
		return XMM_CTLRET_ARG;			/* Result in arg */

	case XMM_CTLSET_VOLUME:
		null_SetVolume( filter->sys.xmm, param );
		return XMM_CTLRET_TRUE;

	case XMM_CTLSET_FLUSH:
		return XMM_CTLRET_TRUE;

	/* Dialogues */
	case XMM_CTLDLG_QUERY:
		return XMM_CTLRET_FALSE;

	case XMM_CTLDLG_DISPLAY:
		return XMM_RET_NOTSUPPORTED;

	default:
		break;
  }

  if( cmd & XMM_CTLMASK_AFILTER )
	return xmm_SetError( filter->sys.xmm, XMM_RET_NOTSUPPORTED, "(NULL) cmd = 0x%x" );

  return xmm_SetError( filter->sys.xmm, XMM_RET_INVALID_ARG, "(NULL) cmd ( 0x%x )" );
}

/*
 * Process data
 */
static int null_Process( XMM_PluginFilterAudio *filter, uint8_t *src, uint32_t isize, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  if( osize )	*osize = 0;	/* Nothing in dest */
  return isize;
}

/*
 * Process data (BQ)
 */
static int null_ProcessBQ( XMM_PluginFilterAudio *filter, XMM_BufferQueue *bq, uint8_t *dest, uint32_t *osize, uint32_t *flags )
{
  return XMM_RET_NOTSUPPORTED;
}

/*
 * Plugin info
 */
XMM_PluginFilterAudio plugin_info = {{ NULL,
				XMM_PLUGIN_ID,
				XMM_PLUGIN_TYPE_AFILTER,
				XMM_FILTER_FLAG_AOUTPUT,
				XMM_VERSION_NUM,
				"",
				"NULL",
				"Sound: NULL",
				"Copyright (c) 1999 Arthur Kleer",
				NULL, NULL },
				null_Open, null_Close, null_Control,
				null_Process, null_ProcessBQ };

/*
 * Internal code
 */

/*
 * Initialize OSS
 */
static int null_init( void *xmm, XMM_AudioFormat *saf, int query )
{

  /* close audio output - only in query mode */
  if( query )
  {
  }

  return XMM_RET_OK;
}

/* * * * * * * * *
 * * * Mixer * * *
 * * * * * * * * */

#define	VOLUME_LEFT	100
#define	VOLUME_RIGHT	100


/*
 * Get volume
 */
static int null_GetVolume( void *xmm, uint32_t *volume )
{
  *volume = VOLUME_LEFT | ( VOLUME_RIGHT << 16 );

  return 0;
}

/*
 * Set volume
 */
static int null_SetVolume( void *xmm, uint32_t volume )
{

/*
 * Split volume:
 * left = volume & 0x0000FFFF
 * right = ( volume & 0xFFFF0000 ) >> 16
 */

  return 0;
}
